from drf_yasg import openapi
from rest_framework import status

from smarttv.droideka.proxy.serializers import serializers
from smarttv.droideka.proxy.serializers.fast_response import CarouselsV7Serializer, CarouselSerializer
from smarttv.droideka.proxy.views.base import BackendAPIException
from smarttv.droideka.proxy import api
from smarttv.droideka.proxy.constants.carousels import VhFeed
from smarttv.droideka.proxy.swagger import episode_spec
from smarttv.droideka.proxy.response.carousels import KpMultiSelectionsResponseFields
from smarttv.droideka.proxy.constants.carousels import CarouselType
from smarttv.utils import headers


FILTER_EXAMPLE = """
\t{
\t\t...
\t\t"filter": {\n
\t\t\t"filters": [\n
\t\t\t\t{\n
\t\t\t\t\t"title": "Cool filter 1",\n
\t\t\t\t\t"keys": [\n
\t\t\t\t\t\t{\n
\t\t\t\t\t\t\t"title": "Key 1",\n
\t\t\t\t\t\t\t"key": "key=1"\n
\t\t\t\t\t\t},\n
\t\t\t\t\t\t{\n
\t\t\t\t\t\t\t"title": "Триллер",\n
\t\t\t\t\t\t\t"key": "key=2"\n
\t\t\t\t\t\t},\n
\t\t\t\t\t\t...\n
\t\t\t\t\t]\n
\t\t\t\t},\n
\t\t\t\t{\n
\t\t\t\t\t"title": "Cool filter w",\n
\t\t\t\t\t"keys": [\n
\t\t\t\t\t\t{\n
\t\t\t\t\t\t\t"title": "Super key 1",\n
\t\t\t\t\t\t\t"key": "superkey=1"\n
\t\t\t\t\t\t},\n
\t\t\t\t\t\t{\n
\t\t\t\t\t\t\t"title": "Super key 2",\n
\t\t\t\t\t\t\t"key": "superkey=2"\n
\t\t\t\t\t\t},\n
\t\t\t\t\t\t...\n
\t\t\t\t\t]\n
\t\t\t\t}
\t\t\t],\n
\t\t\t"base_url": "http://my.url.com/api/v7/carousel?base_cgi_param_1=abc&cbase_gi_param_2=def"\n
\t\t},
\t\t...
\t}
"""


filter_item_schema = openapi.Schema(
    type=openapi.TYPE_OBJECT, description='Элементарный фильтр(Способ фильтрации).',
    properties={
        'title': openapi.Schema(type=openapi.TYPE_STRING, description='Человекочитаемое представление ключа фильтрации '
                                                                      '(можно показывать в интерфейсе)'),
        'key': openapi.Schema(type=openapi.TYPE_STRING, description='Значение ключа')
    })

filter_description_schema = openapi.Schema(
    type=openapi.TYPE_OBJECT, description='Полная информация о фильтре',
    properties={
        'title': openapi.Schema(type=openapi.TYPE_STRING, description='Название / заголовок фильтра'),
        'keys': openapi.Schema(type=openapi.TYPE_ARRAY, description='Ключи фильтра', items=filter_item_schema),
    }
)


def get_carousel_scheme(multiple_carousels=False):
    properties = {
        CarouselSerializer.FIELD_CAROUSEL_ID: openapi.Schema(type=openapi.TYPE_STRING, description='ID карусели'),
        CarouselSerializer.FIELD_CACHE_HASH: openapi.Schema(
            type=openapi.TYPE_STRING, description='ID выдачи карусели. Нужен для недопущения дублей'),
        CarouselSerializer.FIELD_AVATAR: openapi.Schema(type=openapi.TYPE_STRING, description='Картинка карусели'),
        CarouselSerializer.FIELD_INCLUDES: openapi.Schema(
            type=openapi.TYPE_ARRAY, description='Список документов', items=openapi.Schema(
                type=openapi.TYPE_OBJECT, description='Документы карусели',
                properties=episode_spec.get_carousel_episode_properties())),
        KpMultiSelectionsResponseFields.FIELD_CAROUSEL_TYPE: openapi.Schema(
            type=openapi.TYPE_STRING, description='Тип карусели(как она должна выглядеть). Может отсутствовать - '
                                                  'в этом случае, считается что карусель обычная',
            enum=[CarouselType.TYPE_SQUARE]),
        api.vh._TRACKING_PARAM_APP_HOST_REQ_ID: openapi.Schema(
            description="ID request'а видеохостинга в Apphost", type=openapi.TYPE_STRING),
        api.vh._TRACKING_PARAM_REQ_ID: openapi.Schema(
            description="ID request'а видеохостинга", type=openapi.TYPE_STRING),
        api.vh._TRACKING_PARAM_USER_DATA: openapi.Schema(
            description="Дополнительная информация о запросе в VH", type=openapi.TYPE_OBJECT, properties={
                api.vh._TRACKING_PARAM_REQ_ID: openapi.Schema(
                    description="Еще один ID request'а в VH", type=openapi.TYPE_STRING),
            })
    }
    if multiple_carousels:
        properties[CarouselSerializer.FIELD_HAS_VERTICAL] = openapi.Schema(
            type=openapi.TYPE_BOOLEAN, description='Содержит ли карусель горизонтальные постеры')
        properties[CarouselSerializer.FIELD_TITLE] = openapi.Schema(
            type=openapi.TYPE_STRING, description='Заголовок карусели')
        properties[CarouselSerializer.FIELD_RANK] = openapi.Schema(
            type=openapi.TYPE_INTEGER, description='Целое число для задание порядка карусели')
        return openapi.Schema(type=openapi.TYPE_OBJECT, properties=properties, required=[
            CarouselSerializer.FIELD_CAROUSEL_ID, CarouselSerializer.FIELD_INCLUDES,
            CarouselSerializer.FIELD_HAS_VERTICAL, CarouselSerializer.FIELD_TITLE, CarouselSerializer.FIELD_RANK])
    else:
        properties[CarouselSerializer.FIELD_FILTER] = openapi.Schema(
            type=openapi.TYPE_OBJECT,
            description='Список доступных фильтров. \r\n'
                        'Чтобы применить выбранные пользователем фильтры, нужно взять все выбранные фильтры, и '
                        'объеденить значения(если оно не `null`) их поля `key` через символ `|`.\r\n'
                        'Пример:\r\n'
                        f'{FILTER_EXAMPLE}'
                        f'1). Взять базовый `url`:\r\n'
                        f'`http://my.url.com/api/v7/carousel?base_cgi_param_1=abc&cbase_gi_param_2=def`\r\n\r\n'
                        f'2). Взять выбранные пользователем фильтры(предположим что пользователь выбрал `Key 1` в 1ом '
                        f'фильтре, и `Super key 2`) и объеденить в строку символом `|`:\r\n'
                        f'`key=1|superkey=2`, urlencode("`key=1|superkey=2`")="`key%3D1%7Csuperkey%3D2`"\r\n\r\n'
                        f'3). Сделать `urlappend(urlencode(<строка из п.2>))` этой строки в базовый `url` по ключу '
                        f'`filter`:\r\n'
                        f'`http://my.url.com/api/v7/carousel?base_cgi_param_1=abc&cbase_gi_param_2=def&filter='
                        f'key%3D1%7Csuperkey%3D2`\r\n\r\n'
                        f'По этому url\'у можно получить фильтрованную карусель.'
                        f'В случае если фильтры меняются - карусель надо перезапрашивать заново '
                        f'с новым параметром `filter`',
            properties={
                'base_url': openapi.Schema(
                    type=openapi.TYPE_STRING,
                    description='Базовый URL для фильтрования. К нему нужно будет делать '
                                'urlappend cgi-параметры, для непосредственного фильтрования, '
                                'и по этому URLу идти за отфильтрованными данными'),
                'filters': openapi.Schema(type=openapi.TYPE_ARRAY, items=filter_description_schema)
            })
        return openapi.Schema(type=openapi.TYPE_OBJECT, properties=properties, required=[
            CarouselSerializer.FIELD_CAROUSEL_ID, CarouselSerializer.FIELD_INCLUDES])


CarouselsV7Spec = dict(
    operation_description='Ручка для получения списка каруселей',
    operation_summary='Список каруселей',
    responses={
        status.HTTP_200_OK: openapi.Response(
            'Список каруселей для запрошенной категории',
            openapi.Schema(type=openapi.TYPE_OBJECT, properties={
                CarouselsV7Serializer.FIELD_CAROUSELS: openapi.Schema(
                    description='Список каруселей', type=openapi.TYPE_ARRAY, items=get_carousel_scheme(True)),
                VhFeed.FIELD_CACHE_HASH: openapi.Schema(
                    description='Уникальный идентификатор выдачи', type=openapi.TYPE_STRING),
            }, required=[CarouselsV7Serializer.FIELD_CAROUSELS]),
        ),
        BackendAPIException.status_code: openapi.Response(
            'Ошибка валидации результата с сервера: некорректные данные от VH / OTT или VH /OTT недоступен.',
        ),
        status.HTTP_400_BAD_REQUEST: openapi.Response(
            'Ошибка в параметрах запроса.'
        )
    },
    query_serializer=serializers.CarouselsV7Validator,
)

CarouselV7Spec = dict(
    operation_summary='Список документов карусели',
    operation_description='Ручка для получения списка документов карусели',
    responses={
        status.HTTP_200_OK: openapi.Response(
            'Список каруселей для запрошенной категории', get_carousel_scheme()),
        BackendAPIException.status_code: openapi.Response(
            'Ошибка валидации результата с сервера: некорректные данные от VH / OTT или VH /OTT недоступен.',
        ),
        status.HTTP_400_BAD_REQUEST: openapi.Response(
            'Ошибка в параметрах запроса.'
        )
    },
    manual_parameters=[
        openapi.Parameter(
            name='carousel_id',
            in_=openapi.IN_QUERY,
            description='Id карусели VH',
            type=openapi.TYPE_STRING,
            required=True
        ),
        openapi.Parameter(
            name=headers.QUASAR_DEVICE_ID,
            in_=openapi.IN_HEADER,
            description='Квазарный id устройства (нужен для карточки отеля)',
            type=openapi.TYPE_STRING,
            required=False
        ),
    ],
)
