from collections import namedtuple, defaultdict
from typing import List

ChannelInfo = namedtuple('ChannelInfo', ('id', 'categories', 'number'))


class Channels:
    info_category_title = 'Новостные'

    categories_mapping = {
        'yandex': 'Яндекс',
        'inform': info_category_title,
        'business': info_category_title,
        'films': 'Кино и сериалы',
        'child': 'Детские',
        'entertain': 'Развлекательные',
        'music': 'Музыкальные',
        'sport': 'Спортивные',
        'foreign': 'Международные',
        'region': 'Региональные',
        'educate': 'Познавательные',
        'federal': 'Федеральные',
        'smotreshka': 'Эфирные (Смотрёшка)',
    }

    fallback_title = 'Другие'

    @classmethod
    def get_title(cls, category_id):
        return cls.categories_mapping.get(category_id, cls.fallback_title)

    @classmethod
    def get_categorized(cls, channels: List[ChannelInfo]):
        """
        Группирует каналы по категориям

        Категории в ответе отсортированы по следующему принципу:

        "category 1": [1000...1099]
        "category 2": [1100...1199]
        "category 3": [1200...1299]
        ...

        И сами каналы внутри категорий тоже отсортированы по channel.number.
        """
        # сортируем каналы по возрастанию channel_smarttv_number
        channels.sort(key=lambda channel: channel.number)

        categories = []
        titles2ids = defaultdict(list)
        for item in channels:
            if len(item.categories) == 0:
                title = cls.fallback_title
            else:
                title = cls.get_title(item.categories[0])

            if title not in categories:
                categories.append(title)

            # и сами каналы в категории отсортированы от меньших number к большим
            titles2ids[title].append(item.id)

        result = []
        for key in categories:
            result.append(
                {
                    'name': key,
                    'channel_ids': titles2ids[key]
                }
            )

        return result
