import hashlib
import json
import logging
import uuid
from typing import Optional, Iterable
from urllib.parse import urlparse

from alice.memento.proto.api_pb2 import TReqGetUserObjects, EConfigKey, TConfigKeyAnyPair
from alice.memento.proto.user_configs_pb2 import TSmartTvMusicPromoConfig
from blackbox import BlackboxResponseError, BlackboxConnectionError
from django.conf import settings
from django.core.exceptions import SuspiciousOperation
from django.http import HttpResponse
from django.utils.translation import ugettext_lazy as _
from django.views.decorators.http import require_GET
from plus.utils.signals import stophook
from requests.exceptions import ConnectionError, HTTPError, ChunkedEncodingError
from rest_framework import exceptions, status
from rest_framework.request import Request
from rest_framework.views import APIView as BaseAPIView
from smarttv.utils import add_log_context, headers
from yaphone.utils.django import get_http_header
from yaphone.utils.parsers import parsers

from smarttv.droideka import unistat
from smarttv.droideka.proxy import blackbox, tvm
from smarttv.droideka.proxy import kp_profile
from smarttv.droideka.proxy.api import usaas, memento
from smarttv.droideka.proxy.constants.home_app_versions import VER_1_4, VER_1_5
from smarttv.droideka.proxy.serializers import serializers
from smarttv.droideka.proxy.swagger.base import swagger_schema
from smarttv.droideka.proxy.swagger.check_auth import CheckAuthSpec
from smarttv.droideka.utils import PlatformType, PlatformInfo, RequestInfo, calculate_icookie, ParsedVersion, \
    MementoConfigs
from smarttv.droideka.utils.default_values import DEFAULT_DICT

logger = logging.getLogger(__name__)

ping_count = unistat.manager.get_counter('ping_count')
missing_quasar_device_id_header_count = unistat.manager.get_counter('missing_quasar_device_id_header_count')

# version where TVANDROID-2204 bugfix released
FIXED_DEVICE_IDS_VERSION = VER_1_4
# from 1.5 device must send quasar_device_id in header on each request
QUASAR_DEVICE_ID_HEADER_VERSION = VER_1_5


class BackendAPIException(exceptions.APIException):
    status_code = status.HTTP_503_SERVICE_UNAVAILABLE
    default_detail = _('A third party service is either unavailable, or returns invalid data.')


@require_GET
def ping(_):
    ping_count.increment()
    if stophook.is_stop_signal_received():
        return HttpResponse(status=500)
    return HttpResponse(status=200)


@require_GET
def unistat_handler(_):
    metrics = unistat.manager.get_counter_metrics()
    return HttpResponse(json.dumps(metrics), status=200)


class PlatformMixin:
    DEFAULT_APP_VERSION = '1.1'
    DEFAULT_ANDROID_VERSION = '7.1.1'

    platform_check_required = True

    @staticmethod
    def parse_user_agent(user_agent: Optional[str]):
        if user_agent:
            return parsers.UaParser.parse(user_agent)
        return DEFAULT_DICT

    def _get_android_app_version(self, user_agent):
        app_version_string = user_agent.get('app_version_string')
        app_name = user_agent.get('app_name', '')
        if (
            (app_version_string is None) or
            ('.' not in app_version_string) or
            ('Dalvik' in app_name)
        ):
            return self.DEFAULT_APP_VERSION
        else:
            return app_version_string

    def _get_android_platform_version(self, user_agent):
        os_version = user_agent.get('os_version')
        if not os_version or not os_version.replace('.', '').isdigit():
            return self.DEFAULT_ANDROID_VERSION
        return os_version

    def _get_platform_info(self, request):
        context = {PlatformType.KEY: PlatformType.ANDROID}
        platform_data_serializer = serializers.AndroidPlatformSerializer
        user_agent = self.parse_user_agent(get_http_header(request, headers.USER_AGENT_HEADER))
        request_data = {
            PlatformInfo.KEY_PLATFORM_VERSION: self._get_android_platform_version(user_agent),
            PlatformInfo.KEY_APP_VERSION: self._get_android_app_version(user_agent),
            PlatformInfo.KEY_DEVICE_MANUFACTURER: user_agent.get('device_manufacturer'),
            PlatformInfo.KEY_DEVICE_MODEL: user_agent.get('device_model'),
            PlatformInfo.KEY_QUASAR_PLATFORM: get_http_header(request, headers.QUASAR_PLATFORM),
        }
        serializer = platform_data_serializer(instance=request_data, context=context)
        validator = platform_data_serializer(data=serializer.data)
        if not validator.is_valid():
            raise SuspiciousOperation(validator.errors)
        return PlatformInfo(
            platform_version=validator.validated_data.get(PlatformInfo.KEY_PLATFORM_VERSION),
            app_version=validator.validated_data.get(PlatformInfo.KEY_APP_VERSION),
            device_manufacturer=validator.validated_data.get(PlatformInfo.KEY_DEVICE_MANUFACTURER),
            device_model=validator.validated_data.get(PlatformInfo.KEY_DEVICE_MODEL),
            quasar_platform=validator.validated_data.get(PlatformInfo.KEY_QUASAR_PLATFORM)
        )

    # noinspection PyUnresolvedReferences
    def initial(self, request, *args, **kwargs):
        if self.platform_check_required:
            url = request.build_absolute_uri()
            uri = urlparse(url)
            path = uri.path

            logger.debug('Path: %s', path)
            request.platform_info = self._get_platform_info(request)
            logger.info('Platform info: %s', request.platform_info)
        else:
            request.platform_info = PlatformInfo()
        return super().initial(request, *args, **kwargs)


class RequestValidatorMixin:
    validator_class = None
    required_headers = None

    def get_validated_data(self, request, validator_class=None):
        request_data = request.data or request.query_params
        validator_class = validator_class or self.validator_class
        validator = validator_class(data=request_data)
        validator.is_valid(raise_exception=True)
        return validator.validated_data

    def validate_headers(self, request):
        for header in self.required_headers:
            if not get_http_header(request, header):
                raise exceptions.ValidationError(f'{header} is missing but required')

    # noinspection PyUnresolvedReferences
    def initial(self, request, *args, **kwargs):
        if self.required_headers:
            self.validate_headers(request)
        super().initial(request, *args, **kwargs)


class ResponseValidatorMixin:
    """
    Validates response from third party backend
    """
    response_validator_class = None
    response_validator_error = exceptions.APIException
    error_message = 'Error parsing response for %s: %s'

    def get_validated_response(self, data, hook_name, validator_context=None, many=False):
        validator = self.response_validator_class(data=data, context=validator_context, many=many)
        if not validator.is_valid():
            logger.error(self.error_message, hook_name, validator.errors)
            raise self.response_validator_error()
        return validator.data


class APIView(RequestValidatorMixin, ResponseValidatorMixin, BaseAPIView):
    KEY_MESSAGE = 'message'

    PASS_HEADERS = (headers.AUTHORIZATION_HEADER, headers.USER_AGENT_HEADER, headers.COOKIE_HEADER)

    blackbox_client = None
    _user_info = None
    authorization_required = False
    oauth_token_validation_required = True
    need_kp_profile_creation = True

    # noinspection PyAttributeOutsideInit
    def initial(self, request, *args, **kwargs):
        request.request_info = RequestInfo()
        self.user_ip = self.get_user_ip(request)
        self.process_user_profile(request)
        request.request_info.memento_configs = self.load_memento_configs(request)
        request.request_info.is_tandem = self.is_tandem(request)
        self.calculate_ids(request)
        self.add_experiments(request)
        super().initial(request, *args, **kwargs)

    def finalize_response(self, request, response, *args, **kwargs):
        if request.request_info.icookie is not None:
            response[headers.DEVICE_ICOOKIE_HEADER] = request.request_info.icookie
        if request.request_info.experiments_icookie is not None:
            response[headers.EXP_DEVICE_ICOOKIE_HEADER] = request.request_info.experiments_icookie

        return super(APIView, self).finalize_response(request, response, *args, **kwargs)

    def calculate_ids(self, request):
        device_id = get_http_header(request, headers.DEVICE_ID_HEADER)

        icookie = calculate_icookie(device_id)
        request.request_info.icookie = icookie
        add_log_context(device_icookie=icookie)

        platform_info = getattr(request, 'platform_info', None)
        if platform_info is not None and ParsedVersion(platform_info.app_version) < FIXED_DEVICE_IDS_VERSION:

            # Bug workaround: we use more stable "experiments_icookie" for uaas based on ethernet mac
            # For client versions where deviceid is unstable
            # https://st.yandex-team.ru/SMARTTVBACKEND-673 for details

            ethernet_mac = get_http_header(request, headers.ETHERNET_MAC_HEADER)
            if ethernet_mac:
                hardware_id = hashlib.md5(ethernet_mac.encode('ascii')).hexdigest()
                experiments_icookie = calculate_icookie(hardware_id)
                request.request_info.experiments_icookie = experiments_icookie
                add_log_context(experiments_icookie=experiments_icookie)
            else:
                request.request_info.experiments_icookie = None
        else:
            request.request_info.experiments_icookie = icookie
        request.request_info.ethernet_mac = get_http_header(request, headers.ETHERNET_MAC_HEADER)

        quasar_device_id = get_http_header(request, headers.QUASAR_DEVICE_ID)
        if quasar_device_id:
            request.request_info.quasar_device_id = quasar_device_id
        elif platform_info.app_version is not None and \
                ParsedVersion(platform_info.app_version) >= QUASAR_DEVICE_ID_HEADER_VERSION:
            missing_quasar_device_id_header_count.increment()
            logger.warning(
                f'{headers.QUASAR_DEVICE_ID} header is missing for device version {platform_info.app_version}')

    def get_user_info(self, request):
        if self._user_info is None:
            if settings.STRESS_ENV:
                self._user_info = blackbox.UserInfo()
                return self._user_info

            client = self.create_bb_client(request)
            if client is None:
                if not self.authorization_required:
                    self._user_info = blackbox.UserInfo()
                else:
                    raise exceptions.AuthenticationFailed('OAuth token or session_id cookie required')
            else:
                try:
                    self._user_info = client.get_user_info(blackbox.plus_fields, request=request)
                except blackbox.InvalidTokenError:
                    if (client.ignore_expired_token() or
                       not self.authorization_required and not self.oauth_token_validation_required):
                        self._user_info = blackbox.UserInfo()
                    else:
                        raise exceptions.AuthenticationFailed('expired token')
                except tvm.TicketError as ex:
                    logger.error(ex)
                    raise exceptions.APIException(detail='tvm ticket problem', code=503)
                except BlackboxConnectionError:
                    msg = 'Blackbox connection error'
                    logger.warning(msg)
                    raise exceptions.APIException(detail=msg, code=503)
                except BlackboxResponseError as ex:
                    logger.info('Blackbox error, code=%s, response: %s', ex.status, ex.content)
                    if status.is_client_error(ex.status):
                        raise exceptions.PermissionDenied()

                    if status.is_server_error(ex.status):
                        raise exceptions.APIException(detail='Blackbox call error', code=503)

            if self._user_info.passport_uid is not None:
                add_log_context(passport_uid=self._user_info.passport_uid)

        return self._user_info

    @staticmethod
    def get_uuid(request):
        uuid_string = get_http_header(request, headers.UUID_HEADER)
        if not uuid_string:
            raise exceptions.ValidationError(f'{headers.UUID_HEADER} is missing but required')
        try:
            return uuid.UUID(uuid_string)
        except (TypeError, ValueError):
            raise exceptions.ValidationError(f'{headers.UUID_HEADER} has wrong format')

    def vh_headers(self, request):
        vh_headers = {}
        for header_name in self.PASS_HEADERS:
            value = get_http_header(request, header_name)
            if value:
                vh_headers[header_name] = value
        if self.user_ip:
            vh_headers[headers.FORWARDED_HEADER] = self.user_ip
        if request.request_info.icookie is not None:
            vh_headers[headers.VH_ICOOKIE_HEADER] = request.request_info.icookie
        return vh_headers

    @staticmethod
    def get_user_ip(request):
        user_ip = request.headers.get(headers.USER_IP_HEADER)
        if user_ip:
            return user_ip.strip()
        return None

    def create_bb_client(self, request) -> Optional[blackbox.BlackBoxClient]:
        if get_http_header(request, headers.AUTHORIZATION_HEADER):
            return blackbox.AndroidBlackBoxClient(request)
        if get_http_header(request, headers.TVM_USER_TICKET) or get_http_header(request, headers.TVM_SERVICE_TICKET):
            return blackbox.TvmBlackboxClient(request)
        return None

    def process_user_profile(self, request: Request):
        if not get_http_header(request, headers.AUTHORIZATION_HEADER):
            # don't validate token if it's not required, or if token is absent
            return
        try:
            user_info = self.get_user_info(request)
            request.request_info.authorized = True
            if self.need_kp_profile_creation:
                kp_profile.ensure_created(user_info.passport_uid, self.vh_headers(request), request)
        except (exceptions.APIException, exceptions.AuthenticationFailed, exceptions.PermissionDenied):
            if self.oauth_token_validation_required:
                raise
        except ChunkedEncodingError:
            logger.exception('Can not read from memento')
        except Exception:
            logger.exception('Unknown exception in memento')
        finally:
            request.user_info = self._user_info

    def load_memento_configs(self, request) -> Optional[MementoConfigs]:
        if not self._user_info or not self._user_info.user_ticket:
            return None
        request_headers = tvm.add_service_ticket(settings.MEMENTO_CLIENT_ID)
        request_headers[headers.TVM_USER_TICKET] = self._user_info.user_ticket
        device_id = get_http_header(request, headers.QUASAR_DEVICE_ID) or \
                    get_http_header(request, headers.DEVICE_ID_HEADER)
        request_data = TReqGetUserObjects(
            CurrentSurfaceId=device_id,
            Keys=(EConfigKey.CK_SMART_TV_MUSIC_PROMO,)
        )
        try:
            response = memento.client.get_objects(request_data, request_headers)
        except (ConnectionError, HTTPError, exceptions.APIException):
            logger.error('Error loading memento config', exc_info=True)
            return MementoConfigs()
        memento_configs = MementoConfigs(
            music_promo_config=self._find_memento_config(EConfigKey.CK_SMART_TV_MUSIC_PROMO, response.UserConfigs,
                                                         TSmartTvMusicPromoConfig)
        )
        return memento_configs

    def get_app_version(self, request) -> Optional[str]:
        return None

    def add_experiments(self, request):
        validated_data = self.get_validated_data(request, validator_class=serializers.ExperimentsValidator)
        test_ids = validated_data.get('test_ids')
        request.request_info.experiments = usaas.LazyExperiments(
            test_ids=test_ids,
            user_ip=self.user_ip,
            app_version=self.get_app_version(request),
            auth_header=get_http_header(request, headers.AUTHORIZATION_HEADER),
            request=request
        )

    def _find_memento_config(self, key: EConfigKey, user_configs: Iterable[TConfigKeyAnyPair], result_class):
        if not user_configs:
            return None
        for pair in user_configs:
            if pair.Key == key:
                result = result_class()
                pair.Value.Unpack(result)
                return result

    def is_tandem(self, request: Request):
        return request.headers.get(headers.TANDEM) == '1'


class NoExperimentsMixin:
    def add_experiments(self, request):
        request.request_info.experiments = usaas.StubExperiments()


class PlatformAPIView(PlatformMixin, APIView):
    def get_app_version(self, request) -> Optional[str]:
        return request.platform_info.app_version


class CheckAuthView(PlatformAPIView):
    """
    Dummy view to validate authorization token
    """
    required_headers = (headers.AUTHORIZATION_HEADER,)
    authorization_required = True
    need_kp_profile_creation = False

    @swagger_schema(CheckAuthSpec)
    def get(self, request):
        self.get_user_info(request)
        return HttpResponse(status=status.HTTP_204_NO_CONTENT)
