import logging

from typing import Optional
from smarttv.droideka.utils.decorators import cache_memoize
from django.conf import settings
import json
from google.protobuf import json_format

from rest_framework.exceptions import NotFound, APIException
from rest_framework.response import Response

from smarttv.droideka.proxy import api, result_builder, cache
from smarttv.droideka.proxy.api import vh
from smarttv.droideka.proxy.cinema import cinema_db
from smarttv.droideka.proxy.serializers.fast_response import FullCardDetail as FastFullCardDetail, EpisodeInfo, \
    ThinCardDetailsInfo, RatingsCardDetailSerializer, FullCardDetailV7, ProgressCardDetailSerializer
from smarttv.droideka.proxy.serializers.protobuf import ContentDetailsDeviceStateSerializer
from smarttv.droideka.proxy.serializers.serializers import CardDetailValidator, CardDetailV4Validator, \
    CardDetailV9Validator
from smarttv.droideka.proxy.swagger.base import smarttv_swagger_schema, swagger_schema
from smarttv.droideka.proxy.swagger.card import FullCardDetailSpec, ThinCardDetailSpec, RatingsCardDetailSpec, \
    FullCardDetailV6Spec, ProgressCardDetailSpec
from smarttv.droideka.proxy.views.base import PlatformAPIView, NoExperimentsMixin
from smarttv.droideka.utils import get_payload_fields
from smarttv.droideka.proxy.constants.reserved_shared_pref_keys import PREF_PAID_CHANNEL_STUB_HEADER_IMAGE, \
    PREF_PAID_CHANNEL_STUB_BODY_TEXT
from smarttv.droideka.proxy.models import SharedPreferences
from smarttv.droideka.proxy.constants.card import NEED_PAID_CHANNELS_STUB, PAID_CHANNEL_STUB_HEADER_IMAGE, \
    PAID_CHANNEL_STUB_BODY_TEXT
from smarttv.droideka.proxy.s3mds import s3_client
from smarttv.utils.headers import AUTHORIZATION_HEADER
from smarttv.droideka.proxy.mem_cache import onto_id_mapping, content_type_mapping
from alice.protos.data.video.content_details_pb2 import TContentDetailsItem
from smarttv.droideka.unistat.metrics import CacheType, RedisCachePlace
from smarttv.droideka.proxy.common import ensure_ott_content_id
from smarttv.droideka.proxy import blackbox

logger = logging.getLogger(__name__)

CARD_DETAIL = 'content_detail'
FIELD_DEVICE_STATE = 'device_state'

CACHE_CONTROL_SETTINGS = {'private': True, 'max_age': settings.BROWSER_CACHE_TIME}


class ContentDetails:
    series_content_types = (api.es.OO_CONTENT_TYPE_SERIES, api.vh.CONTENT_TYPE_SERIES)

    @cache_memoize(timeout=settings.DEFAULT_RESPONSE_CACHE_TIME,
                   args_rewrite=lambda self, content_id, *args: ['thin', content_id],
                   skipif=lambda _, content_id, headers, *args, **kwargs: AUTHORIZATION_HEADER in headers,
                   **cache.get_cache_memoize_callables(RedisCachePlace.THIN_CARD_DETAIL.value, CacheType.REDIS))
    def get_thin_card_detail(self, content_id, headers, request, need_ad_config=False,
                             additional_context_params=None):
        builder = result_builder.ContentDetailV4ResultBuilder(headers, content_id, request)
        # noinspection PyUnresolvedReferences
        builder.extend(result_builder.ContentDetailResultBuilderExtension(request=self.request))
        result = builder.get_result()
        context = {
            ThinCardDetailsInfo.NEED_AD_CONFIG: need_ad_config,
            'platform_info': request.platform_info
        }
        if additional_context_params:
            context.update(additional_context_params)
        return ThinCardDetailsInfo().serialize(result, context)

    def get_ott_card_detail(self, content_id, onto_id, content_type, headers, passport_uid,
                            serializer, request, need_series=True):
        builder = result_builder.ContentDetailV4ResultBuilder(headers, content_id, request, onto_id, content_type)
        # noinspection PyUnresolvedReferences
        builder.add_extension(result_builder.ContentDetailResultBuilderExtension(request=self.request))
        entity_search_result_builder = result_builder.EntitySearchSingleItemExtension(
            request=request, headers=headers, passport_uid=passport_uid)
        entity_search_result_builder.add_extension(result_builder.TrailersSignExtension(onto_id, request, headers))
        builder.add_extension(entity_search_result_builder)
        builder.add_extension(result_builder.OttMetadataExtension())
        builder.add_extension(result_builder.OttFilmsToWatchExtension(
            content_id=content_id,
            headers=headers,
            request=request,
        ))
        is_series = content_type in self.series_content_types
        if need_series:
            if is_series:
                builder.add_extension(result_builder.OttSeriesStructureExtension())
            else:
                logger.info('Document %s is not a series(%s). Skip getting series structure', content_id, content_type)

        result = builder.get_result()

        es_response_not_presented = result_builder.EntitySearchSingleItemExtension.RESULT_FIELD_NAME not in result
        series_response_not_presented = result_builder.OttSeriesStructureExtension.RESULT_FIELD_NAME not in result
        if es_response_not_presented or need_series and is_series and series_response_not_presented:
            logger.debug('OTT card detail result fields: %r', get_payload_fields(result))
            raise NotFound(f'Document \'{content_id}\' not found')

        serialized_result = serializer.serialize(result, {
            'platform_info': request.platform_info,
            'content_type': content_type,
            'need_series': need_series
        })

        self.process_cinemas(serialized_result)

        if serialized_result and not serialized_result.get('onto_category'):
            logger.warning("Empty 'onto_category' for (%s, %s, %s)", content_id, onto_id, content_type)

        return serialized_result

    def process_cinemas(self, data):
        """
        Удаляет из списка кинотеатры, которые не включены явно в админке
        Заменяет иконки, если в админке прописано значение
        """
        allowed = cinema_db.get_allowed_cinema_codes()
        result = []

        for cinema in data['cinemas']:
            if cinema['code'] in allowed:
                favicon = cinema_db.get_admin_favicon(cinema['code'])
                if favicon:
                    cinema['favicon'] = favicon
                result.append(cinema)
            else:
                logger.info('cinema %s is not enabled in admin interface', cinema['code'])

        data['cinemas'] = result

    def get_vh_card_detail(self, content_id, headers, request):
        builder = result_builder.ContentDetailV4ResultBuilder(headers, content_id, request)
        # noinspection PyUnresolvedReferences
        builder.add_extension(result_builder.ContentDetailResultBuilderExtension(request=self.request))
        result = builder.get_result()

        if CARD_DETAIL not in result:
            logger.debug('VH card detail result fields: %r', get_payload_fields(result))
            raise NotFound(f'Document \'{content_id}\' not found')

        return EpisodeInfo().serialize(result.get(CARD_DETAIL), {'platform_info': request.platform_info})

    @cache_memoize(timeout=settings.DEFAULT_RESPONSE_CACHE_TIME,
                   args_rewrite=lambda self, content_id, *args: [content_id],
                   **cache.get_cache_memoize_callables(RedisCachePlace.RATINGS_CARD_DETAIL.value, CacheType.REDIS))
    def get_ratings_card_detail(self, content_id, headers, passport_uid, request):
        # noinspection PyUnresolvedReferences
        builder = result_builder.ContentDetailResultBuilder(content_id=content_id,
                                                            headers=headers, request=self.request)
        builder.add_extension(result_builder.EntitySearchSingleItemExtension(request, headers, passport_uid))
        result = builder.get_result()

        if result_builder.EntitySearchSingleItemExtension.RESULT_FIELD_NAME not in result:
            logger.error('Not found ratings for (%s, %s)', content_id, result)
            raise NotFound(f'Document \'{content_id}\' not found')

        return RatingsCardDetailSerializer().serialize(result)

    def get_progress_card_detail(self, content_id, headers):
        # noinspection PyUnresolvedReferences
        builder = result_builder.ContentDetailResultBuilder(content_id=content_id,
                                                            headers=headers, request=self.request)
        result = builder.get_result()
        if result.get('includes'):
            result = result['includes'][0]
        return ProgressCardDetailSerializer().serialize(result, context={'platform_info': self.request.platform_info})

    def get_full_card_detail(self, content_id: str, headers: dict, passport_uid, request, ott_card_serializer,
                             vh_card_serializer):
        platform_info = request.platform_info
        builder = result_builder.FullCardDetail(content_id, headers, passport_uid, request)
        content_detail_extension = result_builder.ContentDetailResultBuilderExtension(request)
        entity_search_result_builder = result_builder.EntitySearchSingleItemExtension(
            request=request, headers=headers, passport_uid=passport_uid)
        entity_search_result_builder.add_extension(result_builder.TrailersSignExtension(content_id, request, headers))
        content_detail_extension.add_extension(entity_search_result_builder)
        content_detail_extension.add_extension(result_builder.OttMetadataExtension())
        content_detail_extension.add_extension(result_builder.OttSeriesStructureExtension())
        content_detail_extension.add_extension(result_builder.OttFilmsToWatchExtension(
            content_id=content_id,
            headers=headers,
            request=request
        ))
        builder.add_extension(content_detail_extension)

        result = builder.get_result()
        content_detail = result.get(result_builder.ContentDetailResultBuilderExtension.RESULT_FIELD_NAME)
        if not content_detail or result_builder.EntitySearchSingleItemExtension.RESULT_FIELD_NAME not in result:
            raise NotFound(f'Document \'{content_id}\' not found')
        content_type = content_detail.get('content_type_name')
        if not vh.is_ott_content(content_type):
            # This not OTT document, handle as VH document
            return vh_card_serializer.serialize(content_detail, {'platform_info': platform_info})
        return ott_card_serializer.serialize(result, {'platform_info': platform_info, 'content_type': content_type,
                                                      'need_series': False})

    def get_pirate_card_detail(self, onto_id, vh_headers, request, passport_uid, serializer):

        builder = result_builder.ContentDetailV4ResultBuilder(
            headers=vh_headers,
            content_id=None,
            request=request,
            onto_id=onto_id,
            content_type=None,
        )
        entity_search_result_builder = result_builder.EntitySearchSingleItemExtension(
            request=request, headers=vh_headers, passport_uid=passport_uid
        )
        builder.add_extension(entity_search_result_builder)
        result = builder.get_result()

        serialized_result = serializer.serialize(
            result,
            {
                "platform_info": request.platform_info,
                # "content_type": "MOVIE",  todo: recheck this
                "need_series": False,
            },
        )

        return serialized_result


class CardDetailV4View(ContentDetails, PlatformAPIView):
    validator_class = CardDetailV4Validator
    response_serializer = FastFullCardDetail()
    need_series = True
    need_url_map = False

    @smarttv_swagger_schema(FullCardDetailSpec, 'Ручка для получения полной информации о фильме')
    def get(self, request):
        data = self.get_validated_data(request)
        content_id = data['content_id']
        onto_id = data.get('onto_id')
        content_type = data.get('content_type')
        headers = self.vh_headers(request)
        user_info = self.get_user_info(request)

        ensure_ott_content_id(content_id)
        if vh.is_ott_content(content_type):
            card_details = self.get_ott_card_detail(
                content_id,
                onto_id,
                content_type,
                headers=headers,
                passport_uid=user_info.passport_uid,
                serializer=self.response_serializer,
                request=request,
                need_series=self.need_series
            )
        else:
            card_details = self.get_vh_card_detail(content_id, headers, request)

        return Response(card_details)


class CardDetailV6View(CardDetailV4View):
    validator_class = CardDetailV4Validator
    response_serializer = FastFullCardDetail()
    need_series = False

    @swagger_schema(FullCardDetailV6Spec)
    def get(self, request):
        return super().get(request)


class CardDetailV7View(CardDetailV6View):
    validator_class = CardDetailV4Validator
    response_serializer = FullCardDetailV7()
    need_series = False
    need_url_map = True

    vh_card_serializer = EpisodeInfo()
    device_state_serializer = ContentDetailsDeviceStateSerializer()

    def append_device_state(self, serialized_card: dict):
        device_state_proto = TContentDetailsItem()
        self.device_state_serializer.serialize(serialized_card, device_state_proto)
        serialized_card[FIELD_DEVICE_STATE] = json_format.MessageToDict(device_state_proto)

    @staticmethod
    def try_get_card_missing_params_from_cache(content_id: str) -> tuple:
        return onto_id_mapping.get(content_id), content_type_mapping.get(content_id)

    def get_by_content_id(self, content_id, content_type, onto_id, headers, user_info, request):
        if not onto_id or not content_type:
            onto_id, content_type = self.try_get_card_missing_params_from_cache(content_id)
        if content_type and onto_id:
            if vh.is_ott_content(content_type):
                card_details = self.get_ott_card_detail(
                    content_id,
                    onto_id,
                    content_type,
                    headers=headers,
                    passport_uid=user_info.passport_uid,
                    serializer=self.response_serializer,
                    request=request,
                    need_series=self.need_series
                )
            else:
                raise NotFound(f'Document \'{content_id}\' with content type \'{content_type}\' not found')
        else:
            card_details = self.get_full_card_detail(content_id, headers, user_info.passport_uid, self.request,
                                                     self.response_serializer, self.vh_card_serializer)

        self.append_device_state(card_details)
        return card_details

    @smarttv_swagger_schema(FullCardDetailSpec, 'Ручка для получения полной информации о фильме')
    def get(self, request):
        data = self.get_validated_data(request)
        content_id = data['content_id']
        ensure_ott_content_id(content_id)

        onto_id = data.get('onto_id')
        content_type = data.get('content_type')
        headers = self.vh_headers(request)
        user_info = self.get_user_info(request)

        card_details = self.get_by_content_id(
            content_id=content_id,
            content_type=content_type,
            onto_id=onto_id,
            headers=headers,
            user_info=user_info,
            request=request,
        )

        return Response(card_details)


class CardDetailV9View(CardDetailV7View):
    """
    Карточка контента V9

    В девятой версии появилась возможность загружать данные для пиратской карточки
    только по onto_id.
    """
    NAME = 'card_detail_v9'
    validator_class = CardDetailV9Validator

    def get_by_onto_id(self, onto_id, vh_headers, request, passport_uid, serializer):
        return self.get_pirate_card_detail(
            onto_id=onto_id,
            vh_headers=vh_headers,
            request=request,
            passport_uid=passport_uid,
            serializer=serializer,
        )

    def get(self, request):
        data = self.get_validated_data(request)
        content_id = data.get('content_id')
        onto_id = data.get('onto_id')
        content_type = data.get('content_type')

        vh_headers = self.vh_headers(request)
        user_info = self.get_user_info(request)

        if content_id:
            ensure_ott_content_id(content_id)
            return Response(self.get_by_content_id(
                content_id=content_id,
                content_type=content_type,
                onto_id=onto_id,
                headers=vh_headers,
                user_info=user_info,
                request=request,
            ))

        if onto_id:
            return Response(self.get_by_onto_id(
                onto_id=onto_id,
                vh_headers=vh_headers,
                request=request,
                passport_uid=user_info.passport_uid,
                serializer=self.response_serializer,
            ))

        raise APIException("Either content_id or onto_id must be specified", code=400)


class ThinCardDetailView(ContentDetails, PlatformAPIView):
    validator_class = CardDetailValidator

    need_ad_config = False
    need_paid_channels_stub = False
    paid_channel_header_image = None
    paid_channel_body_text = None

    @property
    def has_subscription(self) -> bool:
        if self._user_info is None:
            return False
        user_info: blackbox.UserInfo = self._user_info
        return user_info.subscription != blackbox.SubscriptionType.UNKNOWN

    @swagger_schema(ThinCardDetailSpec)
    def get(self, request):
        data = self.get_validated_data(request)
        content_id = data['content_id']
        ensure_ott_content_id(content_id)

        headers = self.vh_headers(request)
        additional_context_params = None
        if self.need_paid_channels_stub:
            additional_context_params = {
                NEED_PAID_CHANNELS_STUB: self.need_paid_channels_stub,
                PAID_CHANNEL_STUB_HEADER_IMAGE: self.paid_channel_header_image,
                PAID_CHANNEL_STUB_BODY_TEXT: self.paid_channel_body_text,
            }
        card_details = self.get_thin_card_detail(
            content_id, headers, request, need_ad_config=self.need_ad_config and not self.has_subscription,
            additional_context_params=additional_context_params)
        return Response(card_details)


class ThinCardDetailViewV7(ThinCardDetailView):
    need_ad_config = True
    need_paid_channels_stub = True

    @property
    def paid_channel_header_image(self) -> Optional[list]:
        s3_key = SharedPreferences.get_string(PREF_PAID_CHANNEL_STUB_HEADER_IMAGE)
        if not s3_key:
            return None
        return s3_client.get_url(s3_key)

    @property
    def paid_channel_body_text(self) -> Optional[dict]:
        promo_text_json = SharedPreferences.get_string(PREF_PAID_CHANNEL_STUB_BODY_TEXT)
        if not promo_text_json:
            return None
        return json.loads(promo_text_json)


class RatingsCardDetail(ContentDetails, PlatformAPIView):
    validator_class = CardDetailValidator

    @swagger_schema(RatingsCardDetailSpec)
    def get(self, request):
        data = self.get_validated_data(request)
        content_id = data['content_id']
        ensure_ott_content_id(content_id)
        headers = self.vh_headers(request)
        user_info = self.get_user_info(request)
        return Response(self.get_ratings_card_detail(content_id, headers, user_info.passport_uid, request))


class RatingsCardDetailV4(NoExperimentsMixin, RatingsCardDetail):
    pass


class ProgressCardDetail(ContentDetails, PlatformAPIView):
    validator_class = CardDetailValidator

    @swagger_schema(ProgressCardDetailSpec)
    def get(self, request):
        data = self.get_validated_data(request)
        content_id = data['content_id']
        ensure_ott_content_id(content_id)
        headers = self.vh_headers(request)
        return Response(self.get_progress_card_detail(content_id, headers))
