import logging
from abc import ABC, abstractmethod
from collections.abc import Iterable
from typing import List, Optional
import enum

from django.utils.cache import add_never_cache_headers
from rest_framework.exceptions import NotFound

from django.conf import settings
from rest_framework.response import Response
from smarttv.droideka.proxy import api
from smarttv.droideka.proxy.api.base import BaseApi
from smarttv.droideka.proxy import result_builder
from smarttv.droideka.proxy.serializers import serializers

from smarttv.droideka.proxy.constants.carousels import KpMultiSelection
from smarttv.droideka.proxy.serializers.fast_response import CarouselsSerializer, CarouselsV5Serializer, \
    CarouselSerializer as FastCarouselSerializer, KpCarouselSerializer, CarouselsV7Serializer, \
    KpMultiSelectionCarouselSerializer, EmbeddedSectionSerializer, NEED_MAP_URL
from smarttv.droideka.proxy.response.carousels import KpMultiSelectionsResponseFields
from smarttv.droideka.proxy.constants.carousels import DroidekaCarousel
from smarttv.droideka.proxy.vh.constants import PROMO_CAROUSEL_ID, WhatToSeeOttCarousel, WHAT_TO_SEE_OTT_CAROUSEL_IDS, \
    MAIN_CATEGORY_ID
from smarttv.droideka.proxy.constants.reserved_shared_pref_keys import RECOMMENDATIONS_CAROUSEL_KEY
from smarttv.droideka.proxy.views.base import PlatformAPIView
from smarttv.droideka.proxy.request.carousels import MixedCarouselsInfo, VhCarouselInfo, KpCarouselInfo, \
    KpCategoryInfo, PurchasesCarouselInfo
from smarttv.droideka.proxy.constants.carousels import MusicCategory, VhFeed, KpCarousel, KpCategory, DroidekaCategory, \
    PURCHASES_CAROUSEL_ID
from smarttv.droideka.proxy.categories_provider import categories_provider
from smarttv.droideka.proxy.models import SharedPreferences
from smarttv.droideka.proxy.response.carousels import MixedCarouselsResponse
from smarttv.droideka.proxy.swagger.base import swagger_schema
from smarttv.droideka.proxy.swagger.carousels import CarouselsV7Spec, CarouselV7Spec
from smarttv.droideka.proxy.constants.reserved_shared_pref_keys import PATCHING_CAROUSEL_ID_MAP
from smarttv.droideka.proxy.categories_provider import KEY_CATEGORY_EXPERIMENTS
from smarttv.droideka.proxy.api.usaas import EmptyIcookieException
from smarttv.droideka.utils.url import build_view_url
from smarttv.droideka.proxy.exceptions import GatewayTimeoutError

logger = logging.getLogger(__name__)

carousels_serializer = CarouselsSerializer()
carousel_serializer = FastCarouselSerializer()
kp_carousel_serializer = KpCarouselSerializer()
kp_multi_selection_carousel_serializer = KpMultiSelectionCarouselSerializer()


def log_optional_field_len(obj, key, msg):
    if key in obj and isinstance(obj[key], Iterable):
        logger.debug(msg, len(obj[key]))


class OttExperiment(enum.Enum):
    YA_TV = 'source_ott_3'
    YA_TV_SPECIAL = 'source_ott_3_special'
    YA_TV_SPECIAL_MIX = 'source_ott_3_special_mix'


class CarouselViewName:

    class V5:
        CAROUSEL = 'carousel5'
        CAROUSELS = 'carousels_v5'

    class V6:
        CAROUSEL = 'carousel6'
        CAROUSELS = 'carousels_v6'

    class V7:
        CAROUSEL = 'carousel7'
        CAROUSELS = 'carousels_v7'

    class V8:
        CAROUSEL = 'carousel8'
        CAROUSELS = 'carousels_v8'

    class V9:
        CAROUSEL = 'carousel_v9'
        CAROUSELS = 'carousels_v9'


class NamedItem(ABC):
    # noinspection PyPep8Naming
    @property
    @abstractmethod
    def NAME(self):
        pass


class CarouselsV4View(PlatformAPIView):
    NAME = 'carousels4'
    validator_class = serializers.CarouselsValidator

    def get(self, request):
        data = self.get_validated_data(request)

        category_id = data['category_id']
        response = result_builder.FeedResultBuilder(
            category_id=category_id,
            next_url_name=self.NAME,
            nested_next_url_name=CarouselV4View.NAME,
            max_items_count=data.get('max_items_count'),
            offset=data['offset'],
            limit=data['limit'],
            cache_hash=data.get('cache_hash'),
            restriction_age=data['restriction_age'],
            headers=self.vh_headers(request),
            request=request).get_result()

        context = {'category_id': category_id, 'platform_info': request.platform_info}
        result = carousels_serializer.serialize(response, context)

        response_headers = {}
        api.vh.propagate_vh_tracking_params(response, response_headers)

        response = Response(result, headers=response_headers)
        if not result:
            add_never_cache_headers(response)
        return response


class CarouselV4View(PlatformAPIView):
    NAME = 'carousel4'

    validator_class = serializers.CarouselValidator

    def get(self, request):
        data = self.get_validated_data(request)

        carousel = result_builder.CarouselVideohubResultBuilder(VhCarouselInfo(
            data,
            headers=self.vh_headers(request),
            request=request,
            next_url_name=self.NAME)).get_result()

        context = {'carousel_id': data['carousel_id'], 'platform_info': request.platform_info}
        result = carousel_serializer.serialize(carousel, context)

        api.vh.propagate_vh_tracking_params(carousel, result)

        return Response(result)


class Container(ABC):
    @property
    @abstractmethod
    def nested_item_cls(self) -> NamedItem:
        pass

    @property
    def nested_item_name(self) -> str:
        return self.nested_item_cls.NAME


class CommonCarouselsView(PlatformAPIView, Container, ABC):
    validator_class = serializers.CarouselsValidator
    carousel_view_name = None
    carousels_view_name = None

    def get(self, request):
        data = self.get_validated_data(request)

        offset = data['offset']
        builder = result_builder.FeedResultBuilder(
            category_id=data['category_id'],
            next_url_name=self.NAME,
            nested_next_url_name=self.nested_item_name,
            max_items_count=data.get('max_items_count'),
            offset=offset,
            limit=data['limit'],
            cache_hash=data.get('cache_hash'),
            restriction_age=data['restriction_age'],
            headers=self.vh_headers(request),
            request=request,
        )

        result = builder.get_result()
        context = {'category_id': data['category_id'], 'platform_info': request.platform_info,
                   'request': request,
                   'carousel_view_name': self.carousel_view_name,
                   'carousels_view_name': self.carousels_view_name}
        log_optional_field_len(result, CarouselsSerializer.FIELD_ROOT_LIST, 'Got %s items')

        serialized_result = CarouselsV5Serializer().serialize(result, context)
        if serialized_result and offset == 0 and VhFeed.FIELD_MORE not in serialized_result \
                and data['category_id'] == MAIN_CATEGORY_ID:
            logger.warning('Very small main carousel')

        response = Response(serialized_result)
        if not serialized_result['carousels']:
            add_never_cache_headers(response)
        return response


class CarouselV5View(PlatformAPIView, NamedItem):
    NAME = CarouselViewName.V5.CAROUSEL
    carousel_view_name = CarouselViewName.V5.CAROUSEL
    carousels_view_name = CarouselViewName.V5.CAROUSELS

    validator_class = serializers.CarouselValidator
    need_map_url = False

    result_builder_mapping = {
        KpCarousel.TYPE: (result_builder.KpCarouselResultBuilder, KpCarouselInfo, kp_carousel_serializer)
    }

    KEY_SHOULD_HAVE_PROMO = 'should_have_promo'

    ott_experiment_to_vitrina = {
        OttExperiment.YA_TV: WhatToSeeOttCarousel.YA_TV,
        OttExperiment.YA_TV_SPECIAL: WhatToSeeOttCarousel.YA_TV_SPECIAL,
        OttExperiment.YA_TV_SPECIAL_MIX: WhatToSeeOttCarousel.YA_TV_SPECIAL_MIX,
    }

    def _build_multi_selection_url(
            self, window_id: str, selection_id: str, items_limit: int, nested_next_url_name: str, request):
        args = {
            DroidekaCarousel.FIELD_CAROUSEL_ID: f'{window_id}/{selection_id}',
            DroidekaCarousel.FIELD_LIMIT: settings.EMBEDDED_CAROUSEL_LIMIT,
            DroidekaCarousel.FIELD_CAROUSEL_TYPE: KpCarousel.TYPE,
            DroidekaCarousel.FIELD_MORE_URL_LIMIT: items_limit
        }
        return build_view_url(args, nested_next_url_name, request)

    def get_result_builder(self, carousel_id, carousel_type, offset, internal_recommendation_carousel_id):
        if carousel_id == PURCHASES_CAROUSEL_ID:
            carousel_info_cls = PurchasesCarouselInfo.from_serialized_data
            return result_builder.PurchasesResultBuilder, carousel_info_cls, kp_carousel_serializer
        elif carousel_id in WHAT_TO_SEE_OTT_CAROUSEL_IDS and carousel_type is None:
            carousel_type = KpCarousel.TYPE
        result_builder_by_type = self.result_builder_mapping.get(carousel_type)
        if result_builder_by_type:
            return result_builder_by_type
        if carousel_id == internal_recommendation_carousel_id and not offset:
            return result_builder.RecomendationsCarouselResultBuilder, VhCarouselInfo, carousel_serializer

        return result_builder.CarouselVideohubResultBuilder, VhCarouselInfo, carousel_serializer

    @staticmethod
    def get_patching_carousel_ids() -> dict:
        return SharedPreferences.get_json(PATCHING_CAROUSEL_ID_MAP) or {}

    def get_category_experiments(self, request) -> List[str]:
        try:
            category_experiments = request.request_info.experiments.get_value(KEY_CATEGORY_EXPERIMENTS)
            if not category_experiments or not isinstance(category_experiments, Iterable):
                return []
            return category_experiments
        except BaseApi.RequestError:
            logger.debug('USaaS bad request')
            return []
        except EmptyIcookieException:
            logger.debug('Can not get category experiments because of absent icookie')
            return []

    def extract_ott_experiment(self, category_experiments: List[str]) -> Optional[OttExperiment]:
        for ott_experiment in OttExperiment:
            if ott_experiment.value in category_experiments:
                return ott_experiment
        return None

    def patch_carousel_id_if_necessary(self, serialized_data: dict, request):
        carousel_id = serialized_data['carousel_id']

        category_experiments = self.get_category_experiments(request)
        ott_experiment = self.extract_ott_experiment(category_experiments)
        if carousel_id == PROMO_CAROUSEL_ID and ott_experiment:
            serialized_data['carousel_id'] = self.ott_experiment_to_vitrina[ott_experiment].value
            serialized_data[self.KEY_SHOULD_HAVE_PROMO] = True
            logger.debug('Patch promo carousel id with OTT id: %s', serialized_data['carousel_id'])
            return
        carousel_id_map = self.get_patching_carousel_ids()
        if isinstance(carousel_id_map, dict) and carousel_id in carousel_id_map:
            patching_carousel_id = carousel_id_map[carousel_id]
            if not isinstance(patching_carousel_id, str):
                patching_carousel_id = carousel_id
            serialized_data['carousel_id'] = patching_carousel_id
            serialized_data[self.KEY_SHOULD_HAVE_PROMO] = \
                patching_carousel_id == self.get_internal_recommendations_carousel_id()

    @staticmethod
    def get_internal_recommendations_carousel_id() -> str:
        return SharedPreferences.get_string(RECOMMENDATIONS_CAROUSEL_KEY)

    def patch_kp_multiselection_carousel(self, carousel: dict, request_info: KpCarouselInfo):
        carousel[api.ott.KEY_SELECTION_ID] = request_info.selection_id
        carousel[api.ott.KEY_SELECTION_WINDOW_ID] = request_info.window_id
        for item in carousel[KpMultiSelectionsResponseFields.FIELD_DATA]:
            item_selection_id = item[KpMultiSelectionsResponseFields.FIELD_SELECTION_ID]
            item[api.ott.KEY_SELECTION_ID] = item[KpMultiSelectionsResponseFields.FIELD_SELECTION_ID]
            item[api.ott.KEY_SELECTION_WINDOW_ID] = request_info.window_id
            item[KpMultiSelection.FIELD_MULTISELECTION_URL] = self._build_multi_selection_url(
                request_info.window_id,
                item_selection_id,
                request_info.limit,
                self.carousel_view_name,
                request_info.request
            )

    def get(self, request):
        data = self.get_validated_data(request)

        internal_recommendation_carousel_id = self.get_internal_recommendations_carousel_id()
        self.patch_carousel_id_if_necessary(data, request)
        carousel_id = data['carousel_id']
        carousel_type = data.get('carousel_type')
        offset = data['offset']

        builder_cls, request_info_cls, serializer = self.get_result_builder(
            carousel_id, carousel_type, offset, internal_recommendation_carousel_id)
        request_info = request_info_cls(
            data,
            request,
            self.vh_headers(request),
            self.NAME
        )
        builder = builder_cls(request_info)

        carousel = builder.get_result()
        if carousel is None:
            raise GatewayTimeoutError(f"Timeout while loading carousel '{carousel_id}'")
        context = {'carousel_id': carousel_id, 'platform_info': request.platform_info,
                   NEED_MAP_URL: self.need_map_url, 'request': request, 'carousel_view_name': self.carousel_view_name,
                   'carousels_view_name': self.carousels_view_name}
        if carousel.get(KpMultiSelectionsResponseFields.FIELD_TYPE) == \
                KpMultiSelectionsResponseFields.TYPE_MULTI_SELECTION:
            serializer = kp_multi_selection_carousel_serializer
            self.patch_kp_multiselection_carousel(carousel, request_info)
        result = serializer.serialize(carousel, context)

        api.vh.propagate_vh_tracking_params(carousel, result)

        return Response(result)


class CarouselsV5View(CommonCarouselsView):
    NAME = CarouselViewName.V5.CAROUSELS
    nested_item_cls = CarouselV5View
    carousel_view_name = CarouselViewName.V5.CAROUSEL
    carousels_view_name = CarouselViewName.V5.CAROUSELS


class CarouselV6View(CarouselV5View):
    NAME = CarouselViewName.V6.CAROUSEL
    carousel_view_name = CarouselViewName.V6.CAROUSEL
    carousels_view_name = CarouselViewName.V6.CAROUSELS


class CarouselV7View(CarouselV5View):
    NAME = CarouselViewName.V7.CAROUSEL
    need_map_url = True
    carousel_view_name = CarouselViewName.V7.CAROUSEL
    carousels_view_name = CarouselViewName.V7.CAROUSELS

    @swagger_schema(CarouselV7Spec)
    def get(self, request):
        return super().get(request)


class CarouselV8View(CarouselV7View):
    NAME = CarouselViewName.V8.CAROUSEL

    carousel_view_name = CarouselViewName.V8.CAROUSEL
    carousels_view_name = CarouselViewName.V8.CAROUSELS


class CarouselV9View(CarouselV7View):
    NAME = CarouselViewName.V9.CAROUSEL

    carousel_view_name = CarouselViewName.V9.CAROUSEL
    carousels_view_name = CarouselViewName.V9.CAROUSELS


class CarouselsV6View(CommonCarouselsView):
    NAME = CarouselViewName.V6.CAROUSEL
    nested_item_cls = CarouselV6View
    carousel_view_name = CarouselViewName.V6.CAROUSEL
    carousels_view_name = CarouselViewName.V6.CAROUSELS


class CarouselsV7View(CommonCarouselsView):
    validator_class = serializers.CarouselsV7Validator
    nested_item_cls = CarouselV7View
    carousel_view_name = CarouselViewName.V7.CAROUSEL
    carousels_view_name = CarouselViewName.V7.CAROUSELS
    vh_carousels_serializer = CarouselsV7Serializer()
    kp_carousel_serializer = KpCarouselSerializer()
    embedded_section_serializer = EmbeddedSectionSerializer()
    kp_multi_selection_serializer = KpMultiSelectionCarouselSerializer()

    NAME = 'carousels_v7'
    NESTED_NEXT_URL_NAME = CarouselV7View.NAME

    def mix_result(self, result: MixedCarouselsResponse):
        feed = result.feed
        injectable_carousels: list = result.injectable_carousels
        more_url = result.more_url
        if not feed and not injectable_carousels:
            raise NotFound('Can not find content for requested category')
        if injectable_carousels:
            injectable_carousels.sort(key=lambda container: container.position)
            if feed:
                carousels = feed['carousels']
                for item in injectable_carousels:
                    carousels.insert(item.position, item.carousel)
            else:
                feed = {'carousels': [item.carousel for item in injectable_carousels]}
                more_url = None
        if more_url:
            feed[VhFeed.FIELD_MORE] = more_url
        return feed

    @swagger_schema(CarouselsV7Spec)
    def get(self, request):
        data = self.get_validated_data(request)

        category_id = data[DroidekaCategory.FIELD_CATEGORY_ID]
        category = categories_provider.get_category(category_id)

        if not category:
            logger.info(f'Trying to find category {category_id} in not published')
            category = categories_provider.get_non_published_category(category_id)

        if not category:
            raise NotFound(f'Category {category_id} not found')

        request_info_class, result_builder_class = self.get_result_builder(category['content_type'])

        request_info = request_info_class(
            serialized_data=data,
            request=request,
            headers=self.vh_headers(request),
            next_url_name=self.NAME,
            nested_next_url_name=self.NESTED_NEXT_URL_NAME,
            carousels_next_url_name=CarouselsV7View.NAME,
            carousel_next_url_name=CarouselV7View.NAME,
            category_content_type=category['content_type'],
        )

        result: MixedCarouselsResponse = result_builder_class(request_info).get_result()

        return Response(self.mix_result(result))

    def get_result_builder(self, content_type):
        if content_type == KpCategory.TYPE:
            return KpCategoryInfo, result_builder.MultiSelectionsResultBuilder
        elif content_type == MusicCategory.TYPE:
            # "music" category in left menu
            return MixedCarouselsInfo, result_builder.MusicCategoryResultBuilder
        else:
            # category, based on VH feed - like "main"
            return MixedCarouselsInfo, result_builder.FeedResultBuilderV7


class CarouselsV8View(CarouselsV7View):
    carousel_view_name = CarouselViewName.V8.CAROUSEL
    carousels_view_name = CarouselViewName.V8.CAROUSELS
    nested_item_cls = CarouselV8View

    NAME = 'carousels_v8'
    NESTED_NEXT_URL_NAME = CarouselV8View.NAME


class CarouselsV9View(CarouselsV7View):
    carousel_view_name = CarouselViewName.V9.CAROUSEL
    carousels_view_name = CarouselViewName.V9.CAROUSELS
    nested_item_cls = CarouselV9View

    NAME = CarouselViewName.V9.CAROUSELS
    NESTED_NEXT_URL_NAME = CarouselV9View.NAME
