import logging
from collections import defaultdict
from typing import Optional, Iterable
from rest_framework import status
from rest_framework.response import Response

from smarttv.droideka.proxy.categories_provider import categories_provider, StorageNotAvailable
from smarttv.droideka.proxy.serializers.response import CategorySerializer
from smarttv.droideka.proxy.swagger.base import swagger_schema
from smarttv.droideka.proxy.swagger.categories import CategoriesSpec
from smarttv.droideka.proxy.views.base import PlatformAPIView
from smarttv.droideka.proxy.models import Category2
from smarttv.droideka.proxy.blackbox import SubscriptionType
from smarttv.droideka.proxy.common import geobase
from smarttv.droideka.proxy import api

logger = logging.getLogger(__name__)


class CategoryPatcher:
    """
    Temporary solution for OTT as data source experiment
    OTT provides different categories for users with different subscriptions
    We should handle these categories manually
    For now, we doesn't have targeting by subscription, so this is small hack for experiment that will patch fake
    ott category to required one

    It means, when category `stub_kp_main`, has been obtained - it should be replaced by one of the:
    `ya_plus`, `kp_basic`, `ya_plus_super`, `ya_premium` depending one user's subscription
    If user has no subscription - use category for minimal subscription: `ya_plus`
    """
    KP_CATEGORY_ID_TO_PATCH = 'stub_kp_main'

    # (YA_PLUS, YA_PLUS_3M, YA_PLUS_KP) -> 'ya_plus'
    # KP_BASIC -> kp_basic
    # YA_PLUS_SUPER -> ya_plus_super (more.tv)
    # YA_PREMIUM -> ya_premium (amediateka)
    subscription_to_category_id_mapping = defaultdict(lambda: 'ya_plus', {
        SubscriptionType.YA_PLUS: 'ya_plus',
        SubscriptionType.YA_PLUS_3M: 'ya_plus',
        SubscriptionType.YA_PLUS_KP: 'ya_plus',
        SubscriptionType.KP_BASIC: 'kp_basic',
        SubscriptionType.YA_PLUS_SUPER: 'ya_plus_super',
        SubscriptionType.YA_PREMIUM: 'ya_premium',
    })

    def __init__(self, subscription_type: SubscriptionType, categories: Iterable[Category2]):
        self.subscription_type = subscription_type
        self.categories = categories

    def get_category_to_patch(self) -> Optional[Category2]:
        """
        We need to patch only KP category with ID: stub_kp_main
        :return:
        """
        for category in self.categories:
            if category.category_id == self.KP_CATEGORY_ID_TO_PATCH:
                return category
        return None

    def patch_category_id(self, category: Category2):
        category.category_id = self.subscription_to_category_id_mapping[self.subscription_type]

    def patch(self):
        category_to_patch = self.get_category_to_patch()
        if not category_to_patch:
            logger.debug('Nothing to patch')
            return
        self.patch_category_id(category_to_patch)
        logger.debug('Changed main KP category id to %s', category_to_patch.category_id)


class CategoriesView(PlatformAPIView):
    RUSSIA_ISO_NAME = 'RU'
    TV_CHANNELS_PERSISTENT_CLIENT_ID = 'tv_channels_vertical'

    def get_user_country_region_id(self, ip: str) -> int:
        return geobase.get_country_id(geobase.get_region_id_by_ip(ip))

    def is_from_russia(self, country_region_id: int) -> bool:
        client_country_iso_name = geobase.get_region_by_id(country_region_id).get('iso_name')
        return self.RUSSIA_ISO_NAME == client_country_iso_name

    def get_channels(self, request, country_region_id: int):
        headers = self.vh_headers(request)
        available_channels = api.vh.client.channels_regions(
            initial_request=self.request, headers=headers, region_id=country_region_id)
        return available_channels

    def should_show_tv_channels_category(self, channels_count: int) -> bool:
        return channels_count != 0

    def filter_tv_channels_if_necessary(self, request, categories: list) -> list:
        user_country_region_id = self.get_user_country_region_id(self.user_ip)
        if self.is_from_russia(user_country_region_id):
            logger.info('Request is from Russia, show TV channels')
            return categories
        channels_count = len(self.get_channels(request, user_country_region_id))
        logger.info('Found %s channels for region %s', channels_count, user_country_region_id)
        if self.should_show_tv_channels_category(channels_count):
            return categories
        return [category for category in categories if category.persistent_client_category_id != self.TV_CHANNELS_PERSISTENT_CLIENT_ID]

    @swagger_schema(CategoriesSpec)
    def get(self, request):
        try:
            categories = categories_provider.get_categories(request.platform_info, request_info=request.request_info)
            categories = self.filter_tv_channels_if_necessary(request, categories)
            CategoryPatcher(self.get_user_info(request).subscription, categories).patch()
            serializer = CategorySerializer(categories, many=True, context={'platform_info': request.platform_info})
            return Response(serializer.data)
        except StorageNotAvailable:
            return Response(status=status.HTTP_503_SERVICE_UNAVAILABLE)
