import logging

from typing import Optional
from cache_memoize import cache_memoize
from django.conf import settings
from rest_framework.response import Response

from smarttv.droideka.proxy.api import vh
from smarttv.droideka.proxy.api.usaas import EmptyIcookieException
from smarttv.droideka.proxy.common import geobase
from smarttv.droideka.proxy.serializers.fast_response import EpisodeInfo, EpisodeInfoV8, SmotreshkaChannel
from smarttv.droideka.proxy.serializers.serializers import ChannelsValidator
from smarttv.droideka.proxy.transform import Channels, ChannelInfo
from smarttv.droideka.proxy.views.base import PlatformAPIView
from smarttv.droideka.proxy.models import RecommendedChannel, SharedPreferences
from smarttv.droideka.proxy.constants.reserved_shared_pref_keys import KEY_FILTER_CHANNELS, \
    KEY_FILTER_SUBSCRIPTION_CHANNELS, PREF_SMOTRESHKA_API_ENABLED
from smarttv.droideka.utils import PlatformInfo
from smarttv.droideka.proxy.constants.home_app_versions import VER_1_4
from yaphone.utils.parsed_version import ParsedVersion
from smarttv.droideka.proxy import cache, smotreshka
from smarttv.droideka.unistat.metrics import CacheType, RedisCachePlace
from smarttv.droideka.proxy.api.base import BaseApi
from smarttv.utils.django import get_http_header
from smarttv.utils.headers import QUASAR_DEVICE_ID, TIMEZONE_OFFSET

logger = logging.getLogger(__name__)
episode_info_serializer = EpisodeInfo()
episode_info_serializer_v8 = EpisodeInfoV8()
smotreshka_channel_serializer = SmotreshkaChannel()

RECOMMENDED_CATEGORY_TITLE = 'Рекомендованные'
RECOMMENDED_EXPERIMENT_KEY = 'recommended_channels'
SMOTRESHKA_EXPERIMENT_KEY = 'smotreshka_channels'


class ChannelsView(PlatformAPIView):
    validator_class = ChannelsValidator

    def _log_identification_info(self, vh_channels_response):
        if not settings.LOG_CONTENT_ID:
            return
        def get_identity_info(channel):
            if not channel:
                return None
            identity_info = {'content_id': channel.get('content_id')}
            auto_fields = channel.get('auto_fields')
            if auto_fields:
                identity_info['channel_number'] = auto_fields.get('channel_smarttv_number')
            return identity_info
        logger.info({
            'user_data': vh_channels_response.get('user_data'),
            'apphost-reqid': vh_channels_response.get('apphost-reqid'),
            'identity_info': [get_identity_info(channel) for channel in vh_channels_response.get('set', [])]
        })

    def get_user_region(self, request):
        region_id = None
        country = None

        user_ip = self.get_user_ip(request)
        if user_ip:
            region_id = geobase.get_region_by_ip(user_ip)['id']

        if region_id:
            country = geobase.get_region_by_id(geobase.get_country_id(region_id)).get('iso_name')

        return region_id, country

    def get_cache_key(self, request, region_id, show_hidden, project_alias, smotreshka_enabled, timezone):
        logger.debug(f'Region id: {region_id}')
        return [region_id, show_hidden, project_alias, repr(request.platform_info.app_version),
                self.is_recommended_enabled(request), smotreshka_enabled, timezone]

    def get_channels(self, headers, region_id, project_alias):
        vh_response = vh.channel_client.channels_regions(
            initial_request=self.request, headers=headers, region_id=region_id,
            project_alias=project_alias)
        self._log_identification_info(vh_response)
        return vh_response['set']

    def log_channels_without_number(self, channels: list):
        log_tuples = tuple((ch.get('title'), ch.get('channel_id')) for ch in channels)
        logger.info('Filtered %s channels without channel_smarttv_number (title, id): %r',
                     len(channels), log_tuples)

    def filter_channels(self, channels, show_hidden, filter_subscription_channels, filter_channel_ids):
        result = []
        without_smarttv_number = []
        banned = hidden = paid = missed = normal = 0  # stats
        for channel in channels:
            if not channel or channel.get('banned'):
                banned += 1
                continue
            if not show_hidden and 'status' in channel and 'hidden' in channel['status']:
                hidden += 1
                continue
            if filter_subscription_channels and self.is_paid_channel(channel.get('ottParams')):
                paid += 1
                continue
            if filter_channel_ids and channel['content_id'] in filter_channel_ids:
                continue
            if not channel.get('auto_fields', {}).get('channel_smarttv_number'):
                without_smarttv_number.append(channel)
                missed += 1
                continue

            normal += 1
            result.append(channel)

        logger.info('Channel stats: %d banned, %d hidden, %d paid, %d without number, %d ok',
                    banned, hidden, paid, missed, normal)

        if without_smarttv_number:
            self.log_channels_without_number(without_smarttv_number)

        return result

    def serialize_channels(self, channels):
        return [episode_info_serializer.serialize(channel) for channel in channels]

    def get_filter_channel_ids(self) -> Optional[list]:
        filter_channels = SharedPreferences.get_string(KEY_FILTER_CHANNELS)
        if not filter_channels:
            return None
        return filter_channels.split(';')

    def should_filter_subscription_channels(self, platform_info: PlatformInfo) -> bool:
        app_version_less_1_4 = not platform_info or not platform_info.app_version \
            or ParsedVersion(platform_info.app_version) < VER_1_4

        return app_version_less_1_4 or SharedPreferences.get_bool(KEY_FILTER_SUBSCRIPTION_CHANNELS)

    def is_paid_channel(self, ott_params):
        free, paid = False, True
        if not ott_params or not ott_params.get('licenses'):
            return free

        for item in ott_params['licenses']:
            if item['monetizationModel'] == 'AVOD':
                return free

        return paid

    def get_recommended_category(self) -> Optional[dict]:
        """
        Рекомендуемые к просмотру каналы (настраиваются в админке)
        """
        query = RecommendedChannel.objects.filter(visible=True)\
            .order_by('rank').values_list('channel_id', flat=True)
        channels = [id for id in query]

        if channels:
            return {'name': RECOMMENDED_CATEGORY_TITLE, 'channel_ids': channels}

        return None

    def is_recommended_enabled(self, request) -> bool:
        try:
            enabled = bool(request.request_info.experiments.get_value(RECOMMENDED_EXPERIMENT_KEY))
        except BaseApi.RequestError:
            logger.debug('Error getting experiments from USaaS')
            enabled = False
        except EmptyIcookieException:
            logger.debug('icookie is empty')
            enabled = False

        logger.debug('Recommended channels are enabled: %s', enabled)
        return enabled

    def get_smotreshka_channels(self, device_id, timezone_offset):
        if not device_id:
            logger.info('No device id supplied, skipping')
            return []

        channels = []
        try:
            s_medias = smotreshka.get_medias(device_id, timezone_offset)
        except Exception as err:
            logger.exception(f'Can not load medias because of error: {err}')
            return []
        media_index = {m['channelId']: m for m in s_medias if not m['isLocked']}

        for ch in smotreshka.get_saved_channels_data():
            media = media_index.get(ch['smotreshka_channel_id'])
            if media:
                ch['content_id'] = media['id']
                channels.append(ch)

        return channels

    @cache_memoize(
        timeout=settings.DEFAULT_RESPONSE_CACHE_TIME,
        args_rewrite=get_cache_key,
        **cache.get_cache_memoize_callables(RedisCachePlace.CHANNELS_RESPONSE.value, CacheType.REDIS)
    )
    def get_response(self, request, region_id, show_hidden, project_alias, smotreshka_enabled, timezone_offset):
        headers = self.vh_headers(request)
        channels = self.get_channels(headers, region_id, project_alias)
        channels = self.filter_channels(channels,
                                        show_hidden,
                                        self.should_filter_subscription_channels(request.platform_info),
                                        self.get_filter_channel_ids())

        categories = []
        if self.is_recommended_enabled(request):
            recommended = self.get_recommended_category()
            if recommended:
                categories.append(recommended)

        if smotreshka_enabled:
            # смотрешкины каналы, обогащенные айдишниками медиа из ответа апи
            logger.info('Smotreshka channels are enabled for this client')
            device_id = get_http_header(request, QUASAR_DEVICE_ID)
            sm_channels = self.get_smotreshka_channels(device_id, timezone_offset)
            if sm_channels:
                logger.info('Adding %s smotreshka channels in result', len(sm_channels))
                channels.extend(sm_channels)

        channel_info_list = []
        for item in channels:
            channel_info_list.append(
                ChannelInfo(
                    id=item['channel_id'],
                    categories=item.get('channel_category', []),
                    number=item['auto_fields'].get('channel_smarttv_number')
                )
            )

        categories.extend(Channels.get_categorized(channel_info_list))

        return {
            'channels': self.serialize_channels(channels),
            'categories': categories,
        }

    def get(self, request):
        params = self.get_validated_data(request)
        region_id, country = self.get_user_region(request)
        response = self.get_response(
            request,
            region_id,
            params['show_hidden'],
            params.get('project_alias'),
            False,
            None
        )

        return Response(response)


class ChannelsViewV8(ChannelsView):
    """
    Список каналов, версия 8

    Начиная с этой версии в списке каналов могут появиться каналы стороннего
    провайдера - Смотрешки. Поэтому у каждого элемента добавлено поле channel_provider.
    """
    def serialize_channels(self, channels):
        serialized = []
        for channel in channels:
            if channel.get('channel_provider') == 'smotreshka':
                serialized.append(smotreshka_channel_serializer.serialize(channel))
            else:
                serialized.append(episode_info_serializer_v8.serialize(channel))
        return serialized

    def is_smotreshka_enabled(self, country):
        if not SharedPreferences.get_int(PREF_SMOTRESHKA_API_ENABLED):
            logger.info('Smotreshka disabled by shared preferences')
            return False

        if country != 'RU':
            logger.info(f'Smotreshka disabled in country "{country}"')
            return False

        return True

    def get(self, request):
        params = self.get_validated_data(request)
        timezone_offset = get_http_header(request, TIMEZONE_OFFSET)

        region_id, country = self.get_user_region(request)
        smotreshka_enabled = self.is_smotreshka_enabled(country)

        response = self.get_response(
            request,
            region_id,
            params['show_hidden'],
            params.get('project_alias'),
            smotreshka_enabled,
            timezone_offset
        )

        return Response(response)
