import logging

from rest_framework.response import Response
from rest_framework.exceptions import ParseError
from rest_framework import status as http_status

from smarttv.utils import headers

from smarttv.droideka.proxy.views.base import PlatformAPIView
from smarttv.droideka.utils.strings import to_minified_json
from smarttv.droideka.proxy.api.usaas import EmptyIcookieException
from smarttv.droideka.proxy.api.base import BaseApi

logger = logging.getLogger(__name__)


class ExperimentsView(PlatformAPIView):
    """
    API client for USaaS(user split as a service)

    Documentation for API:
    https://wiki.yandex-team.ru/jandekspoisk/kachestvopoiska/abt/uaas/
    """
    required_headers = (headers.DEVICE_ID_HEADER, headers.USER_AGENT_HEADER,
                        headers.FINGERPRINT_HEADER, headers.ETHERNET_MAC_HEADER)
    need_kp_profile_creation = False

    def get(self, request):
        try:
            all_experiments = request.request_info.experiments.all
        except BaseApi.RequestError:
            logger.info('Error getting client experiments from usaas')
            return Response(status=http_status.HTTP_502_BAD_GATEWAY)
        except EmptyIcookieException:
            raise ParseError('ICookie must not be empty')
        experiments = [
            experiment.serialize()
            for experiment in all_experiments
            if experiment.send_to_client()
        ]

        log_data = request.request_info.experiments.log_data
        log_data['device_icookie'] = request.request_info.icookie
        log_data['experiments_icookie'] = request.request_info.experiments_icookie

        return Response({
            'experiments': experiments,
            'log_data': to_minified_json(log_data),
            'exp_boxes': log_data['exp_boxes']
        })
