import logging

from cache_memoize import cache_memoize
from django.conf import settings
from django.shortcuts import redirect
from rest_framework.response import Response

from smarttv.droideka.proxy import api, cache
from smarttv.droideka.proxy import models
from smarttv.droideka.proxy.management.commands.save_current_programs import KEY_PROGRAMS_ARCHIVE_S3_KEY
from smarttv.droideka.proxy.serializers.fast_response import ProgramsSerializer
from smarttv.droideka.proxy.serializers.serializers import ProgramsValidator
from smarttv.droideka.proxy.s3mds import s3_client
from smarttv.droideka.proxy.swagger.base import swagger_schema
from smarttv.droideka.proxy.swagger.programs import ProgramsSpec
from smarttv.droideka.proxy.views.base import PlatformAPIView, BackendAPIException
from smarttv.droideka.unistat.metrics import CacheType, LocalCachePlace

logger = logging.getLogger(__name__)


def get_episodes_key(view, data, headers):
    return data.values()


class ProgramsView(PlatformAPIView):
    validator_class = ProgramsValidator
    response_serializer = ProgramsSerializer()
    response_validator_error = BackendAPIException

    @swagger_schema(ProgramsSpec)
    def get(self, request):
        data = self.get_validated_data(request)
        headers = self.vh_headers(request)
        episodes = self.load_episodes_cached(data, headers)
        return Response(episodes)

    def load_episodes(self, data, headers):
        response = api.vh.channel_client.episodes(self.request, headers, **data)
        api.vh.fake_stream_url(response)
        if response is None:
            raise BackendAPIException()
        episodes = response.get('set')
        if not episodes:
            if episodes is None:
                raise BackendAPIException()
            logger.warning('Empty programs response for %s', data)
            return []
        serialized_episodes = self.response_serializer.serialize_many(episodes)
        if not serialized_episodes:
            raise BackendAPIException()
        return serialized_episodes

    @cache_memoize(timeout=settings.DEFAULT_RESPONSE_CACHE_TIME,
                   args_rewrite=get_episodes_key,
                   cache_alias='local',
                   **cache.get_cache_memoize_callables(LocalCachePlace.V4_EPISODES.value, CacheType.LOCAL))
    def load_episodes_cached(self, data, headers):
        return self.load_episodes(data, headers)


class AllPrograms(PlatformAPIView):
    def get(self, request):
        key = models.SharedPreferences.get_string(KEY_PROGRAMS_ARCHIVE_S3_KEY)
        if not key:
            logger.error('No programs archive found')
            return Response(status=404)
        url = s3_client.get_url(key)
        return redirect(url)
