import logging
import enum
import random
from typing import Optional
from collections import defaultdict, namedtuple
import os
import datetime
import html

from django.conf import settings
from rest_framework.exceptions import NotFound
from rest_framework.response import Response

from smarttv.utils import headers
from yaphone.utils.django import make_http_header
from yaphone.utils.parsed_version import ParsedVersion

from alice.protos.data.search_result.search_result_pb2 import TSearchResultData
from google.protobuf import json_format
from smarttv.droideka.proxy import api, exceptions
from smarttv.droideka.proxy.api.access_config import get_access_config
from smarttv.droideka.proxy.avatars_image_generator import fill_images_from_id, remove_mds_avatar_size
from smarttv.droideka.proxy.player_detection import SearchRequestPlayerIdFiller, PlayerDetector
from smarttv.droideka.proxy.serializers import serializers, protobuf
from smarttv.droideka.proxy.swagger import base as swagger_base, search as swagger_search
from smarttv.droideka.proxy.views.base import PlatformAPIView
from smarttv.droideka.utils import PlatformInfo
from smarttv.droideka.utils.known_platforms import YANDEX_MODULE
from smarttv.droideka.utils.url import build_view_url
from smarttv.droideka.proxy.common import fix_schema, fix_avatars_mds_suffix
from smarttv.droideka.proxy.vh.constants import TRAILER_CONTENT_TYPES
from smarttv.droideka.proxy.result_builder import CarouselVideohubResultBuilder
from smarttv.droideka.proxy.request.carousels import VhCarouselInfo

logger = logging.getLogger(__name__)

FIELD_PARENT_COLLECTION = 'parent_collection'
FIELD_OBJECT = 'object'
FIELD_IMAGE = 'image'
FIELD_IMAGE_ORIGINAL = 'original'
FIELD_IMAGE_OTHER = 'other'
FIELD_IMAGE_MDS_ID = 'mds_avatar_id'
FIELD_ENTITY_DATA = 'entity_data'
FIELD_RELATED_OBJECT = 'related_object'
FIELD_RELATED_OBJECT_TYPE = 'type'
FIELD_BASE_INFO_TYPE = 'type'
FIELD_ENTREF = 'entref'
FIELD_HORIZONTAL_POSTER = 'horizontal_poster'
FIELD_BASE_INFO = 'base_info'
FIELD_AGE_LIMIT = 'age_limit'
FIELD_RICH_INFO = 'rich_info'
FIELD_VH_META = 'vh_meta'
FIELD_CONTENT_GROUPS = 'content_groups'
FIELD_LICENSES = 'licenses'
FIELD_LEGAL = 'legal'
FIELD_VH_LICENSES = 'vh_licenses'
FIELD_CLIPS = 'clips'
FIELD_TITLE = 'title'
FIELD_DURATION = 'duration'
FIELD_THMB_HREF = 'thmb_href'
FIELD_CONTENT_TYPE = 'content_type'

APP_VERSION_1_2_DEBUG = ParsedVersion('1.2.2147483647')

# by default use key 'is_avaliable' for checking availability of content, but for TVOD / EST - use specific fields
LEGAL_AVAILABILITY_KEY = defaultdict(lambda: 'is_avaliable', {'tvod': 'TVOD', 'est': 'EST'})
PosterChooseConfig = namedtuple('PosterChooseConfig',
                                ('monetization_model_key', 'requires_availability', 'availability_key'))
LEGAL_IMAGE_CHOOSE_ORDER = (
    PosterChooseConfig('tvod', True, LEGAL_AVAILABILITY_KEY['tvod']),
    PosterChooseConfig('est', True, LEGAL_AVAILABILITY_KEY['est']),
    PosterChooseConfig('avod', False, LEGAL_AVAILABILITY_KEY['avod']),
    PosterChooseConfig('svod', False, LEGAL_AVAILABILITY_KEY['svod']),
)


class RelatedObjectType(enum.Enum):
    PROJECTS = 'proj'  # projects of certain actor
    COLLECTIONS = 'collections'  # collections similar for original one
    ASSOC = 'assoc'  # associated objects, with original one(for film, it may be actors, countries, directors, etc.)


class BaseInfoType(enum.Enum):
    PERSON = 'Hum'


def get_base_info(response: dict) -> Optional[dict]:
    try:
        return response[FIELD_ENTITY_DATA][FIELD_BASE_INFO]
    except (KeyError, TypeError):
        return None


def get_related_object(entity_data: Optional[dict]):
    if entity_data:
        return entity_data.get(FIELD_RELATED_OBJECT)
    return None


def get_related_object_by_type(related_objects: Optional[list], related_object_type: RelatedObjectType):
    if not related_objects:
        return None
    for obj in related_objects:
        if obj.get(FIELD_RELATED_OBJECT_TYPE) == related_object_type.value:
            return obj


def fill_parent_collection_more_url(parent_collection, request, more_url_view_name):
    next_start_index = parent_collection.get('next_start_index', -1)
    if next_start_index < 0:
        return
    params = {
        'ento': f"entlist={parent_collection['id']};entlistskip={next_start_index}"
    }

    parent_collection['more_url'] = build_view_url(params, more_url_view_name, request)


def get_image_from_monetization_model(
        monetization_model: Optional[dict],
        requires_availability: bool,
        availability_key: str) -> Optional[str]:
    if (not monetization_model or (requires_availability and not monetization_model.get(availability_key, False))
            or not monetization_model.get('import_poster')):
        return None
    return fix_schema(monetization_model['import_poster'])


def get_collection_image_from_legal(legal: Optional[dict]) -> Optional[str]:
    if not legal or not legal.get('vh_licenses'):
        return None
    vh_licenses = legal['vh_licenses']
    for config in LEGAL_IMAGE_CHOOSE_ORDER:
        image = get_image_from_monetization_model(vh_licenses.get(config.monetization_model_key),
                                                  config.requires_availability,
                                                  config.availability_key)
        if image:
            return fix_avatars_mds_suffix(image)
    licenses = legal['vh_licenses']
    if not licenses.get('import_poster'):
        return None
    return fix_avatars_mds_suffix(fix_schema(licenses['import_poster']))


def patch_collection_image_if_necessary(image: dict, platform_info: PlatformInfo, legal: Optional[dict]):
    legal_poster = get_collection_image_from_legal(legal)
    if legal_poster:
        image[FIELD_IMAGE_ORIGINAL] = legal_poster
        return
    original_image = image.get(FIELD_IMAGE_ORIGINAL)
    mds_avatar_id = image.get(FIELD_IMAGE_MDS_ID)
    fill_images_from_id(
        mds_avatar_id, 'get-entity_search', settings.COLLECTION_IMAGES_REQUIRED_MDS_HOST, FIELD_IMAGE_OTHER, image,
        platform_info
    )
    if original_image and settings.COLLECTION_IMAGES_REQUIRED_MDS_HOST in original_image:
        # collection image has correct url, nothing to do here
        return
    elif mds_avatar_id:
        image[FIELD_IMAGE_ORIGINAL] = \
            f'https://{settings.COLLECTION_IMAGES_REQUIRED_MDS_HOST}/get-entity_search/{mds_avatar_id}/orig'
    else:
        logger.warning("Can't patch url: %s", original_image)


def fix_parent_collections_image_urls(parent_collection_objects, platform_info: PlatformInfo):
    for collection_object in parent_collection_objects:
        patch_collection_image_if_necessary(collection_object[FIELD_IMAGE], platform_info,
                                            collection_object.get('legal'))
        remove_legal_for_trailer(collection_object)


def filter_parent_collection_for_module(parent_collection: dict, platform_info: PlatformInfo):
    if platform_info not in YANDEX_MODULE:
        return
    documents = parent_collection[FIELD_OBJECT]
    parent_collection[FIELD_OBJECT] = [document for document in documents if 'legal' in document]


def get_parent_collection(response, is_search: bool):
    if is_search:
        try:
            return response[FIELD_ENTITY_DATA][FIELD_PARENT_COLLECTION]
        except (KeyError, TypeError):
            return None
    else:
        return response[FIELD_PARENT_COLLECTION]


def process_parent_collection(response, request, more_url_view_name, is_search, platform_info):
    parent_collection = get_parent_collection(response, is_search)

    if not parent_collection:
        return

    if 'type' in parent_collection and parent_collection['type'] == 'tracks':
        if is_search:
            entity_data = response[FIELD_ENTITY_DATA]
        else:
            entity_data = response
        del entity_data[FIELD_PARENT_COLLECTION]
        return

    fill_parent_collection_more_url(parent_collection, request, more_url_view_name)

    filter_parent_collection_for_module(parent_collection, platform_info)
    if FIELD_OBJECT in parent_collection:
        parent_collection_objects = parent_collection[FIELD_OBJECT]
        fix_parent_collections_image_urls(parent_collection_objects, platform_info)
        SearchRequestPlayerIdFiller.fill_player_id_for_parent_collection(parent_collection_objects)


def has_wsubtype(document) -> bool:
    return bool(document) and bool(document.get('wsubtype'))


def filter_objects_without_wsubtype(objects_container: Optional[dict]):
    if not objects_container or FIELD_OBJECT not in objects_container:
        return
    objects = objects_container[FIELD_OBJECT]
    objects_container[FIELD_OBJECT] = [obj for obj in objects if has_wsubtype(obj)]


def remove_legal_for_trailer(assoc: dict):
    try:
        content_type = assoc[FIELD_LEGAL][FIELD_VH_LICENSES][FIELD_CONTENT_TYPE]
    except (TypeError, KeyError):
        return
    if content_type in TRAILER_CONTENT_TYPES:
        del assoc[FIELD_LEGAL]


def get_videosearch_client(request):
    return get_access_config(request).videopoisk_client


class BaseSearchView(PlatformAPIView):
    def prepare_headers(self, request):
        proxy_headers = {
            headers.FORWARDED_HEADER: self.user_ip,
            headers.INTERNAL_REQUEST: '1',
            headers.BALANCING_HINT_HEADER: os.environ.get('DEPLOY_NODE_DC')
        }
        auth_header = make_http_header(headers.AUTHORIZATION_HEADER)
        if auth_header in request.META:
            proxy_headers[headers.AUTHORIZATION_HEADER] = request.META[auth_header]
        return proxy_headers


class ParentCollectionView(PlatformAPIView):
    NAME = 'parent_collection'

    validator_class = serializers.ParentCollectionValidator

    def get_response(self, request) -> dict:
        data = self.get_validated_data(request)

        response = api.es.client.get_object_response(request=request, ento=data['ento'])
        if not response or not response.get(FIELD_PARENT_COLLECTION) or not response[FIELD_PARENT_COLLECTION].get(
                FIELD_OBJECT):
            raise NotFound('Requested parent collection not found')

        filter_objects_without_wsubtype(response[FIELD_PARENT_COLLECTION])
        process_parent_collection(response, request, self.NAME, False, request.platform_info)
        return response

    def get(self, request):
        response = self.get_response(request)
        return Response(status=200, data=response[FIELD_PARENT_COLLECTION])


class ParentCollectionViewV5(ParentCollectionView):
    NAME = 'parent_collection5'


class ParentCollectionViewV6(ParentCollectionView):
    NAME = 'parent_collection6'


class ParentCollectionViewV7(ParentCollectionView):
    NAME = 'parent_collection7'


class SearchView(BaseSearchView):
    NAME = 'search'
    PARENT_COLLECTION_MORE_URL_VIEW = ParentCollectionView

    KEY_KINOPOISK_ID = 'kinopoisk'
    CLIPS_MTIME_FORMAT = '%Y-%m-%dT%H:%M:%S+0000'

    validator_class = serializers.SearchValidator

    @staticmethod
    def filter_entity_data_by_age_restriction(response, restriction_age):
        entity_data = response.get(FIELD_ENTITY_DATA)
        if entity_data:
            base_info = get_base_info(response)
            if not base_info or int(base_info.get(FIELD_AGE_LIMIT, 0)) > restriction_age:
                del response[FIELD_ENTITY_DATA]
                return
            if FIELD_RELATED_OBJECT in entity_data:
                del entity_data[FIELD_RELATED_OBJECT]
            if FIELD_PARENT_COLLECTION in entity_data:
                del entity_data[FIELD_PARENT_COLLECTION]

    @classmethod
    def filter_documents_by_wsubtype(cls, response: dict):
        filter_objects_without_wsubtype(get_parent_collection(response, True))
        filter_objects_without_wsubtype(cls.get_association_object(response))

    @classmethod
    def get_trailer_thumbnail(cls, rich_info: Optional[dict]) -> Optional[str]:
        try:
            return remove_mds_avatar_size(rich_info['vh_meta']['content_groups'][0]['import_thumbnail'])
        except (TypeError, KeyError, IndexError):
            logger.info('Could not obtain thumbnail for trailer')
            return None

    @classmethod
    def _get_trailer_mtime(cls, trailer_info: Optional[dict], content_group: Optional[dict]) -> str:
        try:
            return datetime.datetime.strptime(trailer_info['date'], '%Y-%m-%d').strftime(cls.CLIPS_MTIME_FORMAT)
        except (TypeError, KeyError):
            try:
                return datetime.datetime(content_group['year'], 1, 1).strftime(cls.CLIPS_MTIME_FORMAT)
            except (TypeError, KeyError):
                return datetime.datetime(
                    datetime.datetime.now().year,
                    datetime.datetime.now().month,
                    datetime.datetime.now().day).strftime(cls.CLIPS_MTIME_FORMAT)

    @classmethod
    def _get_trailer_title(cls, content_group: Optional[dict], base_info: dict) -> str:
        if FIELD_TITLE in content_group:
            return content_group[FIELD_TITLE]
        else:
            return base_info['name']

    @classmethod
    def get_trailer_duration(cls, trailer_info: Optional[dict], content_group: Optional[dict]) -> int:
        if trailer_info and FIELD_DURATION in trailer_info:
            return trailer_info[FIELD_DURATION]
        if content_group and FIELD_DURATION in content_group:
            return content_group[FIELD_DURATION]
        return 0

    @classmethod
    def handle_trailer(cls, response):
        try:
            _license = response[FIELD_ENTITY_DATA][FIELD_BASE_INFO][FIELD_LEGAL][FIELD_VH_LICENSES]
        except (TypeError, KeyError, IndexError):
            return
        if _license.get(FIELD_CONTENT_TYPE) not in TRAILER_CONTENT_TYPES:
            return
        ids = response[FIELD_ENTITY_DATA][FIELD_BASE_INFO].get('ids')
        if not ids or cls.KEY_KINOPOISK_ID not in ids:
            # it is not possible for now, to handle non-kp trailers,
            # so, just discard it
            logger.info('Available base_info ids: %s', ids)
            del response[FIELD_ENTITY_DATA][FIELD_BASE_INFO]
            return
        base_info = response[FIELD_ENTITY_DATA][FIELD_BASE_INFO]
        try:
            vh_uuid = _license.get('uuid')
        except (KeyError, TypeError):
            logger.info('No vh_uuid for object: %s', ids)
            return
        try:
            content_group = response[FIELD_ENTITY_DATA][FIELD_RICH_INFO]['vh_meta']['content_groups'][0]
        except (KeyError, TypeError):
            content_group = None
        try:
            trailer_info = response[FIELD_ENTITY_DATA]['view']['Gallery']['Trailer']
        except (KeyError, TypeError):
            trailer_info = None
        result = {'is_avod': "1", 'VisibleHost': 'kinopoisk.ru', PlayerDetector.KEY_PLAYER_ID: 'ott',
                  'title': cls._get_trailer_title(content_group, base_info), 'vh_uuid': vh_uuid,
                  FIELD_DURATION: cls.get_trailer_duration(trailer_info, content_group)}

        trailer_thumbnail = cls.get_trailer_thumbnail(response[FIELD_ENTITY_DATA].get(FIELD_RICH_INFO))
        if trailer_thumbnail:
            result[FIELD_THMB_HREF] = trailer_thumbnail
        url = 'https://frontend.vh.yandex.ru/player/' + vh_uuid
        result['url'] = url
        result['content_url'] = url
        result['players'] = {'autoplay': {"html": f'<iframe src="{html.escape(url)}"></iframe>'}}
        result['mtime'] = cls._get_trailer_mtime(trailer_info, content_group)

        response[FIELD_CLIPS].insert(0, result)
        del response[FIELD_ENTITY_DATA][FIELD_BASE_INFO]

    def get_response(self, request) -> dict:
        headers = self.prepare_headers(request)
        validated_data = self.get_validated_data(request)

        entref = validated_data.get(FIELD_ENTREF)
        params = self.get_restriction_params(validated_data)
        params['text'] = validated_data['text']
        if entref:
            params[FIELD_ENTREF] = entref
        platform_info = request.platform_info
        params['need_people'] = self.need_people(platform_info)
        params['client'] = get_videosearch_client(request)

        response = api.vs.client.search(params=params, headers=headers)

        if not entref:
            self.handle_trailer(response)
            # Client sends request with entref to get detailed data about movie from search.
            # Do not delete base info in this case (https://st.yandex-team.ru/SMARTTVBACKEND-140)
            self.fix_base_info(response)
        if validated_data.get('restriction_mode') == 'kids':
            # We're in kid mode, so we just cut off assoc and and collections (and actors)
            # as they're not properly filtered (c) https://st.yandex-team.ru/SMARTTVBACKEND-344
            # Also we need to delete entity_data if it is above age restriction
            if (restriction_age := validated_data.get('restriction_age')) is not None:
                self.filter_entity_data_by_age_restriction(response, int(restriction_age))
        self.remove_duplicate_clip(response)
        SearchRequestPlayerIdFiller.fill_player_id_for_base_info(response.get(FIELD_ENTITY_DATA))
        association_object = self.get_association_object(response)
        if association_object:
            self.filter_assoc_for_module(response, platform_info)
            SearchRequestPlayerIdFiller.fill_player_id_for_assocs(association_object.get('object'))
        self.fix_collections_image_urls(response, platform_info)
        process_parent_collection(response, request, self.PARENT_COLLECTION_MORE_URL_VIEW.NAME, True, platform_info)
        self.fix_assoc_image_urls(response, platform_info)
        SearchRequestPlayerIdFiller.fill_player_id_for_clips(response.get(FIELD_CLIPS))
        self.filter_documents_by_wsubtype(response)
        return response

    @swagger_base.swagger_schema(swagger_search.SearchSpec)
    def get(self, request):
        return Response(status=200, data=self.get_response(request))

    @classmethod
    def need_people(cls, platform_info: Optional[PlatformInfo]):
        return platform_info and platform_info.app_version and \
            ParsedVersion(platform_info.app_version) >= APP_VERSION_1_2_DEBUG

    @staticmethod
    def get_restriction_params(data):
        mode = data.get('restriction_mode')
        if not mode:
            return {'relev': 'pf_off'}
        if mode == 'family':
            return {'relev': 'pf=strict'}
        if mode == 'moderate':
            return {'family': 'moderate'}
        if mode == 'kids':
            return {'relev': f'age_limit={data["restriction_age"]}'}
        msg = 'Validation works improperly, unknown restriction_mode "%s" passed validation'
        logger.error(msg, mode)
        raise exceptions.InternalServerError(msg % mode)

    @staticmethod
    def remove_entity_data_for_trailer(response):
        try:
            content_type = response[FIELD_ENTITY_DATA][FIELD_BASE_INFO][FIELD_LEGAL][FIELD_VH_LICENSES][FIELD_CONTENT_TYPE]
        except (TypeError, KeyError, IndexError):
            return
        if content_type in ('TRAILER', 'KP_TRAILER'):
            del response[FIELD_ENTITY_DATA][FIELD_BASE_INFO]

    @staticmethod
    def remove_duplicate_clip(response):
        entity_data = response.get(FIELD_ENTITY_DATA)
        if not entity_data:
            # this check is necessary, because sometimes, response has explicitly set {'entity_data': null, ...}
            return
        base_info = entity_data.get(FIELD_BASE_INFO)
        if not base_info:
            # this check is necessary, because sometimes, response has explicitly set {'base_info': '', ...}
            return
        onto_id = base_info.get('id')
        clips = response.get(FIELD_CLIPS)
        if not onto_id or not clips:
            return
        new_clips = [clip for clip in clips if clip.get('onto_id') != onto_id]
        response[FIELD_CLIPS] = new_clips

    @classmethod
    def fix_base_info(cls, response):
        """
        Remove base_info if there is no legal information for movie
        See https://st.yandex-team.ru/SMARTTVBACKEND-111 for details

        But do not remove base_info for persons%
        https://st.yandex-team.ru/SMARTTVBACKEND-517#5f7eefbbefcc290e649392ef
        because persons never has 'legal' field
        """
        entity_data = response.get(FIELD_ENTITY_DATA)
        base_info = get_base_info(response)
        if base_info:
            is_not_person = base_info.get(FIELD_BASE_INFO_TYPE) != BaseInfoType.PERSON.value
            has_no_legal = 'legal' not in base_info
            if is_not_person and has_no_legal:
                del entity_data[FIELD_BASE_INFO]

    @classmethod
    def fix_assoc_obj_image_urls(cls, assoc: dict, platform_info: PlatformInfo):
        if not assoc:
            return
        image = assoc.get('image')
        if not image:
            return
        patch_collection_image_if_necessary(image, platform_info, None)

    @classmethod
    def fix_collection_image_urls(cls, collection: dict, platform_info: PlatformInfo):
        """
        Check if collection has images with host distinct from 'avatars.mds.yandex.net', and generates mds url
        from mds id instead of non mds url
        For example, collection has field 'collection_images', and some object has field 'original' with value
        'https://some-non-mds.url.net/line-to-image.jpg'
        """
        if not collection:
            return
        images = collection.get('collection_images')
        if not images:
            return
        for image in images:
            patch_collection_image_if_necessary(image, platform_info, None)

    @classmethod
    def fix_collections_image_urls(cls, response: dict, platform_info: PlatformInfo):
        collections = get_related_object_by_type(
            get_related_object(response.get(FIELD_ENTITY_DATA)), RelatedObjectType.COLLECTIONS)
        if not collections or not collections.get(FIELD_OBJECT):
            logger.debug('No collections, nothing to patch')
            return
        collections_object = collections[FIELD_OBJECT]
        for collection in collections_object:
            cls.fix_collection_image_urls(collection, platform_info)

    @classmethod
    def fix_assoc_image_urls(cls, response: dict, platform_info: PlatformInfo):
        assoc_container = get_related_object_by_type(
            get_related_object(response.get(FIELD_ENTITY_DATA)), RelatedObjectType.ASSOC)
        if not assoc_container or not assoc_container.get(FIELD_OBJECT):
            logger.debug('No assoc, nothing to patch')
            return
        assoc_object = assoc_container[FIELD_OBJECT]
        for assoc in assoc_object:
            cls.fix_assoc_obj_image_urls(assoc, platform_info)
            remove_legal_for_trailer(assoc)

    @staticmethod
    def get_association_object(response):
        entity_data = response.get(FIELD_ENTITY_DATA)
        if not entity_data:
            return None
        related_objects = response[FIELD_ENTITY_DATA].get(FIELD_RELATED_OBJECT)
        if not related_objects:
            return None
        for obj in related_objects:
            if obj[FIELD_RELATED_OBJECT_TYPE] == RelatedObjectType.ASSOC.value:
                return obj
        return None

    @classmethod
    def filter_assoc_for_module(cls, response: dict, platform_info: PlatformInfo):
        if platform_info not in YANDEX_MODULE:
            return
        assoc_container = cls.get_association_object(response)
        if not assoc_container or not assoc_container.get(FIELD_OBJECT):
            return
        assoc_object = assoc_container[FIELD_OBJECT]
        assoc_container[FIELD_OBJECT] = [assoc for assoc in assoc_object if 'legal' in assoc]


class SearchViewV5(SearchView):
    NAME = 'search5'
    PARENT_COLLECTION_MORE_URL_VIEW = ParentCollectionViewV5


class SearchViewV6(SearchView):
    NAME = 'search6'
    PARENT_COLLECTION_MORE_URL_VIEW = ParentCollectionViewV6


class SearchViewV7(SearchView):
    NAME = 'search7'
    PARENT_COLLECTION_MORE_URL_VIEW = ParentCollectionViewV7


class SearchV4View(SearchView):
    NAME = 'search4'
    PARENT_COLLECTION_MORE_URL_VIEW = ParentCollectionView


class SuggestHistoryView(BaseSearchView):
    validator_class = serializers.SuggestHistoryValidator

    @swagger_base.swagger_schema(swagger_search.SuggestHistorySpec)
    def get(self, request):
        validated_data = self.get_validated_data(request)
        params = {
            'text': validated_data['text'],
            'uuid': self.get_uuid(request),
        }
        headers = self.prepare_headers(request)
        response = api.vs.history.suggest_history(params=params, headers=headers)
        return Response(status=200, data=response)


class AliceSearchV7(SearchView):
    NAME = 'alice_search7'
    search_result_serializer = protobuf.SearchResultSerializer()

    def select_parent_collection_by_entref(self, entref: str, search_response: dict) -> Optional[dict]:
        try:
            parent_collection_entref = search_response[FIELD_ENTITY_DATA][FIELD_PARENT_COLLECTION]['path'][0][FIELD_ENTREF]
        except KeyError:
            return None
        if parent_collection_entref == entref:
            return {FIELD_ENTITY_DATA: {FIELD_PARENT_COLLECTION:
                                            search_response[FIELD_ENTITY_DATA][FIELD_PARENT_COLLECTION]}}
        return None

    def select_related_object_by_entref(self, entref: str, search_response: dict) -> Optional[dict]:
        try:
            related_objects = search_response[FIELD_ENTITY_DATA][FIELD_RELATED_OBJECT]
        except KeyError:
            return None
        if not related_objects:
            return None
        for related_object in related_objects:
            if related_object.get(FIELD_ENTREF) == entref:
                return {FIELD_ENTITY_DATA: {FIELD_RELATED_OBJECT: related_object}}
        return None

    def select_collection_by_entref(self, entref: str, search_response: dict) -> Optional[dict]:
        if not entref:
            return None
        parent_collection = self.select_parent_collection_by_entref(entref, search_response)
        if parent_collection:
            return parent_collection
        related_object = self.select_related_object_by_entref(entref, search_response)
        if related_object:
            return related_object
        return None

    def get_base_info_horizontal_poster(self, search_response: dict) -> Optional[str]:
        try:
            return search_response[FIELD_ENTITY_DATA][FIELD_RICH_INFO][FIELD_VH_META][FIELD_CONTENT_GROUPS][0][FIELD_HORIZONTAL_POSTER]
        except (KeyError, TypeError, IndexError):
            return None

    def is_log_enabled(self):
        current_dc = os.environ.get('DEPLOY_NODE_DC', '')  # ex: sas
        return bool(current_dc and current_dc in settings.LOG_UE2E_SEARCH_REQUESTS_DC)

    def is_log_enabled_sampled(self):
        try:
            # число указывает, в скольки случаях из десяти мы залогируем запрос
            # если не указано, значит = 10 - логируем все
            sampling_rate = int(settings.LOG_UE2E_SEARCH_SAMPLING_RATE)
        except ValueError:
            sampling_rate = 10

        if random.randint(1, 10) <= sampling_rate:
            return True

        logger.info('This request is not logged bacause it is not in the sample')
        return False

    def log_request_and_response(self, request, response):
        """
        Сбор запросов/ответов в ручку поиска для корзинки
        """
        if self.is_log_enabled() and self.is_log_enabled_sampled():
            try:
                logger.info('VSLOG %s %s', request.get_full_path(), response)
            except Exception as err:
                logger.error(f'Can not log search query: {err}')

    def get(self, request):
        search_response = self.get_response(request)
        validated_data = self.get_validated_data(request)
        entref = validated_data.get(FIELD_ENTREF)
        response_by_entref = self.select_collection_by_entref(entref, search_response)
        if response_by_entref:
            search_response = response_by_entref
        result = TSearchResultData()
        context = {protobuf.CTX_PLATFORM_INFO: request.platform_info,
                   protobuf.CTX_BASE_INFO_HORIZONTAL_POSTER: fix_schema(
                       self.get_base_info_horizontal_poster(search_response))}
        self.search_result_serializer.serialize(
            search_response,
            result,
            context)
        response = json_format.MessageToDict(result)

        self.log_request_and_response(request, response)

        return Response(status=200, data=response)


class AliceParentCollectionV7(ParentCollectionView):
    NAME = 'alice_parent_collection7'
    search_result_serializer = protobuf.SearchResultSerializer()

    def get(self, request):
        parent_collection_result = {'entity_data': self.get_response(request)}
        result = TSearchResultData()
        self.search_result_serializer.serialize(parent_collection_result,
                                                result,
                                                {'platform_info': request.platform_info})
        response = json_format.MessageToDict(result)
        return Response(status=200, data=response)


class AliceTopTenV8(PlatformAPIView):
    """
    Ручка отдает карусель для показа на пустом экране поиска
    """
    NAME = 'alice_top_ten'

    def get(self, request):
        builder_request = VhCarouselInfo(
            {'carousel_id': 'hhbderssgqfzfmwbhh', 'offset': 0, 'limit': 10, 'tag': 'search'},
            request,
            self.vh_headers(request),
            self.NAME
        )

        builder = CarouselVideohubResultBuilder(builder_request)
        vh_response = builder.get_result()

        result = TSearchResultData()
        context = {
            protobuf.CTX_PLATFORM_INFO: request.platform_info,
            protobuf.CTX_TITLE: 'Топ-10 за месяц',
        }

        protobuf.VhCarouselAsSearchResultSerializer().serialize(vh_response, result, context)

        response = json_format.MessageToDict(result)
        return Response(status=200, data=response)
