import logging

from rest_framework.response import Response
from smarttv.droideka.proxy import api, result_builder

from smarttv.droideka.proxy.serializers.fast_response import SeriesSeasonV6Serializer, SeriesEpisodeV6Serializer, \
    SeriesEpisodeV7Serializer, SeriesSeasonV7Serializer, UnboundSeriesEpisodeV7Serializer
from smarttv.droideka.proxy.serializers.serializers import SeriesSeasonsValidator, SeriesEpisodesValidator, \
    UnboundSeriesEpisodesValidator
from smarttv.droideka.proxy.swagger.base import swagger_schema
from smarttv.droideka.proxy.swagger.series import SeasonsSpec, EpisodeSpec
from smarttv.droideka.proxy.views.base import PlatformAPIView
from smarttv.droideka.proxy import exceptions
from smarttv.droideka.utils.url import build_view_url
from smarttv.droideka.utils import PlatformInfo
from smarttv.droideka.proxy.result_builder import UnboundEpisodesResultBuilder

logger = logging.getLogger(__name__)

FIELD_SET = 'set'
FIELD_SEASONS = 'seasons'
FIELD_EPISODES = 'episodes'
FIELD_MORE = 'more'
FIELD_SERIES_ID = 'series_id'


class SeasonsView(PlatformAPIView):
    NAME = 'seasons_6'
    validator_class = SeriesSeasonsValidator

    season_serializer = SeriesSeasonV6Serializer()

    @classmethod
    def get_more_seasons_url(cls, params, request):
        return build_view_url(params, cls.NAME, request)

    @classmethod
    def get_more_url_args(cls, series_id, offset, limit, result_episodes_amount):
        if result_episodes_amount < limit:
            raise exceptions.NoMoreError()
        return {
            FIELD_SERIES_ID: series_id,
            'offset': offset + result_episodes_amount,
            'limit': limit
        }

    @classmethod
    def fill_more_seasons_url(cls, response, series_id, offset, limit, request):
        if not response.get(FIELD_SEASONS):
            logger.debug('No seasons in the response')
            return
        try:
            response[FIELD_MORE] = cls.get_more_seasons_url(
                cls.get_more_url_args(
                    series_id=series_id,
                    offset=offset,
                    limit=limit,
                    result_episodes_amount=len(response[FIELD_SEASONS])
                ),
                request)
        except exceptions.NoMoreError:
            logger.debug('No more seasons')
            pass

    def get_serialized_season(self, raw_season, more_episodes_limit, request):
        result = self.season_serializer.serialize(raw_season, {'platform_info': request.platform_info})
        season_id = result.get(SeriesSeasonV6Serializer.FIELD_SEASON_ID)
        EpisodesView.fill_more_episodes_url(
            response=result,
            season_id=season_id,
            offset=0,
            limit=more_episodes_limit,
            request=request,
            force_more_url=True)
        return result

    def get_serialized_seasons(self, raw_season_container, more_episodes_limit, request):
        result = {}
        raw_seasons = raw_season_container.get(FIELD_SET)
        if raw_seasons:
            result[FIELD_SEASONS] = [
                self.get_serialized_season(raw_season, more_episodes_limit, request) for raw_season in raw_seasons]

        return result

    @swagger_schema(SeasonsSpec)
    def get(self, request):
        data = self.get_validated_data(request)
        series_id = data['series_id']
        limit = data['limit']
        offset = data['offset']
        result = result_builder.SeriesSeasonsV6ResultBuilder(
            request=request,
            series_id=series_id,
            limit=limit,
            offset=offset,
            headers=self.vh_headers(request)).get_result()

        if not result or 'set' not in result:
            raise exceptions.NotFoundError(f"'{series_id}' not found")
        response = self.get_serialized_seasons(result, data['more_episodes_limit'], request)
        api.vh.propagate_vh_tracking_params(result, response)
        self.fill_more_seasons_url(response, series_id, offset, limit, request)

        return Response(response)


class SeasonsViewV7(SeasonsView):
    NAME = 'seasons_7'

    season_serializer = SeriesSeasonV7Serializer()


class EpisodesView(PlatformAPIView):
    NAME = 'episodes_6'
    validator_class = SeriesEpisodesValidator

    episode_serializer = SeriesEpisodeV6Serializer()

    @classmethod
    def get_more_episodes_url(cls, params, request):
        return build_view_url(params, cls.NAME, request)

    @classmethod
    def get_more_url_args(cls, season_id, offset, limit, result_episodes_amount, force_more_url):
        if not force_more_url and result_episodes_amount == 0:
            raise exceptions.NoMoreError()
        return {
            SeriesSeasonV6Serializer.FIELD_SEASON_ID: season_id,
            'offset': offset + limit,
            'limit': limit
        }

    @classmethod
    def fill_more_episodes_url(cls, response, season_id, offset, limit, request, force_more_url=False):
        if not response.get(FIELD_EPISODES):
            logger.debug('No episodes in the response')
            return
        try:
            response[FIELD_MORE] = cls.get_more_episodes_url(
                cls.get_more_url_args(
                    season_id=season_id,
                    offset=offset,
                    limit=limit,
                    result_episodes_amount=len(response[FIELD_EPISODES]),
                    force_more_url=force_more_url
                ),
                request)
        except exceptions.NoMoreError:
            logger.debug('No more episodes')
            pass

    def get_serialized_episodes(self, raw_episodes_container, platform_info: PlatformInfo):
        result = {}
        raw_episodes = raw_episodes_container.get(FIELD_SET)
        if not raw_episodes:
            return result
        ctx = {'platform_info': platform_info}
        result[FIELD_EPISODES] = [self.episode_serializer.serialize(raw_episode, ctx) for raw_episode in raw_episodes]

        return result

    @swagger_schema(EpisodeSpec)
    def get(self, request):
        data = self.get_validated_data(request)
        season_id = data['season_id']
        limit = data['limit']
        offset = data['offset']
        rb = result_builder.SeriesEpisodesV6ResultBuilder(
            request=request,
            season_id=season_id,
            limit=limit,
            offset=offset,
            headers=self.vh_headers(request)
        )

        try:
            result = rb.get_result()
        except result_builder.SeriesEpisodesV6ResultBuilder.NoResultError:
            raise exceptions.NotFoundError(f"'{season_id}' not found")

        response = self.get_serialized_episodes(result, request.platform_info)
        self.fill_more_episodes_url(response, season_id, offset, limit, request)

        api.vh.propagate_vh_tracking_params(result, response)
        return Response(response)


class EpisodesViewV7(EpisodesView):
    NAME = 'episodes_7'

    episode_serializer = SeriesEpisodeV7Serializer()


class UnboundEpisodesView(PlatformAPIView):
    NAME = 'unbound_episodes_7'

    validator_class = UnboundSeriesEpisodesValidator
    episode_serializer = UnboundSeriesEpisodeV7Serializer()

    def get(self, request):
        data = self.get_validated_data(request)
        season_number = data['season_number']
        pivot_episode_number = data['pivot_episode_number']

        context = {'pivot_season_number': season_number, 'pivot_episode_number': pivot_episode_number}
        raw_episodes = UnboundEpisodesResultBuilder(data, self.vh_headers(request), request).get_result()
        result = [self.episode_serializer.serialize(e, context) for e in raw_episodes]

        return Response(result)
