import logging
import uuid

import pytz
from django.conf import settings
from django.db import transaction
from django.db.models import Q
from django.utils import timezone
from rest_framework import exceptions, status as http_status
from rest_framework.response import Response
from rest_framework.views import APIView

from smarttv.droideka import unistat
from smarttv.droideka.proxy import models
from smarttv.droideka.proxy.api.mediabilling import MediaBillingAPI
from smarttv.droideka.proxy.constants.reserved_shared_pref_keys import PREF_EXPIRE_DATE, PREF_EXPIRE_DATE_FORMAT, \
    PREF_TEXT
from smarttv.droideka.proxy.serializers import serializers
from smarttv.droideka.proxy.swagger.base import swagger_schema
from smarttv.droideka.proxy.swagger.setup_wizard import ActivationSpec, GetGiveGiftSpec, \
    UserAgreementSpec, PostGiveGiftSpec, KpGiftsSpec
from smarttv.droideka.proxy.views.base import PlatformAPIView, RequestValidatorMixin

from smarttv.utils import headers

logger = logging.getLogger(__name__)
mediabilling_api = MediaBillingAPI(use_tvm=True)

DEFAULT_DATE = timezone.datetime(2020, 1, 1, 0, 0, 0, tzinfo=pytz.utc)
DEFAULT_UUID = uuid.UUID('00000000-0000-0000-0000-000000000000')

# Android OS returns this placeholder to client, when client doesn't have a location permission
# More: https://developer.android.com/about/versions/marshmallow/android-6.0-changes.html#behavior-hardware-id
NOT_ALLOWED_MAC = '02:00:00:00:00:00'

activations_counter = unistat.manager.get_counter('suw_activations')
unknown_devices_counter = unistat.manager.get_counter('suw_unknown_devices')
activated_promocodes_counter = unistat.manager.get_counter('suw_activated_promocodes')


class BaseSUWView(PlatformAPIView):
    validator_class = serializers.BaseSetupWizardValidator
    required_headers = (headers.WIFI_MAC_HEADER, headers.ETHERNET_MAC_HEADER)
    platform_check_required = False
    need_kp_profile_creation = False

    def check_wifi_mac(self, wifi_mac):
        if wifi_mac == NOT_ALLOWED_MAC:
            raise exceptions.ValidationError(f"Bad Wi-Fi's mac: '{NOT_ALLOWED_MAC}'")

    def get_identifiers(self, request):
        validated_data = self.get_validated_data(request)

        if 'serial_number' in validated_data:
            serial_number = validated_data['serial_number'].lower()
            if request.headers.get(headers.SERIAL_HEADER, '').lower() != serial_number:
                raise exceptions.ValidationError("Serials in headers and in query don't match")
        else:
            serial_number = None

        wifi_mac = request.headers.get(headers.WIFI_MAC_HEADER, '').lower()
        self.check_wifi_mac(wifi_mac)
        ethernet_mac = request.headers.get(headers.ETHERNET_MAC_HEADER, '').lower()
        hardware_id = models.calculate_hardware_id(ethernet_mac, wifi_mac)

        return {
            'serial_number': serial_number,
            'wifi_mac': wifi_mac,
            'ethernet_mac': ethernet_mac,
            'hardware_id': hardware_id,
        }


class ActivationView(BaseSUWView):
    @staticmethod
    def activate(serial_number, wifi_mac, ethernet_mac, hardware_id):
        target_db = settings.DB_MASTER_SERIALIZED
        device = models.Device.objects.using(target_db).select_for_update().filter(hardware_id=hardware_id).first()

        if device is None:
            logger.info('New device, serial: %s, hardware_id: %s', serial_number, hardware_id)

            query = Q(type='serial_number', value=serial_number) | Q(type='wifi_mac', value=wifi_mac) | Q(
                type='ethernet_mac', value=ethernet_mac)
            if models.ValidIdentifier.objects.using(target_db).filter(query).count() == 0:
                logger.warning('Unknown TV identifier')
                unknown_devices_counter.increment()
                raise exceptions.PermissionDenied()

            query = Q(wifi_mac=wifi_mac) | Q(ethernet_mac=ethernet_mac)
            if models.Device.objects.using(target_db).filter(query).count() > 0:
                logger.warning('Fraud? Device activated with different mac. Eth: %s, wifi: %s', ethernet_mac, wifi_mac)
                raise exceptions.PermissionDenied()

            device, created = models.Device.objects.using(target_db).select_for_update().get_or_create(
                hardware_id=hardware_id,
                defaults={
                    'wifi_mac': wifi_mac, 'ethernet_mac': ethernet_mac, 'kp_gifts_id': DEFAULT_UUID,
                    'created_at': DEFAULT_DATE, 'serial_number': serial_number, 'kp_gifts_given': False}
            )
            if created:
                device.kp_gifts_id = uuid.uuid4()
                device.created_at = timezone.now()
                device.save()
                activations_counter.increment()
                logger.info('Device successfully created')
            else:
                logger.info('Device created in another request. Skipping')

    @swagger_schema(ActivationSpec)
    def post(self, request):
        identifiers = self.get_identifiers(request)
        with transaction.atomic(using=settings.DB_MASTER_SERIALIZED):
            self.activate(
                serial_number=identifiers['serial_number'],
                wifi_mac=identifiers['wifi_mac'],
                ethernet_mac=identifiers['ethernet_mac'],
                hardware_id=identifiers['hardware_id'],
            )
        logger.info('Device activated, serial: %s, hardware_id: %s',
                    identifiers['serial_number'], identifiers['hardware_id'])
        return Response(status=http_status.HTTP_204_NO_CONTENT)


class GiveGift(BaseSUWView):
    authorization_required = True

    def response_503(self):
        logger.exception('Unexpected mediabilling error in subscription process')
        return Response(status=http_status.HTTP_503_SERVICE_UNAVAILABLE)

    @swagger_schema(PostGiveGiftSpec)
    def post(self, request):
        user_info = self.get_user_info(request)
        identifiers = self.get_identifiers(request)
        serial_number = identifiers['serial_number']
        hardware_id = identifiers['hardware_id']
        try:
            with transaction.atomic():
                self.give_gift(user_info.passport_uid, serial_number, self.user_ip, hardware_id)
        except mediabilling_api.RequestError as e:
            if e.status_code == http_status.HTTP_404_NOT_FOUND:
                return Response(status=http_status.HTTP_404_NOT_FOUND)
            else:
                return self.response_503()
        except mediabilling_api.APIError:
            return self.response_503()
        return Response(status=http_status.HTTP_204_NO_CONTENT)

    @swagger_schema(GetGiveGiftSpec)
    def get(self, request):
        identifiers = self.get_identifiers(request)
        hardware_id = identifiers['hardware_id']

        device = models.Device.objects.using(settings.DB_REPLICA).filter(hardware_id=hardware_id).first()

        if device is None:
            logger.error('Gift call before successful activation')
            raise exceptions.NotFound('Device not found')

        return Response({
            'gift_available': device.is_gift_available(),
            'kp_gifts_id': device.get_kp_gifts_id(),
        }, status=http_status.HTTP_200_OK)

    def give_gift(self, passport_uid, serial_number, user_ip, hardware_id):
        logger.info('Checking if gift is given for device %s', serial_number)

        device = models.Device.objects.select_for_update().filter(hardware_id=hardware_id).first()

        if device is None:
            logger.error('Gift call before successful activation')
            raise exceptions.NotFound('Device not found')

        if not device.is_gift_available():
            if device.subscription_puid != passport_uid:
                logger.error('Trying to get unavailable gift for device %s, %s', serial_number, hardware_id)
                raise exceptions.PermissionDenied('Gifts are unavailable')
            logger.info('Gifts already given for device %s, %s', serial_number, hardware_id)
            return

        logger.info('Issuing promocode for user: %s', passport_uid)
        promocode = mediabilling_api.clone_promocode(prototype=settings.ETALON_PROMOCODE)
        logger.info('Successfully issued promocode %s for user %s', promocode, passport_uid)

        logger.info('Saving gift for device %s, %s, user %s', serial_number, hardware_id, passport_uid)
        device.subscription_puid = passport_uid
        device.promocode = promocode
        device.save()

        logger.info('Activating promocode %s', promocode)
        self.consume_promocode(passport_uid, promocode, user_ip)
        activated_promocodes_counter.increment()
        logger.info('Successfully activated promocode %s for user %s', promocode, passport_uid)

    @staticmethod
    def consume_promocode(passport_uid, promocode, user_ip):
        response = mediabilling_api.consume_promocode(uid=passport_uid, promocode=promocode, user_ip=user_ip, timeout=4)

        try:
            mediabilling_status = response['status'].lower()
        except KeyError:
            raise mediabilling_api.APIError('Unexpected mediabilling answer')
        if mediabilling_status != 'success':
            logger.error('Unexpected status %s from mediabilling', mediabilling_status)
            raise mediabilling_api.APIError(f'Unsuccessful request to mediabilling, status={mediabilling_status}')


class UserAgreement(APIView):
    DEFAULT_EXPIRE_DATE_FORMAT = '%d.%m.%Y'

    @swagger_schema(UserAgreementSpec)
    def get(self, _):
        user_agreement_expire_date = models.SharedPreferences.get_datetime(PREF_EXPIRE_DATE)
        user_agreement_expire_date_format = models.SharedPreferences.get_string(
            PREF_EXPIRE_DATE_FORMAT)
        user_agreement_text = models.SharedPreferences.get_string(PREF_TEXT)
        if user_agreement_expire_date is None or user_agreement_text is None:
            return Response(data={'detail': 'User agreement not found'}, status=404)
        return Response({
            'text': user_agreement_text,
            'to_date': user_agreement_expire_date.strftime(
                user_agreement_expire_date_format or UserAgreement.DEFAULT_EXPIRE_DATE_FORMAT)
        })


class KpGiftsGiven(APIView, RequestValidatorMixin):
    validator_class = serializers.KpGiftsGivenValidator

    @swagger_schema(KpGiftsSpec)
    def post(self, request):
        validated_data = self.get_validated_data(request)
        with transaction.atomic():
            try:
                device = models.Device.objects.select_for_update().get(kp_gifts_id=validated_data['kp_gifts_id'])
                device.kp_gifts_given = True
                device.save()
            except models.Device.DoesNotExist:
                raise exceptions.NotFound('Device not found')
        return Response(status=http_status.HTTP_204_NO_CONTENT)
