import logging

from rest_framework import status as http_status
from rest_framework.response import Response
from smarttv.droideka.proxy.exceptions import InternalServerError
from smarttv.droideka.proxy.models import SharedPreferences
from smarttv.droideka.proxy.constants.reserved_shared_pref_keys import PREF_SMOTRESHKA_API_ENABLED
from smarttv.droideka.proxy.serializers.serializers import SmotreshkaTvPlaybackBeginRequestValidator
from smarttv.droideka.proxy import smotreshka
from smarttv.droideka.proxy.views.base import PlatformAPIView
from smarttv.utils.headers import QUASAR_DEVICE_ID, USER_AGENT_HEADER

logger = logging.getLogger(__name__)


class SmotreshkaTvPlaybackBeginView(PlatformAPIView):
    NAME='smotreshka_begin_v8'
    validator_class = SmotreshkaTvPlaybackBeginRequestValidator
    required_headers = (QUASAR_DEVICE_ID, )

    def get_device_ip(self, validated_data, request):
        if validated_data.get('device_ip'):
            return validated_data['device_ip']
        user_ip = self.get_user_ip(request)
        if not user_ip:
            msg = 'Could not get user ip to open session'
            logger.error(msg)
            raise InternalServerError(msg)
        return user_ip

    def get_media_id(self, validated_data):
        return validated_data['content_id']

    def post(self, request):
        logger.info('begin smotreshka session %s', request.data)
        validated_data = self.get_validated_data(request)

        if not SharedPreferences.get_int(PREF_SMOTRESHKA_API_ENABLED):
            logger.info('Smotreshka is disabled by shared preference')
            return Response({'error': 'disabled'}, http_status.HTTP_403_FORBIDDEN)

        quasar_device_id = request.headers.get(QUASAR_DEVICE_ID)
        user_agent = request.headers.get(USER_AGENT_HEADER, '')
        is_module = request.platform_info.is_module()

        params = {
            'media_id': self.get_media_id(validated_data),
            'quasar_device_id': quasar_device_id,
            'user_agent': user_agent,
            'device_ip': self.get_device_ip(validated_data, request),
            'device_type': 'stb' if is_module else 'smarttv',
            'purpose': validated_data.get('purpose'),
            'playback_session_id': validated_data.get('playback_session_id'),
        }

        api_response = smotreshka.begin(**params)

        logger.info('begin response: %s', api_response)

        response = {
            'playback_session_id': api_response['playbackSession']['id'],
            'playback_url': api_response['playbackUrl'],
        }
        return Response(response, status=http_status.HTTP_200_OK)
