import json
import logging
import os
from http import client

if bool(os.environ.get('YA_TEST_RUNNER')):
    os.environ['YENV_TYPE'] = 'development'
    os.environ['ONTO_ID_MAPPING_PATH'] = ''
    os.environ['CONTENT_TYPE_MAPPING_PATH'] = ''

import psycopg2.extensions
import yenv
from django_idm_api.settings import *  # noqa
from django_pgaas import HostManager
from django_yauth.settings import *  # noqa
from requests import adapters
from tvmauth import BlackboxTvmId as BlackboxClientId
from tvm2.protocol import BLACKBOX_MAP

from smarttv.utils import headers
from smarttv.utils.version import VERSION


def get_hosts(hosts_str):
    """
    Splits the hosts string and returns list or empty list
    """
    if not hosts_str:
        return []
    return hosts_str.split(',')


BASE_DIR = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
os.environ.setdefault("REQUESTS_CA_BUNDLE", '/etc/ssl/certs')

DEBUG = yenv.type == 'development'

if DEBUG:
    os.environ.setdefault('SECRET_KEY', '_________')

# SECURITY WARNING: keep the secret key used in production secret!
SECRET_KEY = os.environ['SECRET_KEY']

BASE_URL = 'droideka'

FULL_HTTP_LOG = bool(os.environ.get('FULL_HTTP_LOG', 0))
if DEBUG and FULL_HTTP_LOG:
    client.HTTPConnection.debuglevel = 1

ALLOWED_HOSTS = ['*']

# Application definition

INSTALLED_APPS = [
    'django_idm_api',
    'django_pgaas',
    'django_yauth',
    'django.contrib.admin',
    'django.contrib.auth',
    'django.contrib.contenttypes',
    'django.contrib.sessions',
    'django.contrib.messages',
    'django.contrib.staticfiles',
    'smarttv.droideka.proxy.apps.SmartTvProxyConfig',
    'rest_framework',
    'drf_yasg',
]

REST_FRAMEWORK = {
    'DEFAULT_RENDERER_CLASSES': [
        'rest_framework.renderers.JSONRenderer'
    ],
    'EXCEPTION_HANDLER': 'smarttv.droideka.proxy.exceptions.custom_exception_handler',
}

MIDDLEWARE = [
    'smarttv.utils.middleware.FixEmptyHostMiddleware',
    'smarttv.utils.middleware.LoggingContextMiddleware',
    'smarttv.droideka.middleware.CorsMiddleware',
    'smarttv.droideka.middleware.UnistatMiddleware',
    'csp.middleware.CSPMiddleware',
    'django.middleware.common.CommonMiddleware',
    'django.middleware.security.SecurityMiddleware',
    'django.contrib.sessions.middleware.SessionMiddleware',
    'django.middleware.common.CommonMiddleware',
    'django.middleware.csrf.CsrfViewMiddleware',
    'plus.utils.yauth.EmptyAuthMiddleware',
    'django_idm_api.middleware.TVMMiddleware',
    'django.contrib.messages.middleware.MessageMiddleware',
    'django.middleware.clickjacking.XFrameOptionsMiddleware',
    'django.middleware.http.ConditionalGetMiddleware',
]

CORS_ORIGIN_ALLOW_ALL = False
CORS_HOSTS = [
    'api.smarttv.yandex.ru',
    'api.pre.smarttv.yandex.ru',
    'api.tst.smarttv.yandex.ru',
    'droideka.smarttv.yandex.net',
    'droideka.pre.smarttv.yandex.net',
    'droideka.tst.smarttv.yandex.net',
]

if DEBUG:
    DEBUG_CORS_HOST = os.environ.get('DEBUG_CORS_HOST')
    if DEBUG_CORS_HOST:
        CORS_HOSTS.append(DEBUG_CORS_HOST)

ROOT_URLCONF = 'smarttv.droideka.urls'

TEMPLATES = [
    {
        'BACKEND': 'library.python.django.template.backends.arcadia.ArcadiaTemplates',
        'OPTIONS': {
            'debug': DEBUG,
            'loaders': [
                'library.python.django.template.loaders.resource.Loader',
                'library.python.django.template.loaders.app_resource.Loader',
            ],
            'context_processors': [
                'django.template.context_processors.debug',
                'django.template.context_processors.request',
                'django.contrib.auth.context_processors.auth',
                'django.contrib.messages.context_processors.messages',
            ],
        },
    },
]

FORM_RENDERER = 'library.python.django.template.backends.forms_renderer.ArcadiaRenderer'

LANGUAGE_CODE = 'en-us'
USE_I18N = False

###########################
# Date and Time settings
TIME_ZONE = 'Europe/Moscow'
USE_TZ = True

# setting USE_L10N=False to use DATE_FORMAT and DATETIME_FORMAT for formatting instead of local formatting
USE_L10N = False
DATE_FORMAT = 'Y-m-d'
DATETIME_FORMAT = 'Y-m-d H:i:s'
###########################


# Database
# https://docs.djangoproject.com/en/1.11/ref/settings/#databases

DB_HOSTS = yenv.choose_key_by_type({
    'production': [
        ('man-pt9iq3zmm0avcqsh.db.yandex.net', 'man'),
        ('sas-28cbxh0hex2i6f94.db.yandex.net', 'sas'),
        ('vla-idsdurmoyqhgh3ru.db.yandex.net', 'vla'),
    ],
    'prestable': [
        ('man-1gqgmyadtni8zhfn.db.yandex.net', 'man'),
        ('sas-jt6ryyjuwa7taio3.db.yandex.net', 'sas'),
        ('vla-am32qj44xy9ryzcp.db.yandex.net', 'vla'),
    ],
    'testing': [
        ('man-alrjs014rkrhzpxj.db.yandex.net', 'man'),
        ('sas-zembq6id6fi2me5z.db.yandex.net', 'sas'),
        ('vla-yukwk4ahzmiv5wo1.db.yandex.net', 'vla'),
    ],
    'stress': [
        ('man-tcgu5jzvw7nbtpst.db.yandex.net', 'man'),
        ('sas-f1naplr0ireoiam9.db.yandex.net', 'sas'),
        ('vla-71cs04vc16yiu3om.db.yandex.net', 'vla'),
    ],
}, fallback=True)
host_manager = HostManager(DB_HOSTS)

DB_USER = os.environ.get('DB_USER')
DB_PASSWORD = os.environ.get('DB_PASSWORD')

DB_MASTER = 'default'
DB_MASTER_SERIALIZED = 'master_serialized'
DB_REPLICA = 'replica'

DB_MASTER_OPTIONS_CONFIG = {
    'target_session_attrs': 'read-write'
}

DB_MASTER_SERIALIZED_CONFIG = {
    'isolation_level': psycopg2.extensions.ISOLATION_LEVEL_SERIALIZABLE,
}

DB_REPLICA_OPTIONS_CONFIG = {
    'target_session_attrs': 'any'
}

BASE_DB_OPTIONS = {
    'MAX_CONNS': 20,
    'sslmode': 'require',
}

BASE_DB_CONFIG = {
    'ENGINE': 'smarttv.droideka.db_backend',
    'HOST': host_manager.host_string,
    'PORT': 6432,
    'USER': DB_USER,
    'PASSWORD': DB_PASSWORD,
    'NAME': 'smarttv',
    'CONN_MAX_AGE': int(os.getenv('PG_CONN_MAX_AGE', 0)),
}

DATABASES = {
    DB_MASTER: dict(BASE_DB_CONFIG, OPTIONS=dict(BASE_DB_OPTIONS, **DB_MASTER_OPTIONS_CONFIG)),
    DB_MASTER_SERIALIZED: dict(BASE_DB_CONFIG, OPTIONS=dict(BASE_DB_OPTIONS, **DB_MASTER_OPTIONS_CONFIG,
                                                            **DB_MASTER_SERIALIZED_CONFIG)),
    DB_REPLICA: dict(BASE_DB_CONFIG, OPTIONS=dict(BASE_DB_OPTIONS, **DB_REPLICA_OPTIONS_CONFIG)),
}

###################################
# Logging
ADDITIONAL_LOGGING_HEADERS = {
    'device_id': headers.DEVICE_ID_HEADER,
    'uuid': headers.UUID_HEADER,
    'wifi_mac': headers.WIFI_MAC_HEADER,
    'ethernet_mac': headers.ETHERNET_MAC_HEADER,
    'build_fingerprint': headers.FINGERPRINT_HEADER,
    'serial': headers.SERIAL_HEADER,
    'user_agent': headers.USER_AGENT_HEADER,
    'quasar_device_id': headers.QUASAR_DEVICE_ID,
    'quasar_platform': headers.QUASAR_PLATFORM,
}

UNIFIED_AGENT_LOGS_URI = os.getenv('UNIFIED_AGENT_LOGS_URI')


def get_log_level():
    log_level = os.getenv('LOGGING_LEVEL')
    if log_level not in ('CRITICAL', 'FATAL', 'ERROR', 'WARNING', 'WARN', 'INFO', 'DEBUG'):
        log_level = yenv.choose_key_by_type({
            'production': logging.INFO,
            'testing': logging.DEBUG,
            'stress': logging.INFO,
        }, fallback=True)
    return log_level


def get_log_handlers():
    handlers = {
        'console': {
            'class': 'logging.StreamHandler',
            'formatter': 'default',
        },
        'qloud': {
            'class': 'logging.StreamHandler',
            'formatter': 'json',
        },
    }

    if UNIFIED_AGENT_LOGS_URI:
        handlers['logbroker'] = {
            'class': 'logbroker.unified_agent.client.python.UnifiedAgentYdHandler',
            'logger_name': 'stdout',
            'formatter': 'json',
            'uri': UNIFIED_AGENT_LOGS_URI,
        }

    return handlers


def get_log_handlers_list():
    handlers = yenv.choose_key_by_type({
        'production': ['qloud'],
        'stress': ['qloud'],
        'development': ['console'],
    }, fallback=True)

    if UNIFIED_AGENT_LOGS_URI:
        handlers.append('logbroker')

    return handlers


LOGGING = {
    'version': 1,
    'disable_existing_loggers': False,
    'formatters': {
        'default': {
            'format': '[%(asctime)s] [%(process)d] [%(levelname)s] [%(name)s] %(message)s',
            'datefmt': '%Y-%m-%d %H:%M:%S',
            'class': 'logging.Formatter',
        },
        'raw': {
            'format': '%(message)s',
            'datefmt': '',
            'class': 'logging.Formatter',
        },
        'json': {
            '()': 'ylog.format.QloudJsonFormatter',
        },
    },
    'handlers': get_log_handlers(),
    'loggers': {
        '': {
            'level': get_log_level(),
            'handlers': get_log_handlers_list(),
            'propagate': True,
        },
        'requests': {
            'level': 'WARN'
        },
        'django.db.backends': {
            'level': yenv.choose_key_by_type({
                'production': logging.WARNING,
                'testing': logging.INFO,
                'development': logging.DEBUG,
            }, fallback=True),
        },
        'gunicorn.access': {
            'handlers': get_log_handlers_list(),
        },
        'gunicorn.error': {
            'handlers': get_log_handlers_list(),
        },
    },
}

###################################
# Sentry
UNIFIED_AGENT_ERRORS_URI = os.getenv('UNIFIED_AGENT_ERRORS_URI')

if yenv.type in ('testing', 'prestable', 'production'):
    import sentry_sdk
    from sentry_sdk.integrations.logging import LoggingIntegration
    from sentry_sdk.integrations.django import DjangoIntegration

    sentry_init_params = dict(
        debug=DEBUG,
        environment=yenv.type,
        release=VERSION,
        send_default_pii=True,
        request_bodies='small',
        # sample_rate=0.05,  # TODO: enable sampling
        integrations=[
            DjangoIntegration(),
            LoggingIntegration(
                level=logging.INFO,  # Capture info and above as breadcrumbs
                event_level=logging.ERROR,  # Send errors as events
            )
        ],
    )

    if UNIFIED_AGENT_ERRORS_URI:
        from smarttv.utils.errorbooster import get_ua_transport
        sentry_init_params.update(
            transport=get_ua_transport(
                project_name='Smarttv-Droideka',
                unified_agent_uri=UNIFIED_AGENT_ERRORS_URI,
            ),
        )
    else:
        sentry_init_params.update(
            dsn=os.getenv('SENTRY_DSN', ''),
        )

    logging.info('Initializing Sentry SDK')
    sentry_sdk.init(**sentry_init_params)

    # only for usage by USaaS
    USAAS_DROIDEKA_ENVIRONMENT_HOST = os.environ['DROIDEKA_ENVIRONMENT_HOST']

    ONTO_ID_MAPPING_PATH = os.environ['ONTO_ID_MAPPING_PATH']
    CONTENT_TYPE_MAPPING_PATH = os.environ['CONTENT_TYPE_MAPPING_PATH']

###########################################################

# Static files (CSS, JavaScript, Images)
# https://docs.djangoproject.com/en/1.11/howto/static-files/
STATIC_ROOT = os.path.join(BASE_DIR, 'static')
STATIC_URL = f'/{BASE_URL}/static/'

STATICFILES_FINDERS = [
    'django.contrib.staticfiles.finders.FileSystemFinder',
    'django.contrib.staticfiles.finders.AppDirectoriesFinder',
    'library.python.django.contrib.staticfiles.finders.ArcadiaAppFinder'
]

SECURE_CONTENT_TYPE_NOSNIFF = True
SECURE_PROXY_SSL_HEADER = ('HTTP_X_FORWARDED_PROTO', 'https')

NETWORK_CONNECTION_POOL_MAX_SIZE = \
    int(os.environ.get('NETWORK_CONNECTION_POOL_MAX_SIZE', adapters.DEFAULT_POOLSIZE))
NETWORK_CONNECTION_POOL_SIZE = \
    int(os.environ.get('NETWORK_CONNECTION_POOL_SIZE', adapters.DEFAULT_POOLSIZE))

# TODO: think about API versions
FRONTENDVH_API_URL = 'https://frontend.vh.yandex.ru/'
VH_API_VERSION = 'v23'
VH_BETA_API_VERSION = 'test-sync'
FRONTENDVH_DEFAULT_TIMEOUT = int(os.environ.get('FRONTENDVH_DEFAULT_TIMEOUT', 3))
FRONTENDVH_DEFAULT_RETRIES = int(os.environ.get('FRONTENDVH_DEFAULT_RETRIES', 3))
VH_CHANNEL_ID_OFFSET = int(os.environ.get('VH_CHANNEL_ID_OFFSET', 0))

ENTITY_SEARCH_API_URL = 'https://yandex.ru/search/entity'
ENTITY_SEARCH_DEFAULT_TIMEOUT = 3
ENTITY_SEARCH_DEFAULT_RETRIES = 3

VIDEO_SEARCH_API_URL = 'http://knoss-video.yandex-team.ru/video/result'
VIDEO_SEARCH_DEFAULT_TIMEOUT = 1.5
VIDEO_SEARCH_DEFAULT_RETRIES = 3

SUGGEST_HISTORY_API_URL = 'https://yandex.ru/search/suggest-history'
SUGGEST_HISTORY_DEFAULT_TIMEOUT = 1
SUGGEST_HISTORY_DEFAULT_RETRIES = 3

OTT_CONTENT_API_URL = yenv.choose_key_by_type({
    'production': 'https://api.ott.yandex.ru/',
    'testing': 'https://api.ott.yandex.ru/prestable/',
    'stress': 'https://api.ott.yandex.ru/',
}, fallback=True)
OTT_API_VERSION = 'v9'
OTT_CONTENT_DEFAULT_TIMEOUT = 1
OTT_CONTENT_DEFAULT_RETRIES = 3

MUSIC_API_URL = 'https://api.music.yandex.net/'
MUSIC_DEFAULT_TIMEOUT = 3
MUSIC_DEFAULT_RETRIES = 1
MUSIC_DEFAULT_COVER_PATTERN = 'https://avatars.mds.yandex.net/get-dialogs/758954/music-default-cover/%%'

ALICE_BUSINESS_API_URL = yenv.choose_key_by_type({
    'testing': 'https://paskills-common-testing.alice.yandex.net/priemka/b2b/internal/',
    'production': 'https://dialogs.alice.yandex.net/b2b/internal/',
    'stress': 'https://dialogs.alice.yandex.net/b2b/internal/',
}, fallback=True)
ALICE_BUSINESS_DEFAULT_TIMEOUT = 1
ALICE_BUSINESS_DEFAULT_RETRIES = 3
ALICE_BUSINESS_CLIENT_ID = '2015561'

###################################
# S3 MDS settings
os.environ.setdefault('AWS_CA_BUNDLE', '/etc/ssl/certs/yandex-ca.pem')

S3_MDS_PUBLIC_HOST = 's3.yandex.net'
S3_MDS_HOST = 's3.mds.yandex.net'
S3_MDS_ANDROID_BUCKET_NAME = yenv.choose_key_by_type({
    'production': 'androidtvprod',
    'prestable': 'androidtvpre',
    'testing': 'androidtv',
}, fallback=True)
S3_MDS_ACCESS_KEY_ID = os.environ.get('S3_MDS_ACCESS_KEY_ID')
S3_MDS_ACCESS_SECRET_KEY = os.environ.get('S3_MDS_ACCESS_SECRET_KEY')

###################################

###################################
# Content Secure Policy (django-csp)
CSP_DEFAULT_SRC = ("'none'",)
CSP_SCRIPT_SRC = ("'unsafe-inline'", "'self'")
CSP_STYLE_SRC = ("'unsafe-inline'", "'self'")
CSP_FONT_SRC = ("'self'",)
CSP_IMG_SRC = yenv.choose_key_by_type({
    'production': ("'self'", 'androidtvprod.s3.yandex.net'),
    'testing': ("'self'", 'androidtv.s3.yandex.net'),
}, fallback=True)
###################################

###################################
# Configuration parameters for interacting with third party services
DEFAULT_CHUNK_SIZE = 5  # amount of data in the one response of sequential data source(which provides many objects)
DEFAULT_NETWORK_GREENLET_TIMEOUT = int(os.environ.get('GREENLET_DEFAULT_TIMEOUT', 4))

PERIODIC_TASK_INTERVAL = int(os.environ.get('PERIODIC_TASK_INTERVAL', 3 * 60))  # default: 3 minutes
PERIODIC_TASK_WAIT_TIMEOUT = int(os.environ.get('PERIODIC_TASK_WAIT_TIMEOUT', 60))  # default: 1 minute

###################################

BROWSER_CACHE_TIME = 5 * 60  # 5 minutes

###################################
# Redis
SHARED_REDIS_AUTH = os.environ.get('SHARED_REDIS_AUTH', '')
SHARED_REDIS_DB = yenv.choose_key_by_type({
    'production': 7,
    'stress': 8,
    'testing': 9,
    'prestable': 10,
}, fallback=True)

REDIS_MAX_CONNECTIONS = 100
REDIS_SOCKET_TIMEOUT = 0.5
REDIS_SOCKET_CONNECT_TIMEOUT = 0.2
REDIS_SOCKET_KEEPALIVE = True
REDIS_HEALTH_CHECK_INTERVAL = 30
REDIS_RETRY_ON_TIMEOUT = True
REDIS_CHECK_CONNECTION = True

SENTINEL_REDIS_HOSTS = yenv.choose_key_by_type({
    'production': [
        'man-767s609b2pdc92y6.db.yandex.net',
        'sas-mpuhsuco55ub9laa.db.yandex.net',
        'sas-nm00ovuitjfz4n1h.db.yandex.net',
        'vla-2u5wojk2et4ljag1.db.yandex.net',
        'vla-sl7tn6748pewgmdb.db.yandex.net',
    ],
    # todo: раскомментировать, когда запустится кластер redis-prestable
    # 'prestable': [
    #     'man-p3m8p02ye804pzy9.db.yandex.net',
    #     'sas-uxii6eguvqb0qudr.db.yandex.net',
    #     'vla-th79qx6nkt9gkvly.db.yandex.net',
    # ],
    'testing': [
        'man-166c6wuupj2n6ywo.db.yandex.net',
        'sas-186z3yuofen7iwep.db.yandex.net',
        'vla-q6y8wic2zwsad7gm.db.yandex.net',
    ],
    'stress': [
        'man-6ynpy6v97vxyvksk.db.yandex.net',
        'sas-faoemwab777aftp2.db.yandex.net',
        'vla-ddtmsu5ldlenn891.db.yandex.net',
    ],
}, fallback=True)

SENTINEL_REDIS_PORT = 26379
SENTINEL_SOCKET_TIMEOUT = 0.5

REDIS_SENTINELS = [(host, SENTINEL_REDIS_PORT) for host in SENTINEL_REDIS_HOSTS]

REDIS_CLUSTER_NAME = yenv.choose_key_by_type({
    'production': 'redis-prod',
    'testing': 'redis-test',
    'stress': 'redis-stress',
}, fallback=True)

RAW_REDIS_CACHE_CONFIGURATION = {
    'BACKEND': 'smarttv.droideka.cache_backend.base.RedisSentinelCache',
    'LOCATION': REDIS_SENTINELS,
    'TIMEOUT': 60*60*24,  # 1 day
    'OPTIONS': {
        'CLUSTER_NAME': REDIS_CLUSTER_NAME,
        'DB': SHARED_REDIS_DB,
        'PASSWORD': SHARED_REDIS_AUTH,
        'MAX_CONNECTIONS': REDIS_MAX_CONNECTIONS,
        'SOCKET_TIMEOUT': REDIS_SOCKET_TIMEOUT,
        'SOCKET_CONNECT_TIMEOUT': REDIS_SOCKET_CONNECT_TIMEOUT,
        'SOCKET_KEEPALIVE': REDIS_SOCKET_KEEPALIVE,
        'HEALTH_CHECK_INTERVAL': REDIS_HEALTH_CHECK_INTERVAL,
        'RETRY_ON_TIMEOUT': REDIS_RETRY_ON_TIMEOUT,
        'CHECK_CONNECTION': REDIS_CHECK_CONNECTION,
        'SENTINEL_SOCKET_TIMEOUT': SENTINEL_SOCKET_TIMEOUT,
        'SERIALIZER_CLASS': 'redis_cache.serializers.JSONSerializer',
    }
}

CACHES = {
    'default': dict({'VERSION': VERSION}, **RAW_REDIS_CACHE_CONFIGURATION),
    'unversioned': dict({'VERSION': 'static_version'}, **RAW_REDIS_CACHE_CONFIGURATION),
    'local': {
        'BACKEND': 'django.core.cache.backends.locmem.LocMemCache',
        'LOCATION': 'local-in-mem',
    }
}

if DEBUG:
    DEFAULT_RESPONSE_CACHE_TIME = 1
else:
    DEFAULT_RESPONSE_CACHE_TIME = 5 * 60  # 5 minutes

###################################

# see https://sandbox.yandex-team.ru/resources?author=viknet&type=GEODATA6BIN_STABLE&limit=20&created=14_days
GEODATA_BIN_PATH = os.environ.get('GEODATA_BIN_PATH', '')

###################################

TVM_CLIENT_ID = yenv.choose_key_by_type({
    'development': '2018594',
    'production': '2018021',
    'testing': '2018019'
}, fallback=True)

TVM_CLIENT_SECRET = os.environ.get('TVM_CLIENT_SECRET')

BLACKBOX_CLIENT = yenv.choose_key_by_type({
    'production': BlackboxClientId.Prod,
    'testing': BlackboxClientId.Mimino,
}, fallback=True)

TVM_ALLOWED_CLIENTS = yenv.choose_key_by_type({
    'production': [
        2002635,  # billing
        2026778,  # plant
    ],
    'testing': [
        2002641,  # billing
        2029706,  # plant
    ],
}, fallback=True)

BLACKBOX_CLIENT_ID = BLACKBOX_CLIENT.value

ETALON_PROMOCODE = os.environ.get('ETALON_PROMOCODE')

###################################

EMPTY_OBJECT_THRESHOLD = float(os.environ.get('EMPTY_OBJECT_THRESHOLD', 0.0))

# How much empty responses can we get in filterable data source
ALLOWED_EMPTY_RESPONSES = int(os.environ.get('ALLOWED_EMPTY_RESPONSES', 1))

LOG_CONTENT_ID = int(os.environ.get('LOG_CONTENT_ID', 0))

SWAGGER_SETTINGS = {
    'VALIDATOR_URL': None,
    'DEEP_LINKING': True
}

NEED_SWAGGER = yenv.choose_key_by_type({
    'production': False,
    'testing': True,
    'development': True
}, fallback=True)

COLLECTION_IMAGES_REQUIRED_MDS_HOST = os.environ.get('COLLECTION_IMAGES_REQUIRED_MDS_HOST', 'avatars.mds.yandex.net')

###################################
# YAuth

YAUTH_TYPE = 'intranet'
YAUTH_USE_NATIVE_USER = True
YAUTH_CREATE_USER_ON_ACCESS = True
YAUTH_USE_OAUTH = False
YAUTH_USE_SITES = False
YAUTH_USER_EXTRA_FIELDS = (
    ('is_staff', 'is_staff'),
)

if yenv.type in ('development', 'stress'):
    YAUTH_DEV_USER_LOGIN = 'robot-smart-tv'
    YAUTH_DEV_USER_UID = 1
    YAUTH_MECHANISMS = ['plus.utils.yauth.dev']
else:
    YAUTH_USE_TVM2_FOR_BLACKBOX = True
    YAUTH_TVM2_CLIENT_ID = TVM_CLIENT_ID
    YAUTH_TVM2_SECRET = TVM_CLIENT_SECRET
    YAUTH_TVM2_BLACKBOX_CLIENT = BlackboxClientId.ProdYateam
    YAUTH_BLACKBOX_URL = BLACKBOX_MAP[YAUTH_TVM2_BLACKBOX_CLIENT]['url']
    YAUTH_MECHANISMS = ['plus.utils.yauth.cookie']

###################################

###################################
# IDM

IDM_URL_PREFIX = f'{BASE_URL}/idm/'
IDM_TVM_CLIENT_ID = 2001600
IDM_INSTANCE = "production"
IDM_API_TVM_SETTINGS = {
    'allowed_clients': (IDM_TVM_CLIENT_ID,),
    'client_id': TVM_CLIENT_ID,
    'secret': TVM_CLIENT_SECRET,
}

###################################

USAAS_URL = 'https://conflagexp.yandex-team.ru/conflagexp/smart_tv_android'
USAAS_TIMEOUT = 0.5
USAAS_RETRIES = 1
USAAS_SERVICE = 'smart_tv_android'
USAAS_TESTIDS_FROM_PREFERENCES = yenv.choose_key_by_type({
    'production': False,
    'stress': False,
    'testing': True,
}, fallback=True)
LOG_USAAS_REQUESTS = os.environ.get('LOG_USAAS_REQUESTS', 0)


SMARTTV_OTT_SERVICE_ID = 167
YANDEX_MODULE_OTT_SERVICE_ID = os.environ.get('YANDEX_MODULE_OTT_SERVICE_ID', 234)
EMBEDDED_CAROUSEL_LIMIT = 36

DEPLOY_POD_IP_ADDRESS = os.environ.get('DEPLOY_POD_IP_0_ADDRESS') or os.environ.get('DEPLOY_POD_IP_1_ADDRESS')

####################################
# Stress testing

STRESS_ENV = yenv.choose_key_by_type({
    'production': False,
    'stress': True
}, fallback=True)

THROTTLE_EXTERNAL_API = bool(os.getenv('THROTTLE_EXTERNAL_API', STRESS_ENV))

THROTTLE_FAKE_TO_REAL_RATE = int(os.getenv('THROTTLE_FAKE_TO_REAL_RATE', 100))

####################################
# URL generation for different resolutions
DEFAULT_IMAGE_GENERATION_CONFIG = {
    'vh': [
        {'w': 120, 'h': 90},
        {'w': 400, 'h': 300},
        {'w': 360, 'h': 540},
        {'w': 1920, 'h': 1080},
        {'orig': True},
    ],
    'ott': [
        {'w': 120, 'h': 90},
        {'w': 160, 'h': 90},
        {'w': 400, 'h': 300},
        {'w': 360, 'h': 540},
        {'w': 1920, 'h': 1080},
        {'orig': True},
    ],
    'es': [
        {'w': 120, 'h': 90},
        {'w': 160, 'h': 90},
        {'w': 400, 'h': 300},
        {'w': 360, 'h': 540},
        {'w': 1920, 'h': 1080},
        {'orig': True},
    ],
    'music': [
        {'w': 400, 'h': 400},
        {'orig': True},
    ],
    'hotels': [
        {'orig': True},
    ],
    'dialogs': [
        {'orig': True},
    ],
    'cover': [
        {'w': 1344, 'h': 756},
        {'w': 1920, 'h': 1080},
        {'orig': True},
    ],
}
AVATARS_MDS_IMAGE_GENERATION_CONFIGS = os.environ.get(
    'AVATARS_MDS_IMAGE_GENERATION_CONFIGS',
    json.dumps(DEFAULT_IMAGE_GENERATION_CONFIG, ensure_ascii=False)
)

AVATARS_MDS_SUFFIX_IMAGE_GENERATION_CONFIGS = json.dumps({
    'vertical': [
        {'w': 120, 'h': 90},
        {'w': 400, 'h': 300},
        {'w': 360, 'h': 540},
        {'w': 1920, 'h': 1080},
        {'orig': True},
    ],
    'horizontal': [
        {'orig': True},
    ],
}, ensure_ascii=False)

MEMENTO_URL = yenv.choose_key_by_type({
    'testing': 'https://paskills-common-testing.alice.yandex.net',
    'production': 'http://memento.alice.yandex.net',
    'stress': 'https://paskills-common-testing.alice.yandex.net',
}, fallback=True)
MEMENTO_DEFAULT_TIMEOUT = 1
MEMENTO_DEFAULT_RETRIES = 3
MEMENTO_CLIENT_ID = yenv.choose_key_by_type({
    'production': '2021572',
    'testing': '2021570'
}, fallback=True)

# common destinations where we go with service ticket
TVM2_DESTINATIONS = (BLACKBOX_CLIENT_ID, ALICE_BUSINESS_CLIENT_ID, MEMENTO_CLIENT_ID)

# Zora settings
GO_ZORA_URL='go.zora.yandex.net'
GO_ZORA_PORT='1080'
GO_ZORA_TVM_CLIENT_ID = '2023123'
GO_ZORA_SOURCE_NAME = 'smarttv'

# Smotreshka settings
SMOTRESHKA_URL = 'https://yandextv.partner-api.lfstrm.tv'
SMOTRESHKA_DEFAULT_TIMEOUT = 1
SMOTRESHKA_DEFAULT_RETRIES = 2

SMOTRESHKA_BILLING_URL = 'https://yandextv.proxy.lfstrm.tv'
SMOTRESHKA_BILLING_DEFAULT_TIMEOUT = 3
SMOTRESHKA_BILLING_DEFAULT_RETRIES = 0

LOG_UE2E_SEARCH_REQUESTS_DC = os.environ.get('LOG_UE2E_SEARCH_REQUESTS_DC', '')
LOG_UE2E_SEARCH_SAMPLING_RATE = os.environ.get('LOG_UE2E_SEARCH_SAMPLING_RATE', '10')
