import operator
import uuid
from typing import Optional

from alice.memento.proto.user_configs_pb2 import TSmartTvMusicPromoConfig
from smarttv.droideka.proxy.constants.home_app_versions import VER_2_96, VER_2_118
from yaphone.utils.parsed_version import ParsedVersion
from yweb.webdaemons.icookiedaemon.icookie_lib.utils_py import GenerateIcookieFromUuid

YANDEX_MODULE_MANUFACTURER = 'Yandex'
YANDEX_MODULE_MODEL = 'YandexModule2-00001'
YANDEX_MODULE_QUASAR_PLATFORM = 'yandexmodule_2'


class PlatformType:
    KEY = 'platform_type'

    ANDROID = 'android'
    ANY = 'any'  # for internal usage only
    NONE = 'none'  # for internal usage only


class PlatformInfo:
    KEY_PLATFORM_VERSION = 'platform_version'
    KEY_APP_VERSION = 'app_version'
    KEY_DEVICE_MANUFACTURER = 'device_manufacturer'
    KEY_DEVICE_MODEL = 'device_model'
    KEY_QUASAR_PLATFORM = 'quasar_platform'

    class NotComparable(Exception):
        """
        Thrown when impossible to compare 2 instances of 'PlatformInfo'
        """
        pass

    def __init__(self,
                 platform_type: str = PlatformType.ANDROID,
                 platform_version: str = None,
                 app_version: str = None,
                 device_manufacturer: str = None,
                 device_model: str = None,
                 quasar_platform: str = None):
        self.platform_type = platform_type
        self.platform_version = platform_version
        self.app_version = app_version
        self.device_manufacturer = device_manufacturer
        self.device_model = device_model
        self.quasar_platform = quasar_platform

        self._fields = tuple(self.__dict__.keys())

    def is_module(self):
        return self.quasar_platform == 'yandexmodule_2'

    def is_service_headers_broken(self):
        # SMARTTVBACKEND-1169
        return VER_2_96 <= ParsedVersion(self.app_version) < VER_2_118

    def _raise_if_no_versions(self, platform_info):
        platform_versions = (platform_info.platform_version, self.platform_version)
        app_versions = (platform_info.app_version, self.app_version)
        if not all(platform_versions) and not all(app_versions):
            raise self.NotComparable('One of instances has no any version')

    def _is_constant_fields_valid(self, platform_info: 'PlatformInfo') -> bool:
        return all((
            not platform_info.device_manufacturer or platform_info.device_manufacturer == self.device_manufacturer,
            not platform_info.device_model or platform_info.device_model == self.device_model,
            not platform_info.quasar_platform or platform_info.quasar_platform == self.quasar_platform
        ))

    def _get_compare_result(
            self, platform_info: 'PlatformInfo', operator_lax, operator_strict=None):
        self._raise_if_no_versions(platform_info)
        if not operator_strict:
            operator_strict = operator_lax

        if not self._is_constant_fields_valid(platform_info):
            return False
        elif self.platform_version and self.app_version:
            self_platform_version = ParsedVersion(self.platform_version)
            self_app_version = ParsedVersion(self.app_version)

            other_platform_version = ParsedVersion(platform_info.platform_version)
            other_app_version = ParsedVersion(platform_info.app_version)
            return (not platform_info.platform_version or operator_lax(self_platform_version, other_platform_version)
                    ) and (
                not platform_info.app_version or operator_strict(self_app_version, other_app_version))
        elif not self.app_version:
            self_platform_version = ParsedVersion(self.platform_version)
            other_platform_version = ParsedVersion(platform_info.platform_version)
            return operator_strict(self_platform_version, other_platform_version) and not platform_info.app_version
        elif not self.platform_version:
            self_app_version = ParsedVersion(self.app_version)
            other_app_version = ParsedVersion(platform_info.app_version)
            return operator_strict(self_app_version, other_app_version) and not platform_info.platform_version

    def __contains__(self, platform_info: 'PlatformInfo') -> bool:
        if self.platform_type == PlatformType.NONE or self.platform_type != self.platform_type:
            return False
        elif self.platform_type == PlatformType.ANY:
            # this platform includes any platform
            return True
        elif ((self.device_manufacturer and self.device_manufacturer != platform_info.device_manufacturer) or
              (self.device_model and self.device_model != platform_info.device_model) or
              (self.quasar_platform and self.quasar_platform != platform_info.quasar_platform)):
            # manufacturer or model or quasar platform is / are invalid
            return False
        elif not any((self.platform_version, self.app_version, self.device_manufacturer,
                      self.device_model, self.quasar_platform)):
            # candidate must be on any platform with specified type
            return True
        elif not any((self.platform_version, self.app_version, self.device_model, self.quasar_platform)) and \
                self.device_manufacturer and self.device_manufacturer == platform_info.device_manufacturer:
            # candidate has only manufacturer specified, so, candidate must be on any platform with matching
            # platform_type and manufacturer
            return True
        elif not any((self.platform_version, self.app_version, self.quasar_platform)) and \
                self.device_manufacturer and self.device_manufacturer == platform_info.device_manufacturer and \
                self.device_model and self.device_model == platform_info.device_model:
            # candidate has only manufacturer and model specified, so, candidate must be on any platform with matching
            # platform_type, manufacturer and model
            return True
        elif not any((self.platform_version, self.app_version)) and \
                self.device_manufacturer and self.device_manufacturer == platform_info.device_manufacturer and \
                self.device_model and self.device_model == platform_info.device_model and \
                self.quasar_platform and self.quasar_platform == platform_info.quasar_platform:
            return True
        elif self.platform_version == platform_info.platform_version and not self.app_version:
            # candidate has valid platform type, manufacturer, model and quasar_platform
            # has no app version
            # and matched by platform version
            return True
        elif self.platform_version == platform_info.platform_version and self.app_version == platform_info.app_version:
            # match by app and platform versions
            return True
        return False

    def __eq__(self, platform_info: 'PlatformInfo') -> bool:
        return all((
            self.platform_type == platform_info.platform_type,
            ParsedVersion(self.platform_version) == ParsedVersion(platform_info.platform_version),
            ParsedVersion(self.app_version) == ParsedVersion(platform_info.app_version),
            self.device_manufacturer == platform_info.device_manufacturer,
            self.device_model == platform_info.device_model,
            self.quasar_platform == platform_info.quasar_platform
        ))

    def __ne__(self, platform_info: 'PlatformInfo') -> bool:
        return any([getattr(self, field) != getattr(platform_info, field) for field in self._fields])

    def __gt__(self, platform_info: 'PlatformInfo') -> bool:
        return self._get_compare_result(platform_info, operator.ge, operator.gt)

    def __lt__(self, platform_info: 'PlatformInfo') -> bool:
        return self._get_compare_result(platform_info, operator.le, operator.lt)

    def __ge__(self, platform_info: 'PlatformInfo') -> bool:
        return self._get_compare_result(platform_info, operator.ge)

    def __le__(self, platform_info: 'PlatformInfo') -> bool:
        return self._get_compare_result(platform_info, operator.le)

    def __str__(self) -> str:
        if not self.platform_type:
            return 'Platform(None)'
        return f'Platform({self.platform_type}, {self.platform_version}, {self.app_version}, ' \
               f'{self.device_manufacturer}, {self.device_model}, {self.quasar_platform})'


class MementoConfigs:
    music_promo_config: TSmartTvMusicPromoConfig

    def __init__(self, music_promo_config: Optional[TSmartTvMusicPromoConfig] = None):
        self.music_promo_config = music_promo_config


class RequestInfo:
    quasar_device_id: str
    is_tandem: bool

    def __init__(self,
                 authorized=False,
                 experiments=None,
                 icookie=None,
                 experiments_icookie=None,
                 ethernet_mac=None,
                 quasar_device_id: str = None,
                 memento_configs: MementoConfigs = None,
                 is_tandem=False):
        self.authorized = authorized
        self.experiments = experiments
        self.icookie = icookie
        self.experiments_icookie = experiments_icookie
        self.ethernet_mac = ethernet_mac
        self.quasar_device_id = quasar_device_id
        self.memento_configs = memento_configs
        self.is_tandem = is_tandem


def merge_dict(*dicts):
    result = {}
    for d in dicts:
        if d:
            result.update(d)
    return result


def dict_to_string(dict_to_format: dict) -> str:
    """
    Transforms dict to string, with key-value format
    For example, {'key1': 'value1', 'key2': 'value2'}
    will be transformed to 'key1=value1, key2=value2'
    """
    if not dict_to_format:
        return ''
    result = ', '.join(f'{key}={value}' for key, value in dict_to_format.items())
    return result


def get_payload_fields(obj):
    if not obj:
        return None
    return [item for item in obj.keys()]


def calculate_icookie(device_id: str) -> Optional[str]:
    if not device_id:
        return None
    try:
        uuid.UUID(device_id)
    except ValueError:
        return None
    return GenerateIcookieFromUuid(device_id).decode('ascii')


def is_only_one_true(*args):
    """
    Returns 'True' only if strictly 1 argument is not None and other are None or not presented
    """
    return sum(arg is not None for arg in args) == 1
