import logging
from functools import wraps
from typing import Union, Type

from django.core.cache import DEFAULT_CACHE_ALIAS


from cache_memoize import cache_memoize as original_cache_memoize

logging.getLogger(__name__)


def cache_memoize(
    timeout,
    prefix=None,
    args_rewrite=None,
    hit_callable=None,
    miss_callable=None,
    key_generator_callable=None,
    store_result=True,
    cache_exceptions=(),
    cache_alias=DEFAULT_CACHE_ALIAS,
    skipif=None,
):
    """
    Our enhanced cache_memoize can bypass cache call if `skipif` argument
    evaluates to True or if it's a callable returning True

    :arg int timeout: Number of seconds to store the result if not None
    :arg string prefix: If None becomes the function name.
    :arg function args_rewrite: Callable that rewrites the args first useful
    if your function needs nontrivial types but you know a simple way to
    re-represent them for the sake of the cache key.
    :arg function hit_callable: Gets executed if key was in cache.
    :arg function miss_callable: Gets executed if key was *not* in cache.
    :arg key_generator_callable: Custom cache key name generator.
    :arg bool store_result: If you know the result is not important, just
    that the cache blocked it from running repeatedly, set this to False.
    :arg Exception cache_exceptions: Accepts an Exception or a tuple of
    Exceptions. If the cached function raises any of these exceptions is the
    exception cached and raised as normal. Subsequent cached calls will
    immediately re-raise the exception and the function will not be executed.
    this tuple will be cached, all other will be propagated.
    :arg string cache_alias: The cache alias to use; defaults to 'default'.
    """
    def decorator(func):
        cache_memoized = original_cache_memoize(
            timeout=timeout,
            prefix=prefix,
            args_rewrite=args_rewrite,
            hit_callable=hit_callable,
            miss_callable=miss_callable,
            key_generator_callable=key_generator_callable,
            store_result=store_result,
            cache_exceptions=cache_exceptions,
            cache_alias=cache_alias
        )(func)

        @wraps(func)
        def inner(*args, **kwargs):
            if skipif:
                if callable(skipif):
                    skip = skipif(*args, **kwargs)
                else:
                    skip = bool(skipif)

                if skip:
                    logging.debug('Skipping cache memoize, because skipif is %s', skipif)
                    return func(*args, **kwargs)

            # call original cache memoize
            return cache_memoized(*args, **kwargs)

        return inner

    return decorator


def ignore_errors(errors_to_ignore: Union[tuple[Type[BaseException], ...], Type[BaseException]],
                  callback_function: callable = lambda: None):
    def inner(function: callable):
        def wrapper(*args, **kwargs):
            try:
                function(*args, **kwargs)
            except errors_to_ignore:
                callback_function()
        return wrapper
    return inner
