import json
import logging
import math
from tempfile import NamedTemporaryFile

from django.contrib import messages
from django.contrib.auth.decorators import permission_required
from django.http import HttpResponse, JsonResponse
from django.http.response import HttpResponseNotFound
from django.shortcuts import redirect, render
from django.urls import reverse
from django.views import View

from smarttv.plant.plant.integrations import droideka
from smarttv.plant.plant.macs import MacsImporter
from smarttv.plant.plant.models import (Assembler, Brand, Device, Factory,
                                        MacAddress, MacsImport, MacsSyncTask,
                                        Partner, Platform)
from smarttv.plant.plant.reports import DeviceReport
from smarttv.plant.plant.integrations.quasar import PlatformError, quasar_uploader


logger = logging.getLogger(__name__)


def get_import_session(import_id):
    try:
        return MacsImport.objects.get(id=import_id)
    except MacsImport.DoesNotExist:
        return None


@permission_required('plant.view_device')
def report_basic(request):
    """
    Базовый отчет об устройствах в производстве
    """
    # для выпадающих селектов фильтра
    partner_list = Partner.objects.all()
    brand_list = Brand.objects.all()
    factory_list = Factory.objects.all()
    assembler_list = Assembler.objects.all()
    platform_list = Platform.objects.all()
    context = {
        'partner_list': partner_list,
        'brand_list': brand_list,
        'factory_list': factory_list,
        'assembler_list': assembler_list,
        'platform_list': platform_list,
    }

    return render(request, 'admin/plant/report_basic.html', context)


@permission_required('plant.view_device')
def macs_import(request):
    """
    Импорт мак-адресов устройств
    """
    context = {}
    error = False
    if request.method == "POST":
        try:
            device = Device.objects.get(id=request.POST['device_id'])
        except Device.DoesNotExist:
            context['error'] = 'Устройство с таким id не найдено'
            error = True
        except ValueError:
            context['error'] = 'Неправильный id устройства'
            error = True

        if error:
            return render(request, 'admin/plant/macs_import.html', context)

        import_session = MacsImport()
        import_session.device = device
        import_session.imported_string = request.POST['macs']
        import_session.save()

        importer = MacsImporter()
        try:
            importer.run(import_session, device, request.POST['macs'])
        except MacsImporter.IncorrectMac as err:
            context['error'] = str(err)
            error = True

        if not error:
            messages.add_message(request, messages.SUCCESS, 'Macs successfuly imported')
            return redirect(reverse('admin:plant_macsimport_change', args=[import_session.id]))

    return render(request, 'admin/plant/macs_import.html', context)


@permission_required('plant.view_device')
def api_macs_sync(request, import_id):
    """
    Ручка, запускающая пуш всех маков из импорт-сессии в принимающие сервисы
    """
    import_session = get_import_session(import_id)
    if import_session is None:
        return HttpResponseNotFound(f'Macsimport session with id {import_id} was not found')
    task = MacsSyncTask(macsimport=import_session)
    task.save()

    macs = [m.macstr for m in MacAddress.objects.filter(import_session=import_session)]

    # экспорт в дройдеку
    response = droideka.client.upload_macs(macs)

    return HttpResponse(json.dumps(response))


class ApiDevices(View):
    DEFAULT_LIMIT = 20

    def get(self, request):
        if request.GET.get('format') == 'excel':
            return self.excel(request)
        return self.json(request)

    def json(self, request):
        """
        Список девайсов в джейсоне с фильтрами и пейджинацией (ajax backend)
        """
        report = DeviceReport()
        filters = self._get_filters(request)
        page, limit, start, stop = self._get_pagination(request)

        columns, rows, total = report.json(filters, start, stop)

        pageinfo = {
            'total': total,
            'page': page,
            'limit': limit,
            'pages': math.ceil(total / limit),
        }

        return JsonResponse({
            'columns': columns,
            'rows': rows,
            'pagination': pageinfo,
        })

    def excel(self, request):
        """
        Обработчик кнопки Скачать как excel
        """
        filters = self._get_filters(request)

        report = DeviceReport()
        wb = report.excel(filters)

        with NamedTemporaryFile() as f:
            wb.save(f.name)
            f.seek(0)
            stream = f.read()

        response = HttpResponse(stream, content_type='application/vnd.ms-excel')
        response['Content-Disposition'] = 'attachment; filename="report.xls"'
        return response

    def _get_filters(self, request):
        filters = {}
        keys = ['partner', 'brand', 'factory', 'assembler', 'platform']
        for key in keys:
            if request.GET.get(key):
                filters[key] = request.GET[key]

        return filters

    def _get_pagination(self, request):
        try:
            page = int(request.GET['page'])
        except (ValueError, KeyError):
            page = None

        try:
            limit = int(request.GET['limit'])
        except (ValueError, KeyError):
            limit = None

        page = page or 1
        limit = limit or self.DEFAULT_LIMIT

        # превращаем номер страницы в слайс
        start = (page - 1) * limit
        stop = start + limit

        return page, limit, start, stop


api_devices = permission_required('plant.view_device')(ApiDevices.as_view())


@permission_required('plant.view_device')
def sync_with_quasar(request, import_id):
    """
    Синхронизация мак-адресов из текущей сессии с квазаром
    """
    import_session = get_import_session(import_id)
    if import_session is None:
        return HttpResponseNotFound(f'Macsimport session with id {import_id} was not found')

    addresses = list(MacAddress.objects.filter(import_session=import_session)
                        .select_related('device', 'device__platform'))
    logger.info('Start importing %s addresses to quasar', len(addresses))

    try:
        response = json.dumps(quasar_uploader.upload_macs(addresses))
    except (ValueError, PlatformError) as error:
        return HttpResponse(f'Error: {error}')

    return HttpResponse(response)
