import logging
from typing import Mapping

import yt.wrapper as yt

from smarttv.plant.plant.integrations.quasar import settings
from smarttv.plant.plant.integrations.yt import read_table_chunked
from smarttv.plant.plant.models import MacAddress
from smarttv.utils.machelpers import quasar_device_id

yt.config.set_proxy(settings.CLUSTER)

logger = logging.getLogger(__name__)


class PlatformError(AttributeError):
    pass


class QuasarDeviceTable:
    def __init__(self):
        self.table = settings.PATH_TO_QUASAR_TABLE
        self.chunk_size = settings.CHUNK_SIZE
        self.yt = yt

    def add_devices(self, devices: list[Mapping[str, str]]) -> tuple[int, int]:
        if not devices:
            return 0, 0

        new, exising = self._find_exising(set(d['device_id'] for d in devices))

        logger.info('We have %d new and %d existing devices', len(new), len(exising))
        if len(new) > 0:
            self._write(devices, skip=exising)
        else:
            logging.info('No new devices found, nothing to write to yt')

        return len(new), len(exising)

    def _find_exising(self, ids: set[str]) -> tuple[set[str], set[str]]:
        """
        Разбивает айдишники устройств на те, которые уже есть в целевой таблице и те,
        которых там еще нет - новые

        Это нужно, потому что мы не хотим вставлять дубликаты.

        Поскольку таблица YT очень большая (на момент написания там около 250к записей),
        то мы читаем ее частями по X строк и удаляем пересечения.
        """
        new_ids = ids.copy()

        for chunk in read_table_chunked(self.table, self.chunk_size, yt_client=self.yt,
                                        columns=['device_id']):
            logger.debug(f'Got {len(chunk)} rows from yt table')
            existing = set(row['device_id'] for row in chunk)

            before = len(new_ids)
            new_ids -= existing
            after = len(new_ids)

            if before != after:
                logger.debug(f'Removed {before - after} existing ids')

            if after == 0:
                logger.debug('All ids are filtered out, stop iteration')
                break

        return new_ids, ids - new_ids

    def _write(self, devices: list[Mapping[str, str]], skip=None):
        skip = skip or []
        def generator():
            # пропускает девайсы с айдишниками из skip
            for item in devices:
                if skip and item['device_id'] in skip:
                    continue
                yield item

        yt.write_table(yt.TablePath(self.table, append=True), generator())


class QuasarUploader:
    def __init__(self, quasar_yt: QuasarDeviceTable):
        self.quasar_yt = quasar_yt

    def upload_macs(self, addresses: list[MacAddress]):
        table_rows = [convert_address(obj) for obj in addresses]

        logger.info('Adding %d devices in quasar table', len(table_rows))
        new, duplicates = self.quasar_yt.add_devices(table_rows)

        result = {
            'macs': len(table_rows),
            'new': new,
            'duplicates': duplicates,
            'table': settings.PATH_TO_QUASAR_TABLE,
        }

        return result


def convert_address(address: MacAddress) -> dict:
    if not address.device:
        raise PlatformError(f'No linked device for address {address}')

    if not address.device.platform:
        raise PlatformError('Please specify platform for device')

    if not address.device.platform.quasar_platform:
        raise PlatformError(f'Please set quasar platform name for platform {address.device.platform.name}')

    mac = address.macstr
    platform = address.device.platform.quasar_platform
    return {
        'ethernet_mac': mac,
        'platform': platform,
        'device_id': quasar_device_id(platform, mac),
    }
