import csv
import re
import logging
import datetime

from django.core.management.base import BaseCommand
from smarttv.plant.plant.models import (Assembler, Brand, Device, Factory,
                                        Panel, Partner, Platform, Order)

logger = logging.getLogger(__name__)
logger.setLevel(logging.DEBUG)


class Command(BaseCommand):
    help = 'Import orders CSV file'

    def add_arguments(self, parser):
        # positional arguments
        parser.add_argument('file_path')
        parser.add_argument('--clear', help='Delete all devices before import',
                            action='store_true')

    def handle(self, *args, **options):
        logger.info('Starting plant_import')
        importer = Importer()
        importer.import_file(options['file_path'], options['clear'])
        logger.info('plant_import is done')


class Importer:
    """
    Парсит строки из CSV файла и сохраняет их в базу
    """
    def clear_database(self):
        logger.info('Clearing database')
        Device.objects.all().delete()

    def import_file(self, file_path, clear=False):
        with open(file_path) as f:
            if clear:
                self.clear_database()
            # названия ключей берутся из первой строки
            spamreader = csv.DictReader(f, delimiter=',')
            for row in spamreader:
                if not row['Model']:
                    continue
                try:
                    self.import_row(row)
                except Exception as err:
                    logger.error('Skipping row: %s, error: %r', row, err)

    def import_row(self, row):
        partner_id = self.save_named_model(Partner, row['Partner'])
        brand_id = self.save_named_model(Brand, row['Brand'])
        factory_id = self.save_named_model(Factory, row['Factory'])
        assembler_id = self.save_named_model(Assembler, row['Assembler'])
        platform_id = self.save_named_model(Platform, row['Platform'])
        panel_id = self.save_named_model(Panel, row['Panel'])

        obj = Device()
        obj.partner_id = partner_id
        obj.brand_id = brand_id
        obj.factory_id = factory_id
        obj.assembler_id = assembler_id
        obj.platform_id = platform_id
        obj.panel_id = panel_id

        obj.model = row['Model']
        obj.screen_size = row['Screen']
        obj.resolution = row['Resolution']
        obj.firmware_ticket = row['Startrek']

        # создадим устройство, если оно еще не сохранено в базе
        device = self.save_device(obj)

        # сохраним строку как новый заказ
        self.save_order(device, row)

    def save_named_model(self, model, name):
        name = name.strip()
        if not name:
            return

        try:
            obj = model.objects.get(name=name)
        except model.DoesNotExist:
            obj = model(name=name)
            obj.save()

        return obj.id

    def save_device(self, device):
        try:
            obj = Device.objects.get(brand_id=device.brand_id, model=device.model,
                                     panel_id=device.panel_id, platform_id=device.platform_id)
            # объект не будет обновлен данными из device, возвращается
            # сохраненный в базе объект
            return obj
        except Device.DoesNotExist:
            device.save()
            return device

    def save_order(self, device, row):
        order = Order(device=device)
        order.firmware = row['MP SW']
        order.firmware_link = row['Most recent MP SW']

        if row['Amount'].strip():
            order.amount_plan = row['Amount'].strip()
        if row['Amount fact'].strip():
            order.amount_fact = row['Amount fact'].strip()
        if row['SKD date']:
            order.skd_plan = parse_date(row['SKD date'])
        if row['SKD date fact']:
            order.skd_fact = parse_date(row['SKD date fact'])
        if row['Assemble date']:
            order.assemble_plan = parse_date(row['Assemble date'])
        if row['Assemble date fact']:
            order.assemble_fact = parse_date(row['Assemble date fact'])
        order.save()
        return order


def parse_date(date_str):
    try:
        m = re.match(r'(?P<month>\d{1,2})/(?P<day>\d{1,2})/(?P<year>\d{2,4})', date_str)
        if m:
            if len(m.group('year')) == 4:
                year = int(m.group('year'))
            else:
                year = 2000 + int(m.group('year'))
            return datetime.date(year, int(m.group('month')), int(m.group('day')))
    except ValueError as err:
        logger.warning(f'Can not parse date: {date_str}: {err}')
        return
