from functools import total_ordering

from django.db import models
from django.db.models.deletion import CASCADE

from smarttv.utils.machelpers import validate_mac, normalize_mac


class Brand(models.Model):
    name = models.CharField('Бренд', max_length=256, unique=True)

    class Meta:
        verbose_name = 'Бренд'
        verbose_name_plural = 'Бренды'

    def __str__(self) -> str:
        return self.name


class Factory(models.Model):
    """
    Завод, на котором выполнятся производство узлов
    """
    name = models.CharField('Завод', max_length=256, unique=True)

    class Meta:
        verbose_name = 'Завод'
        verbose_name_plural = 'Заводы'

    def __str__(self) -> str:
        return self.name


class Assembler(models.Model):
    """
    Предприятие, осуществляющее конечную сборку
    """
    name = models.CharField('Название', max_length=256, unique=True)

    class Meta:
        verbose_name = 'Предприятие сборки'
        verbose_name_plural = 'Предприятия сборки'

    def __str__(self) -> str:
        return self.name


class Partner(models.Model):
    """
    Партнером может быть крупная ритейл-сеть, например
    Например: m-video
    """
    name = models.CharField('Партнер', max_length=256, unique=True)

    class Meta:
        verbose_name = 'Партнер'
        verbose_name_plural = 'Партнеры'

    def __str__(self) -> str:
        return self.name


class Platform(models.Model):
    """
    Платформа - это чип или набор системных компонент, которые управляют телевизором
    Ex: Hikeen rtk2871, СVTE 351
    """
    name = models.CharField('Платформа', max_length=256, unique=True)
    quasar_platform = models.CharField('Название платформы в квазаре', max_length=100, default='', blank=True)

    class Meta:
        verbose_name = 'Платформа'
        verbose_name_plural = 'Платформы'

    def __str__(self) -> str:
        return self.name


class Panel(models.Model):
    """
    ЖК-панель
    """
    name = models.CharField('Модель', max_length=256, unique=True)
    size = models.PositiveSmallIntegerField(default=0, blank=True)

    class Meta:
        verbose_name = 'Панель'
        verbose_name_plural = 'Панели'

    def __str__(self) -> str:
        return self.name


@total_ordering
class Firmware(models.Model):
    """
    Версия базовой прошивки, которую потом затачивают под каждое устройство
    """
    major = models.PositiveIntegerField(default=0, blank=True)
    minor = models.PositiveIntegerField(default=0, blank=True)
    patch = models.PositiveIntegerField(default=0, blank=True)
    extra = models.CharField('Суффикс', max_length=256, blank=True)

    class Meta:
        verbose_name = 'Версия прошивки'
        verbose_name_plural = 'Версии прошивок'

    def __eq__(self, other):
        return isinstance(other, Firmware) and self.as_tuple() == other.as_tuple()

    def __lt__(self, other):
        return self.as_tuple() < other.as_tuple()

    def __str__(self):
        version = f'{self.major}.{self.minor}.{self.patch} {self.extra}'
        return version.strip()

    def as_tuple(self):
        return self.major, self.minor, self.patch, self.extra


class DeviceGroup(models.Model):
    """
    Группа одинаковых устройств с разными именами (алиасы)
    """
    name = models.CharField('Название группы', max_length=250)

    class Meta:
        verbose_name = 'Группа устройств'
        verbose_name_plural = 'Группы устройств'

    def __str__(self) -> str:
        return self.name


class Device(models.Model):
    """
    Устройство, телевизор

    Два устройства с одинаковым брендом и моделью, но произведенные на разных заводах,
    считаются разными.
    """
    RESOLUTION_CHOICES = [
        ('HD', 'HD'),
        ('FHD', 'FHD'),
        ('QHD', 'QHD'),
        ('UHD', 'UHD'),
    ]

    visible = models.BooleanField('Опубликовано', default=False)
    partner = models.ForeignKey(Partner, verbose_name='Партнер', null=True, on_delete=models.SET_NULL, blank=True)
    brand = models.ForeignKey(Brand, verbose_name='Бренд', null=False, on_delete=CASCADE)
    model = models.CharField('Модель', max_length=256, null=False, blank=True)
    screen_size = models.IntegerField('Диагональ экрана в дюймах', null=False, default=0)
    resolution = models.CharField('Разрешение', max_length=32, null=False, choices=RESOLUTION_CHOICES)
    panel = models.ForeignKey(Panel, verbose_name='Панель', null=True, on_delete=models.SET_NULL, blank=True)
    factory = models.ForeignKey(Factory, verbose_name='Завод-изготовитель', null=True, on_delete=models.SET_NULL, blank=True)
    assembler = models.ForeignKey(Assembler, verbose_name='Предприятие сборки', null=True, on_delete=models.SET_NULL, blank=True)
    platform = models.ForeignKey(Platform, verbose_name='Платформа', null=True, on_delete=models.SET_NULL, blank=True)
    firmware_ticket = models.CharField('Ссылка на тикет прошивки', max_length=500, blank=True)
    need_attention = models.BooleanField('Требует внимания', default=False)
    note = models.TextField('Заметка (не публикуется)', blank=True)
    latest_firmware = models.ForeignKey(Firmware, verbose_name='OTA', on_delete=models.SET_NULL, blank=True,
                                        help_text='Последняя выложенная версия прошивки для этого устройства', null=True)
    group = models.ForeignKey(DeviceGroup, verbose_name='Группа', null=True, on_delete=models.SET_NULL, blank=True,
                              help_text='Устройства в одной группе одинаковые, но с разными названиями')
    created_at = models.DateTimeField(auto_now_add=True)

    class Meta:
        verbose_name = 'Устройство'
        verbose_name_plural = 'Устройства'

    def __str__(self) -> str:
        return f'{self.brand.name} {self.model} {self.panel} {self.platform}'


class Order(models.Model):
    """
    Заказ - это когда на заводе запланировали произвести какое-то
    устройство в необходимом количестве к какой-то дате
    """
    visible = models.BooleanField('Опубликован', default=False)
    device = models.ForeignKey(Device, verbose_name='Устройство', on_delete=CASCADE)
    firmware_version = models.ForeignKey(Firmware, verbose_name='Прошивка с завода', null=True, blank=True,
                                         on_delete=models.SET_NULL)
    firmware = models.CharField('Имя прошивки', max_length=250, blank=True)
    firmware_link = models.CharField('Ссылка на прошивку', max_length=1000, blank=True)

    amount_plan = models.PositiveIntegerField(default=0)
    amount_fact = models.PositiveIntegerField(default=0)
    skd_plan = models.DateField(null=True, blank=True)
    skd_fact = models.DateField(null=True, blank=True)
    assemble_plan = models.DateField(null=True, blank=True)
    assemble_fact = models.DateField(null=True, blank=True)
    need_attention = models.BooleanField('Требует внимания', default=False)
    note = models.TextField('Заметка (не публикуется)', blank=True)
    created_at = models.DateTimeField(auto_now_add=True)

    def __str__(self) -> str:
        return f'Order {self.id}'


class MacsImport(models.Model):
    """
    Сессия импорта маков через админку, нужна для учета
    """
    created_at = models.DateTimeField(auto_now_add=True)
    device = models.ForeignKey(Device, on_delete=CASCADE)
    imported_string = models.TextField(blank=True)


class MacAddress(models.Model):
    """
    Ethernet mac устройства
    """
    macstr = models.CharField('Мак-адрес (текстовая форма)', max_length=17)
    device = models.ForeignKey(Device, null=True, on_delete=CASCADE)
    import_session = models.ForeignKey(MacsImport, on_delete=CASCADE)

    def __str__(self):
        return self.macstr

    @staticmethod
    def from_string(raw_mac: str):
        obj = MacAddress()
        normalized = normalize_mac(raw_mac)

        if validate_mac(normalized):
            obj.macstr = normalized
            return obj
        else:
            raise Exception(f'Can not create mac from incorrect address: {raw_mac}')


class MacsSyncTask(models.Model):
    """
    Сессия, в рамках которой маки пушатся в дройдеку и квазар
    """
    macsimport = models.ForeignKey(MacsImport, on_delete=CASCADE)
