from django.urls import reverse
from openpyxl import Workbook

from smarttv.plant.plant.models import Device, Order


class DeviceReport:
    """
    Общий отчет по устройствам
    """
    COLUMNS_JSON = [
        'Партнер', 'Бренд', 'Модель', 'Экран', 'Разрешение', 'Завод/Cборка', 'Панель',
        'Платформа', 'План', 'Факт', 'Заказы', 'Тикет', 'Min sw', 'OTA',
    ]
    COLUMNS_EXCEL = [
        'Партнер', 'Бренд', 'Модель', 'Экран', 'Разрешение', 'Завод', 'Cборка', 'Панель',
        'Платформа', 'План', 'Факт', 'Заказы', 'Тикет', 'Min sw', 'OTA',
    ]

    def json(self, filters=None, start=None, stop=None):
        rows = []
        columns = self.COLUMNS_JSON

        query = self.basic_query()
        query = self.apply_filters(query, filters)

        # тотал считается до пейджинации
        total = query.count()

        devices = list(query[start:stop])  # run sql

        order_data = self.order_data(devices)

        for device in devices:
            row = [
                str(device.partner),
                str(device.brand),
                {'cell_type': 'link', 'text': device.model, 'link': reverse('admin:plant_device_change', args=[device.id])},
                device.screen_size,
                device.resolution,
                f'{device.factory} / {device.assembler}',
                device.panel.name if device.panel else '',
                device.platform.name if device.platform else '',
                order_data[device.id]['amount_plan'],
                order_data[device.id]['amount_fact'],
                order_data[device.id]['orders'],
            ]

            if device.firmware_ticket:
                row.append({'cell_type': 'link', 'text': 'ссылка', 'link': device.firmware_ticket})
            else:
                row.append('')

            # минимальная версия прошивки из заказов
            row.append(str(order_data[device.id]['min_sw']))

            # последнее выложенное обновление
            ota = device.latest_firmware
            row.append(str(ota) if ota else '')

            rows.append(row)

        return columns, rows, total

    def excel(self, filters=None):
        """
        Данные в формате excel без пейджинации и с немного другим набором колонок
        """
        wb = Workbook(write_only=True)
        ws = wb.create_sheet()
        ws.append(self.COLUMNS_EXCEL)

        query = self.basic_query()
        query = self.apply_filters(query, filters)
        devices = list(query)
        order_data = self.order_data(devices)

        for device in devices:
            row = [
                device.partner.name if device.partner else '',
                device.brand.name if device.brand else '',
                device.model,
                device.screen_size,
                device.resolution,
                device.factory.name if device.factory else '',
                device.assembler.name if device.assembler else '',
                device.panel.name if device.panel else '',
                device.platform.name if device.platform else '',
                order_data[device.id]['amount_plan'],
                order_data[device.id]['amount_fact'],
                order_data[device.id]['orders'],
                device.firmware_ticket,
                str(order_data[device.id]['min_sw']),
                str(device.latest_firmware) if device.latest_firmware else '',
            ]

            ws.append(row)

        return wb

    def basic_query(self):
        return Device.objects.filter().select_related(
            'partner', 'brand', 'factory', 'assembler', 'panel', 'platform', 'latest_firmware'
        ).order_by('-id')

    def apply_filters(self, query, filters):
        if filters:
            if filters.get('partner'):
                query = query.filter(partner_id=filters['partner'])
            if filters.get('brand'):
                query = query.filter(brand_id=filters['brand'])
            if filters.get('factory'):
                query = query.filter(factory_id=filters['factory'])
            if filters.get('assembler'):
                query = query.filter(assembler_id=filters['assembler'])
            if filters.get('platform'):
                query = query.filter(platform_id=filters['platform'])
        return query

    def order_data(self, devices):
        """
        Готовит словарь с вычисленными агрегатами по заказам для девайсов

        Например: {device1: {orders: 5, amount: 2000, amount_fact: 1000, etc}}
        """
        def default_item():
            return {
                'orders': 0,
                'amount_plan': 0,
                'amount_fact': 0,
                'min_sw': '',
            }

        data = {d.id: default_item() for d in devices}

        orders = Order.objects.select_related('firmware_version').filter(device_id__in=data.keys())
        for order in orders:
            data_item = data[order.device_id]
            data_item['orders'] += 1
            data_item['amount_plan'] += order.amount_plan
            data_item['amount_fact'] += order.amount_fact

            if order.firmware_version:
                if data_item['min_sw']:
                    data_item['min_sw'] = min(order.firmware_version, data_item['min_sw'])
                else:
                    data_item['min_sw'] = order.firmware_version

        return data
