"""
Django settings for plant project.
"""
import logging
import os

import yenv
from django_idm_api.settings import *  # noqa
from django_pgaas import HostManager
from django_yauth.settings import *  # noqa
from tvmauth import BlackboxTvmId as BlackboxClientId
from tvm2.protocol import BLACKBOX_BY_NAME_MAP


# Build paths inside the project like this: os.path.join(BASE_DIR, ...)
BASE_DIR = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))

DEBUG = yenv.choose_key_by_type({
    'production': False,
    'development': True,
}, fallback=True)


# SECURITY WARNING: keep the secret key used in production secret!
if DEBUG:
    os.environ.setdefault('DJANGO_SECRET_KEY', '_________')
SECRET_KEY = os.environ['DJANGO_SECRET_KEY']

ALLOWED_HOSTS = ['*']

# Application definition

INSTALLED_APPS = [
    'django_idm_api',
    'django_yauth',
    'django_pgaas',
    'django.contrib.admin',
    'django.contrib.auth',
    'django.contrib.contenttypes',
    'django.contrib.sessions',
    'django.contrib.messages',
    'django.contrib.staticfiles',
    'smarttv.plant.plant.apps.PlantConfig',
]

MIDDLEWARE = [
    'django.middleware.security.SecurityMiddleware',
    'django.contrib.sessions.middleware.SessionMiddleware',
    'django.middleware.common.CommonMiddleware',
    'django.middleware.csrf.CsrfViewMiddleware',
    'plus.utils.yauth.EmptyAuthMiddleware',
    'django_yauth.middleware.YandexAuthMiddleware',
    'django_idm_api.middleware.TVMMiddleware',
    'django.contrib.messages.middleware.MessageMiddleware',
    'django.middleware.clickjacking.XFrameOptionsMiddleware',
]

ROOT_URLCONF = 'smarttv.plant.urls'

TEMPLATES = [
    {
        'BACKEND': 'library.python.django.template.backends.arcadia.ArcadiaTemplates',
        'OPTIONS': {
            'debug': DEBUG,
            'loaders': [
                'library.python.django.template.loaders.resource.Loader',
                'library.python.django.template.loaders.app_resource.Loader',
            ],
            'context_processors': [
                'django.template.context_processors.debug',
                'django.template.context_processors.request',
                'django.contrib.auth.context_processors.auth',
                'django.contrib.messages.context_processors.messages',
            ],
        },
    },
]

FORM_RENDERER = 'library.python.django.template.backends.forms_renderer.ArcadiaRenderer'

# Database
# https://docs.djangoproject.com/en/2.2/ref/settings/#databases

DB_NAME = os.environ.get('DB_NAME', 'plant')
DB_USER = os.environ.get('DB_USER')
DB_PORT = os.environ.get('DB_PORT', '6432')
DB_PASSWORD = os.environ.get('DB_PASSWORD')

DB_HOSTS = [
    ('man-0o8li40i2qduanck.db.yandex.net', 'man'),
    ('sas-nno2bopglfyebwdb.db.yandex.net', 'sas'),
    ('vla-fn8s79gl8gg6dae5.db.yandex.net', 'vla'),
]

DATABASES = yenv.choose_key_by_type({
    'production': {
        'default': {
            'ENGINE': 'django_pgaas.backend',
            'HOST': HostManager(DB_HOSTS).host_string,
            'PORT': DB_PORT,
            'USER': DB_USER,
            'PASSWORD': DB_PASSWORD,
            'NAME': DB_NAME,
            'DISABLE_SERVER_SIDE_CURSORS': True,
            'OPTIONS': {
                'target_session_attrs': 'read-write',
                'sslmode': 'require',
            },
        },
    },
    'development': {
        'default': {
            'ENGINE': 'django.db.backends.postgresql',
            'HOST': '/var/run/postgresql/',
            'USER': DB_USER,
            'PASSWORD': DB_PASSWORD,
            'NAME': DB_NAME,
        },
    }
}, fallback=True)

# Date and Time settings
TIME_ZONE = 'Europe/Moscow'
USE_TZ = True

LANGUAGE_CODE = 'en-us'
USE_I18N = False

# setting USE_L10N=False to use DATE_FORMAT and DATETIME_FORMAT for formatting instead of local formatting
USE_L10N = False
DATE_FORMAT = 'Y-m-d'
DATETIME_FORMAT = 'Y-m-d H:i:s'

# Images, etc
# BASE_DIR inside running binary file is something not related
# real file system, so you better provide STATIC_ROOT via env vars
STATIC_ROOT = os.environ.get('STATIC_ROOT', os.path.join(BASE_DIR, 'static'))
STATIC_URL = '/plant/static/'

# static root is hard to find when you're inside bin file
# what is __file__ if you're running ./plant?
# So the only approach I see is to define path to static folder in env file

STATICFILES_FINDERS = [
    'django.contrib.staticfiles.finders.FileSystemFinder',
    'django.contrib.staticfiles.finders.AppDirectoriesFinder',
    'library.python.django.contrib.staticfiles.finders.ArcadiaAppFinder'
]


def get_log_level():
    log_level = os.getenv('LOGGING_LEVEL')
    if log_level not in ('CRITICAL', 'FATAL', 'ERROR', 'WARNING', 'WARN', 'INFO', 'DEBUG'):
        log_level = yenv.choose_key_by_type({
            'production': logging.INFO,
            'development': logging.DEBUG,
        }, fallback=True)
    return log_level

LOGGING = {
    'version': 1,
    'disable_existing_loggers': False,
    'formatters': {
        'default': {
            'format': '[%(asctime)s] [%(process)d] [%(levelname)s] [%(name)s] %(message)s',
            'datefmt': '%Y-%m-%d %H:%M:%S',
            'class': 'logging.Formatter',
        },
        'qloud_json': {
            '()': 'ylog.format.QloudJsonFormatter',
        },
    },
    'handlers': {
        'console': {
            'class': 'logging.StreamHandler',
            'formatter': 'default',
        },
        'qloud': {
            'class': 'logging.StreamHandler',
            'formatter': 'qloud_json',
        },
    },
    'loggers': {
        'smarttv.plant.plant.management': {
            'handlers': ['console'],
            'level': logging.DEBUG,
        },
        'django.db': {
            'handlers': ['console'],
            'level': logging.WARNING,
            'propagate': False,
        },
        '': {
            'level': get_log_level(),
            'handlers': yenv.choose_key_by_type({
                'production': ['qloud'],
                'development': ['console'],
            }, fallback=True),
        },
    },
}

TVM_CLIENT_SECRET = os.environ.get('TVM_CLIENT_SECRET')
TVM_CLIENT_ID = yenv.choose_key_by_type({
    'development': '2029706',
    'production': '2026778',
}, fallback=True)
TVM2_DESTINATIONS = []

BLACKBOX_CLIENT = yenv.choose_key_by_type({
    'production': BlackboxClientId.Prod,
    'testing': BlackboxClientId.Mimino,
}, fallback=True)

###################################
# YAuth
YAUTH_TYPE = 'intranet'
YAUTH_USE_NATIVE_USER = True
YAUTH_CREATE_USER_ON_ACCESS = True
YAUTH_USE_OAUTH = False
YAUTH_USE_SITES = False
YAUTH_USER_EXTRA_FIELDS = (
    ('is_staff', 'is_staff'),
)

if yenv.type in ('development', 'stress'):
    YAUTH_DEV_USER_LOGIN = 'robot-smart-tv'
    YAUTH_DEV_USER_UID = 1
    YAUTH_MECHANISMS = ['plus.utils.yauth.dev']
else:
    YAUTH_USE_TVM2_FOR_BLACKBOX = True
    YAUTH_TVM2_CLIENT_ID = TVM_CLIENT_ID
    YAUTH_TVM2_SECRET = TVM_CLIENT_SECRET
    YAUTH_TVM2_BLACKBOX_CLIENT = BlackboxClientId.ProdYateam
    YAUTH_BLACKBOX_URL = BLACKBOX_BY_NAME_MAP[YAUTH_TVM2_BLACKBOX_CLIENT.name]['url']
    YAUTH_MECHANISMS = ['plus.utils.yauth.cookie']

###################################

###################################
# IDM

IDM_URL_PREFIX = 'plant/idm/'
IDM_TVM_CLIENT_ID = 2001600
IDM_INSTANCE = "production"
IDM_API_TVM_SETTINGS = {
    'allowed_clients': (IDM_TVM_CLIENT_ID,),
    'client_id': TVM_CLIENT_ID,
    'secret': TVM_CLIENT_SECRET,
}

###################################
