import logging
import os
import time

from django.core.management.base import BaseCommand
from django.db import connection

from yandex.type_info import typing
from yt.wrapper import YtClient, TablePath, YsonFormat, ypath
from yt.wrapper.schema import TableSchema, ColumnSchema

from smarttv.report.bugreport import models

logger = logging.getLogger(__name__)


YT_TABLE_SCHEMA = TableSchema(
    columns=[
        ColumnSchema('exported_at', typing.Datetime, sort_order='ascending'),
        ColumnSchema('report_id', typing.Uint64),
        ColumnSchema('email', typing.Optional[typing.String]),
        ColumnSchema('message', typing.Optional[typing.String]),
        ColumnSchema('manufacturer', typing.Optional[typing.String]),
        ColumnSchema('product_name', typing.Optional[typing.String]),
        ColumnSchema('model', typing.Optional[typing.String]),
        ColumnSchema('file_url', typing.String),
        ColumnSchema('reported_at', typing.Datetime),
        ColumnSchema('created_at', typing.Datetime),
    ],
    strict=True,
)


def prepare_rows(last_report_id):
    export_time = int(time.time())
    reports = models.Report.objects
    if last_report_id:
        reports = reports.filter(id__gt=last_report_id)
    for report in reports.all():
        yield {
            'report_id': report.id,
            'email': report.email,
            'message': report.message,
            'manufacturer': report.manufacturer,
            'product_name': report.product_name,
            'model': report.model,
            'file_url': report.file.url,
            'reported_at': int(report.reported_at.timestamp()),
            'created_at': int(report.created_at.timestamp()),
            'exported_at': export_time,
        }


def get_last_report_id():
    with connection.cursor() as cursor:
        cursor.execute("SELECT max(id) FROM bugreport_report")
        row = cursor.fetchone()
        return row[0]


class Command(BaseCommand):
    help = 'Export bugreports to YT table'

    def add_arguments(self, parser):
        parser.add_argument('--yt-proxy', default='hahn.yt.yandex.net')
        parser.add_argument('--yt-table', default='//home/smarttv/bugreports')

    def handle(self, *args, **options):
        logger.info('Starting...')
        YT_TOKEN = os.getenv('YT_TOKEN')
        yt = YtClient(proxy=options['yt_proxy'], token=YT_TOKEN)

        yt_table = options['yt_table']

        last_report_id = None
        if yt.exists(yt_table):
            logger.info(f'Table {yt_table} exists')
            last_report_id = yt.get(ypath.ypath_join(yt_table, '@last_report_id'))
            logger.info(f'Last exported ID = {last_report_id}')
        else:
            logger.info(f'Creating table {yt_table}')
            yt.create(
                'table',
                path=yt_table,
                attributes={
                    'schema': YT_TABLE_SCHEMA,
                    'optimize_for': 'scan',
                },
            )

        logger.info('Exporting reports')
        with yt.Transaction():
            yt_path = TablePath(
                name=yt_table,
                append=True,
                sorted_by=['exported_at'],
            )
            yt.write_table(yt_path, prepare_rows(last_report_id), format=YsonFormat())
            new_last_report_id = get_last_report_id()
            yt.set(ypath.ypath_join(yt_table, '@last_report_id'), new_last_report_id)
        logger.info('Done')
