import logging
from io import BytesIO

import boto3
import botocore.config as boto_config
import botocore.exceptions as boto_exceptions
from django.conf import settings
from django.core.files.base import File
from django.core.files.storage import Storage
from django.utils.deconstruct import deconstructible
from django.utils.functional import SimpleLazyObject

logger = logging.getLogger(__name__)


class S3ConnectionError(Exception):
    pass


class AlreadyExists(Exception):
    pass


class S3Client:
    extension_name = 's3mds'
    _client = None

    @property
    def client(self):
        if self._client is None:
            self._client = self._get_client()
        return self._client

    def _get_client(self):
        try:
            session = boto3.session.Session(
                aws_access_key_id=self.access_key_id,
                aws_secret_access_key=self.access_secret_key
            )

            def inject_header(params, **kwargs):
                params['headers']['connection'] = 'Keep-Alive'
            session.events.register('before-call.s3', inject_header)

            config = boto_config.Config(
                signature_version='s3',
                max_pool_connections=settings.S3_MDS_MAX_POOL_CONNECTIONS,
                retries=settings.S3_MDS_RETRIES,
            )
            return session.client(
                service_name='s3',
                endpoint_url=f'https://{self.host}',
                config=config,
                verify=True
            )

        except Exception as exc:
            msg = 'Cannot connect to s3 server'
            logger.exception(msg)
            raise S3ConnectionError(msg) from exc

    @property
    def host(self):
        return settings.S3_MDS_HOST

    @property
    def public_host(self):
        return settings.S3_MDS_PUBLIC_HOST

    @property
    def bucket_name(self):
        return settings.S3_MDS_BUCKET_NAME

    @property
    def access_key_id(self):
        return settings.S3_MDS_ACCESS_KEY_ID

    @property
    def access_secret_key(self):
        return settings.S3_MDS_ACCESS_SECRET_KEY

    def get_url(self, key):
        return f'https://{self.bucket_name}.{self.host}/{key}'

    def get_public_url(self, key):
        return f'https://{self.bucket_name}.{self.public_host}/{key}'

    def head_object(self, key):
        return self.client.head_object(Bucket=self.bucket_name, Key=key)

    def obj_exists(self, key):
        try:
            self.head_object(key)
            return True
        except boto_exceptions.ClientError:
            return False

    def delete_object(self, key):
        self.client.delete_object(Bucket=self.bucket_name, Key=key)

    def download_fileobj(self, key):
        data = BytesIO()
        self.client.download_fileobj(self.bucket_name, key, data)
        return data

    def upload_fileobj(self, key, obj, content_type, replace=False):
        if self.obj_exists(key):
            if replace:
                self.delete_object(key)
            else:
                raise AlreadyExists(key)
        self.client.upload_fileobj(Fileobj=obj, Bucket=self.bucket_name, Key=key, ExtraArgs={'ContentType': content_type})

    def list_objects(self, prefix=''):
        # TODO: implement paging if 'IsTruncated' == True
        return self.client.list_objects_v2(Bucket=self.bucket_name, Delimiter='/', Prefix=prefix)['Contents']


s3_client = SimpleLazyObject(lambda: S3Client())


@deconstructible
class S3FileStorage(Storage):
    def _open(self, name, mode='rb'):
        return File(s3_client.download_fileobj(key=name), name)

    def _save(self, name, content):
        s3_client.upload_fileobj(key=name, obj=content.file, content_type=content.content_type, replace=True)
        return name

    def delete(self, name):
        s3_client.delete_object(key=name)

    def exists(self, name):
        return s3_client.obj_exists(key=name)

    def listdir(self, path):
        return [item['Key'] for item in s3_client.list_objects(path)]

    def size(self, name):
        obj = s3_client.head_object(name)
        return obj['ContentLength']

    def url(self, name):
        return s3_client.get_url(name)

    def get_modified_time(self, name):
        obj = s3_client.head_object(name)
        return obj['LastModified']

    def get_accessed_time(self, name):
        return self.get_modified_time(name)

    def get_created_time(self, name):
        return self.get_modified_time(name)

    def get_valid_name(self, name):
        # TODO: check if name is valid s3 Key value
        return name

    def get_available_name(self, name, max_length=None):
        return self.get_valid_name(name)


def join_path(*args):
    return '/'.join(args)
