import datetime as dt
from rest_framework.exceptions import ValidationError
from rest_framework import serializers

from yaphone.utils.parsers import parsers


def datetime_from(milliseconds):
    return dt.datetime.utcfromtimestamp(milliseconds / 1000).replace(tzinfo=dt.timezone.utc)


class DatetimeMillisValidator(serializers.IntegerField):
    def to_internal_value(self, data):
        value = super().to_internal_value(data)
        return datetime_from(value)


# noinspection PyAbstractClass
class BugreportDataValidator(serializers.Serializer):
    message = serializers.CharField(required=False)
    reported_at = DatetimeMillisValidator(required=True)
    email = serializers.CharField(required=False)
    file = serializers.FileField(required=True)


HEADER_MISSING_ERROR_TEXT = 'Header "%s" is missing'
REGEX_NOT_MATCHED_ERROR_TEXT = 'Header "%s" should match regex: "%s"'


class HeaderValidator:
    def __init__(self, header_name, required=False):
        self.header_name = header_name
        self.required = required

    def __call__(self, data):
        if self.required and self.header_name not in data:
            raise ValidationError(HEADER_MISSING_ERROR_TEXT % self.header_name)
        return data.get(self.header_name)


class RegexHeaderParser:
    def __init__(self, header_name, regex, required=False):
        self.header_name = header_name
        self.regex = regex
        self.required = required

    def __call__(self, data):
        if self.header_name not in data:
            if self.required:
                raise ValidationError(HEADER_MISSING_ERROR_TEXT % self.header_name)
            return None

        rematch = self.regex.match(data[self.header_name])
        if not rematch:
            raise ValidationError(REGEX_NOT_MATCHED_ERROR_TEXT % (self.header_name, self.regex.pattern))
        return rematch.groupdict()


class BugreportHeadersValidator:
    fields = {
        'ya_uuid': HeaderValidator('X-YaUuid'),
        'user_agent': HeaderValidator('User-Agent', required=True),
        'wifi_mac': HeaderValidator('X-Wifi-Mac'),
        'ethernet_mac': HeaderValidator('X-Ethernet-Mac'),
        'build_fingerprint': RegexHeaderParser('Build-Fingerprint', regex=parsers.BUILD_FINGERPRINT_REGEXP, required=True)
    }

    def __init__(self, data=None):
        self.initial_data = data

    def is_valid(self):
        self._validated_data = {field_name: validator(self.initial_data) for field_name, validator in self.fields.items()}

    @property
    def validated_data(self):
        if not hasattr(self, '_validated_data'):
            msg = 'You must call `.is_valid()` before accessing `.validated_data`.'
            raise AssertionError(msg)
        return self._validated_data
