import hashlib
import logging

from rest_framework.response import Response
from rest_framework.views import APIView

from smarttv.report.bugreport import models, s3mds, validators

logger = logging.getLogger(__name__)

TAG_SIZE = 8


def generate_report_key(report):
    tag = hashlib.sha1('&'.join(
        (
            report.manufacturer,
            report.product_name,
            report.model,
            report.reported_at.strftime('%Y-%m-%d %H:%M:%S'),
            report.email,
            report.file.name
        )
    ).encode('utf-8')).hexdigest()[:TAG_SIZE]

    return s3mds.join_path(
        report.reported_at.strftime('%Y-%m-%d'),
        tag,
        report.file.name
    )


class BugreportView(APIView):
    permission_classes = ()
    authentication_classes = ()
    validator_class = validators.BugreportDataValidator
    header_validator_class = validators.BugreportHeadersValidator

    def initial(self, request, *args, **kwargs):
        super().initial(request, *args, **kwargs)

    def get_validated_data(self, request):
        request_data = request.data or request.query_params
        validator = self.validator_class(data=request_data)
        validator.is_valid(raise_exception=True)
        return validator.validated_data

    def get_validated_headers(self, request):
        validator = self.header_validator_class(data=request.headers)
        validator.is_valid()
        return validator.validated_data

    def post(self, request):
        request_data = self.get_validated_data(request)
        headers_data = self.get_validated_headers(request)

        report = models.Report(
            message=request_data.get('message', ''),
            email=request_data.get('email', ''),
            reported_at=request_data['reported_at'],
            manufacturer=headers_data['build_fingerprint']['brand'],
            product_name=headers_data['build_fingerprint']['product'],
            model=headers_data['build_fingerprint']['device'],
            file=request_data['file'],
        )

        report.file.name = generate_report_key(report)
        report.save()
        return Response()
