import logging
import os

import yenv
from django_idm_api.settings import *  # noqa
from django_pgaas import HostManager
from django_yauth.settings import *  # noqa
from tvmauth import BlackboxTvmId as BlackboxClientId
from tvm2.protocol import BLACKBOX_MAP

from smarttv.utils import headers
from smarttv.utils.version import VERSION

os.environ.setdefault('REQUESTS_CA_BUNDLE', '/etc/ssl/certs')

# Build paths inside the project like this: os.path.join(BASE_DIR, ...)
BASE_DIR = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))

# SECURITY WARNING: keep the secret key used in production secret!
os.environ.setdefault('DJANGO_SECRET_KEY', '_________')
SECRET_KEY = os.environ.get('DJANGO_SECRET_KEY')

DEBUG = yenv.choose_key_by_type({
    'production': False,
    'development': True,
}, fallback=True)

ALLOWED_HOSTS = ['*']

# Application definition

INSTALLED_APPS = [
    'django_idm_api',
    'django_yauth',
    'django.contrib.admin',
    'django.contrib.auth',
    'django.contrib.contenttypes',
    'django.contrib.sessions',
    'django.contrib.messages',
    'django.contrib.staticfiles',
    'smarttv.report.bugreport.apps.ReportConfig',
    'django_pgaas',
    'rest_framework',
]

MIDDLEWARE = [
    'smarttv.utils.middleware.FixEmptyHostMiddleware',
    'smarttv.utils.middleware.LoggingContextMiddleware',
    'csp.middleware.CSPMiddleware',
    'django.middleware.security.SecurityMiddleware',
    'django.contrib.sessions.middleware.SessionMiddleware',
    'django.middleware.common.CommonMiddleware',
    # 'django.middleware.csrf.CsrfViewMiddleware',
    'plus.utils.yauth.EmptyAuthMiddleware',
    'django_idm_api.middleware.TVMMiddleware',
    'django.contrib.messages.middleware.MessageMiddleware',
    'django.middleware.clickjacking.XFrameOptionsMiddleware',
]

ROOT_URLCONF = 'smarttv.report.project.urls'

TEMPLATES = [
    {
        'BACKEND': 'library.python.django.template.backends.arcadia.ArcadiaTemplates',
        'OPTIONS': {
            'debug': DEBUG,
            'loaders': [
                'library.python.django.template.loaders.resource.Loader',
                'library.python.django.template.loaders.app_resource.Loader',
            ],
            'context_processors': [
                'django.template.context_processors.debug',
                'django.template.context_processors.request',
                'django.contrib.auth.context_processors.auth',
                'django.contrib.messages.context_processors.messages',
            ],
        },
    },
]

FORM_RENDERER = 'library.python.django.template.backends.forms_renderer.ArcadiaRenderer'

# Database
DB_HOSTS = yenv.choose_key_by_type({
    'production': [
        ('man-pt9iq3zmm0avcqsh.db.yandex.net', 'man'),
        ('sas-28cbxh0hex2i6f94.db.yandex.net', 'sas'),
        ('vla-idsdurmoyqhgh3ru.db.yandex.net', 'vla'),
    ],
    'testing': [
        ('man-alrjs014rkrhzpxj.db.yandex.net', 'man'),
        ('sas-zembq6id6fi2me5z.db.yandex.net', 'sas'),
        ('vla-yukwk4ahzmiv5wo1.db.yandex.net', 'vla'),
    ]
}, fallback=True)

DB_NAME = os.environ.get('DB_NAME')
DB_USER = os.environ.get('DB_USER')
DB_PORT = os.environ.get('DB_PORT', '6432')
DB_PASSWORD = os.environ.get('DB_PASSWORD')

host_manager = HostManager(DB_HOSTS)

DATABASES = yenv.choose_key_by_type({
    'production': {
        'default': {
            'ENGINE': 'smarttv.report.project.db_backend',
            'HOST': host_manager.host_string,
            'PORT': DB_PORT,
            'USER': DB_USER,
            'PASSWORD': DB_PASSWORD,
            'NAME': DB_NAME,
            'CONN_MAX_AGE': 0,
            'OPTIONS': {
                'MAX_CONNS': 20,
                'target_session_attrs': 'read-write',
                'sslmode': 'require',
            },
        },
    },
    'development': {
        'default': {
            'ENGINE': 'django.db.backends.postgresql',
            'HOST': 'localhost',
            'USER': DB_USER,
            'PASSWORD': DB_PASSWORD,
            'NAME': DB_NAME,
        },
    }
}, fallback=True)

###########################
# Date and Time settings
TIME_ZONE = 'Europe/Moscow'
USE_TZ = True

# USE_L10N=False to use DATE_FORMAT and DATETIME_FORMAT for formatting instead of local formatting
USE_L10N = False
DATE_FORMAT = 'Y-m-d'
DATETIME_FORMAT = 'Y-m-d H:i:s'
###########################
ADDITIONAL_LOGGING_HEADERS = {
    'device_id': headers.DEVICE_ID_HEADER,
    'uuid': headers.UUID_HEADER,
    'wifi_mac': headers.WIFI_MAC_HEADER,
    'ethernet_mac': headers.ETHERNET_MAC_HEADER,
    'build_fingerprint': headers.FINGERPRINT_HEADER,
}

UNIFIED_AGENT_LOGS_URI = os.getenv('UNIFIED_AGENT_LOGS_URI')


def get_log_level():
    log_level = os.getenv('LOGGING_LEVEL')
    if log_level not in ('CRITICAL', 'FATAL', 'ERROR', 'WARNING', 'WARN', 'INFO', 'DEBUG'):
        log_level = yenv.choose_key_by_type({
            'production': logging.INFO,
            'testing': logging.DEBUG,
            'stress': logging.INFO,
        }, fallback=True)
    return log_level


def get_log_handlers():
    handlers = {
        'console': {
            'class': 'logging.StreamHandler',
            'formatter': 'default',
        },
        'qloud': {
            'class': 'logging.StreamHandler',
            'formatter': 'json',
        },
    }

    if UNIFIED_AGENT_LOGS_URI:
        handlers['logbroker'] = {
            'class': 'logbroker.unified_agent.client.python.UnifiedAgentYdHandler',
            'logger_name': 'stdout',
            'formatter': 'json',
            'uri': UNIFIED_AGENT_LOGS_URI,
        }

    return handlers


def get_log_handlers_list():
    handlers = yenv.choose_key_by_type({
        'production': ['qloud'],
        'stress': ['qloud'],
        'development': ['console'],
    }, fallback=True)

    if UNIFIED_AGENT_LOGS_URI:
        handlers.append('logbroker')

    return handlers

LOGGING = {
    'version': 1,
    'disable_existing_loggers': False,
    'formatters': {
        'default': {
            'format': '[%(asctime)s] [%(process)d] [%(levelname)s] [%(name)s] %(message)s',
            'datefmt': '%Y-%m-%d %H:%M:%S',
            'class': 'logging.Formatter',
        },
        'raw': {
            'format': '%(message)s',
            'datefmt': '',
            'class': 'logging.Formatter',
        },
        'json': {
            '()': 'ylog.format.QloudJsonFormatter',
        },
    },
    'handlers': get_log_handlers(),
    'loggers': {
        '': {
            'level': get_log_level(),
            'handlers': get_log_handlers_list(),
            'propagate': True,
        },
        'requests': {
            'level': 'WARN'
        },
        'django.db.backends': {
            'level': yenv.choose_key_by_type({
                'production': logging.WARNING,
                'testing': logging.INFO,
                'development': logging.DEBUG,
            }, fallback=True),
        },
        'gunicorn.access': {
            'handlers': get_log_handlers_list(),
        },
        'gunicorn.error': {
            'handlers': get_log_handlers_list(),
        },
    },
}

###################################
# Sentry
UNIFIED_AGENT_ERRORS_URI = os.getenv('UNIFIED_AGENT_ERRORS_URI')

if yenv.type in ('testing', 'prestable', 'production'):
    import sentry_sdk
    from sentry_sdk.integrations.logging import LoggingIntegration
    from sentry_sdk.integrations.django import DjangoIntegration

    sentry_init_params = dict(
        debug=DEBUG,
        environment=yenv.type,
        release=VERSION,
        send_default_pii=True,
        request_bodies='small',
        integrations=[
            DjangoIntegration(),
            LoggingIntegration(
                level=logging.INFO,  # Capture info and above as breadcrumbs
                event_level=logging.WARNING,  # Send warnings/errors as events
            )
        ],
    )

    if UNIFIED_AGENT_ERRORS_URI:
        from smarttv.utils.errorbooster import get_ua_transport
        sentry_init_params.update(
            transport=get_ua_transport(
                project_name='Smarttv-Report',
                unified_agent_uri=UNIFIED_AGENT_ERRORS_URI,
            ),
        )
    else:
        sentry_init_params.update(
            dsn=os.getenv('SENTRY_DSN', ''),
        )

    logging.info('Initializing Sentry SDK')
    sentry_sdk.init(**sentry_init_params)

###########################################################

STATIC_ROOT = os.path.join(BASE_DIR, 'static')
STATIC_URL = '/report/static/'

STATICFILES_FINDERS = [
    'django.contrib.staticfiles.finders.FileSystemFinder',
    'django.contrib.staticfiles.finders.AppDirectoriesFinder',
    'library.python.django.contrib.staticfiles.finders.ArcadiaAppFinder',
]

SECURE_CONTENT_TYPE_NOSNIFF = True
SECURE_PROXY_SSL_HEADER = ('HTTP_X_FORWARDED_PROTO', 'https')

# Django REST Framework
REST_FRAMEWORK = {
    'EXCEPTION_HANDLER': 'smarttv.report.bugreport.exceptions.custom_exception_handler',
}

###################################

###################################
# S3 MDS settings

os.environ.setdefault('AWS_CA_BUNDLE', '/etc/ssl/certs/yandex-ca.pem')

S3_MDS_HOST = 's3.mds.yandex.net'
S3_MDS_PUBLIC_HOST = 's3.yandex.net'
S3_MDS_BUCKET_NAME = yenv.choose_key_by_type({
    'production': 'report-prod',
    'testing': 'report-test',
}, fallback=True)
S3_MDS_ACCESS_KEY_ID = os.environ.get('S3_MDS_ACCESS_KEY_ID')
S3_MDS_ACCESS_SECRET_KEY = os.environ.get('S3_MDS_ACCESS_SECRET_KEY')
S3_MDS_MAX_POOL_CONNECTIONS = 2
S3_MDS_RETRIES = {'max_attempts': 1}
###################################

###################################
# TVM2
TVM_CLIENT_ID = yenv.choose_key_by_type({
    'production': 2020303,
    'testing': 2020301,
    'development': 2020299,
}, fallback=True)

TVM_CLIENT_SECRET = os.environ.get('TVM_CLIENT_SECRET', '')

###################################

###################################
# YAuth

YAUTH_TYPE = 'intranet'
YAUTH_USE_NATIVE_USER = True
YAUTH_CREATE_USER_ON_ACCESS = True
YAUTH_USE_OAUTH = False
YAUTH_USE_SITES = False
YAUTH_USER_EXTRA_FIELDS = (
    ('is_staff', 'is_staff'),
)

if yenv.type == 'development':
    YAUTH_DEV_USER_LOGIN = 'robot-smart-tv'
    YAUTH_DEV_USER_UID = 1
    YAUTH_MECHANISMS = ['plus.utils.yauth.dev']
else:
    YAUTH_USE_TVM2_FOR_BLACKBOX = True
    YAUTH_TVM2_CLIENT_ID = TVM_CLIENT_ID
    YAUTH_TVM2_SECRET = TVM_CLIENT_SECRET
    YAUTH_TVM2_BLACKBOX_CLIENT = BlackboxClientId.ProdYateam
    YAUTH_BLACKBOX_URL = BLACKBOX_MAP[YAUTH_TVM2_BLACKBOX_CLIENT]['url']
    YAUTH_MECHANISMS = ['plus.utils.yauth.cookie']

###################################

###################################
# IDM

IDM_URL_PREFIX = 'report/idm/'
IDM_TVM_CLIENT_ID = 2001600
IDM_INSTANCE = "production"
IDM_API_TVM_SETTINGS = {
    'allowed_clients': (IDM_TVM_CLIENT_ID,),
    'client_id': TVM_CLIENT_ID,
    'secret': TVM_CLIENT_SECRET,
}

###################################
