# -*- coding: utf-8 -*-
import hashlib
import logging
import string
import sys
from functools import partial

import gevent
from gevent import pool
from gevent import monkey; monkey.patch_all()  # noqa

import click
import requests
import secrets
import yt.wrapper
from yt.wrapper import YtClient
from tvm2 import TVM2
from tvmauth import BlackboxTvmId as BlackboxClientId
from urllib3.util.retry import Retry
from requests.packages.urllib3.exceptions import InsecureRequestWarning

requests.packages.urllib3.disable_warnings(InsecureRequestWarning)

logging.basicConfig(stream=sys.stdout, level=logging.INFO)
logger = logging.getLogger(__name__)

RETRY_METHODS = frozenset(('GET', 'POST'))
RETRY_STATUSES = frozenset((451, 500, 502, 504))

SMOTRESHKA_REQUEST_TIMEOUT_SECONDS = 120

connection_limit = 8
thread_pool_limit = 5
gevent_pool = pool.Pool(thread_pool_limit)


output_table_schema = [
    {'name': 'quasar_device_id', 'type_v3': {'type_name': 'string'}},
    {'name': 'smotreshka_device_id', 'type_v3': {'type_name': 'string'}},
    {'name': 'smotreshka_account_id', 'type_v3': {'type_name': 'string'}},
]

password_alphabet = string.ascii_letters + string.digits


def get_requests_session():
    max_retries = 3
    retry = Retry(
        total=max_retries,
        read=max_retries,
        connect=max_retries,
        status=max_retries,
        status_forcelist=RETRY_STATUSES,
        allowed_methods=RETRY_METHODS,
        raise_on_status=True
    )
    adapter = requests.adapters.HTTPAdapter(
        max_retries=retry, pool_connections=connection_limit, pool_maxsize=connection_limit, pool_block=True)
    session = requests.session()
    session.mount('http://', adapter)
    session.mount('https://', adapter)
    return session


def generate_password(length=20):
    return ''.join(secrets.choice(password_alphabet) for i in range(length))


def get_proxies(tvm_client):
    tvm_ticket = tvm_client.get_service_ticket('2023123')
    zora_url = f'http://smarttv:{tvm_ticket}@go.zora.yandex.net:1080/'
    return {
        'http': zora_url,
        'https': zora_url,
    }


def register_account(quasar_device_id, tvm_client, requests_session):
    smotreshka_device_id = hashlib.md5(quasar_device_id.encode('ascii')).hexdigest()
    smotreshka_user_name = f'yatv_{smotreshka_device_id}'
    smotreshka_email = f'{smotreshka_user_name}@yandex-tv.users-mail.lfstm.tv'
    post_body = {
        'username': smotreshka_user_name,
        'email': smotreshka_email,
        'purchases': ['100'],
        'password': generate_password()
    }

    response = requests_session.post(
        'https://yandextv.proxy.lfstrm.tv/v2/accounts',
        json=post_body,
        headers={'Content-Type': 'application/json'},
        proxies=get_proxies(tvm_client),
        verify=False,
    )
    print(response.json(), 'elapsed', response.elapsed)
    account_id = response.json()['id']

    return {'quasar_device_id': quasar_device_id, 'smotreshka_device_id': smotreshka_device_id,
            'smotreshka_account_id': account_id}


def fill_yt_wrapper_info(yt_proxy, yt_token, mr_account, robot_name):
    yt.wrapper.config['proxy']['url'] = yt_proxy
    yt.wrapper.config['token'] = yt_token
    tmp_tables_dir = '//home/%s/%s/tmp' % (mr_account, robot_name)
    yt.wrapper.config['remote_temp_tables_directory'] = tmp_tables_dir


def get_tvm_client(client_id, client_secret):
    return TVM2(
        client_id=client_id,
        secret=client_secret,
        blackbox_client=BlackboxClientId.Prod,
        destinations=('2023123',),
    )


def register_devices(data, tvm_client, excluded_device_ids):
    threads = []
    requests_session = get_requests_session()
    for container in data:
        quasar_device_id = container['quasar_device_id']
        if quasar_device_id in excluded_device_ids:
            continue
        prepared_request = partial(
            register_account, quasar_device_id=quasar_device_id, tvm_client=tvm_client, requests_session=requests_session)
        threads.append(gevent_pool.spawn(prepared_request))
    if threads:
        gevent.joinall(threads)
    return threads


def get_registered_devices(data, tvm_client, excluded_device_ids):
    threads = register_devices(data, tvm_client, excluded_device_ids)
    registered_devices = []
    for thread in threads:
        if thread.ready() and thread.successful():
            registered_devices.append(thread.value)
    logger.info('Registered %s devices', len(registered_devices))
    return registered_devices


def get_exclude_device_ids(output_table_path):
    result = set()
    if not yt.wrapper.exists(output_table_path):
        return result
    exclude_table_iterator = yt.wrapper.read_table(output_table_path, format='json')
    for row in exclude_table_iterator:
        result.add(row['quasar_device_id'])
    return result


def create_table_if_not_exists(output_table):
    if not yt.wrapper.exists(output_table):
        yt.wrapper.create(type='table', path=output_table, attributes={'schema': output_table_schema})
        logger.info('Created output table')


def save_registered_devices_to_yt(registered_devices, output_table):
    table = yt.wrapper.ypath.TablePath(output_table, append=True)
    yt.wrapper.write_table(table=table, input_stream=registered_devices, format='json')
    logger.info('Saved %s devices', len(registered_devices))


@click.command()
@click.option('--input-table', required=True, help='Table with quasar device ids')
@click.option('--output-table', required=True, help='Table derivative ids')
@click.option('--yt-proxy', required=True, envvar='YT_PROXY', help='YT cluster proxy, envvar: YT_PROXY')
@click.option('--yt-token', required=True, envvar='YT_TOKEN', help='Yt token, envvar: YT_TOKEN')
@click.option('--mr-account', required=True, show_default=True, help='Yt quota account')
@click.option('--robot-name', required=True, help='Name of robot, who processes tables')
@click.option('--tvm-client-id', required=True, help='TVM client id')
@click.option('--tvm-client-secret', required=True, help='TVM client secret')
def main(input_table, output_table, yt_proxy, yt_token, mr_account, robot_name, tvm_client_id, tvm_client_secret):
    fill_yt_wrapper_info(yt_proxy, yt_token, mr_account, robot_name)
    client = YtClient(proxy="hahn")

    tvm_client = get_tvm_client(tvm_client_id, tvm_client_secret)
    excluded_device_ids = get_exclude_device_ids(output_table)
    data = client.read_table(input_table, format='json')

    registered_devices = get_registered_devices(data, tvm_client, excluded_device_ids)
    if registered_devices:
        create_table_if_not_exists(output_table)
        save_registered_devices_to_yt(registered_devices, output_table)
    else:
        logger.info('No registered devices to save YT')


if __name__ == '__main__':
    main()
