# -*- coding: utf-8 -*-
import click
import logging
import json
import yt.wrapper
from redis.sentinel import Sentinel
from smarttv.droideka.protos.profile.profile_pb2 import TSmotreshkaProfile

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)


sentinel_redis_hosts = {
    'production': [
        'man-767s609b2pdc92y6.db.yandex.net',
        'sas-mpuhsuco55ub9laa.db.yandex.net',
        'vla-2u5wojk2et4ljag1.db.yandex.net',
    ],
    'prestable': [
        'man-p3m8p02ye804pzy9.db.yandex.net',
        'sas-uxii6eguvqb0qudr.db.yandex.net',
        'vla-th79qx6nkt9gkvly.db.yandex.net',
    ],
    'testing': [
        'man-166c6wuupj2n6ywo.db.yandex.net',
        'sas-186z3yuofen7iwep.db.yandex.net',
        'vla-q6y8wic2zwsad7gm.db.yandex.net',
    ],
}

redis_database = {
    'production': 7,
    'testing': 9,
    'prestable': 10,
}

redis_cluster_name = {
    'production': 'redis-prod',
    'testing': 'redis-test',
}

sentinel_redis_port = 26379


def get_sentinel(environment, auth_key):
    connection_kwargs = dict(
        db=redis_database[environment],
        password=auth_key,
        max_connections=6,
        socket_timeout=2,
        socket_connect_timeout=1,
        socket_keepalive=True,
        health_check_interval=60,
        retry_on_timeout=True,
    )
    redis_sentinels = [(host, sentinel_redis_port) for host in sentinel_redis_hosts[environment]]
    sentinel_kwargs = dict(socket_timeout=2)
    return Sentinel(redis_sentinels, sentinel_kwargs=sentinel_kwargs, **connection_kwargs)


def fill_yt_wrapper_info(yt_proxy, yt_token, mr_account, robot_name):
    yt.wrapper.config['proxy']['url'] = yt_proxy
    yt.wrapper.config['token'] = yt_token
    tmp_tables_dir = '//home/%s/%s/tmp' % (mr_account, robot_name)
    yt.wrapper.config['remote_temp_tables_directory'] = tmp_tables_dir


def get_cache_key(raw_key, environment):
    return f':static_version:{environment}:smotreshka_"profile":{raw_key}'


def set_many(redis_master, data):
    if not data:
        logger.info('No data for redis')
        return
    pipe = redis_master.pipeline(transaction=False)
    for key, value in data.items():
        pipe.get(key)
    logger.info('Executing gets')
    result = pipe.execute()

    existing = 0
    new = 0
    new_values = []

    for item in zip(data.items(), result):
        key, value = item[0]
        stored_value = item[1]

        if stored_value is None:
            print(f'No value for key {key}')
            new += 1
            new_values.append((key, value))
            continue
        else:
            stored_value = stored_value.decode()

        if stored_value == value:
            existing += 1
            print('.', end='')
        else:
            print('mismatch for key %s:\n-%s\n+%s' % (key, value, stored_value))
            raise Exception('Mismatch!')

    print('')
    print('New items: %s, existing: %s' % (new, existing))

    print('Inserting new values')
    pipe = redis_master.pipeline(transaction=False)
    for key, value in new_values:
        pipe.set(key, value, ex=None)
    print('executing pipe')
    result = pipe.execute()

    inserted = sum(result)
    failed = len(result) - inserted
    print(f'Inserted items: {inserted}, failed to insert: {failed}')


def get_smotreshka_registered_devices(registered_accounts_iterator, environment):
    result = {}
    for account in registered_accounts_iterator:
        quasar_device_id = account['quasar_device_id']
        smotreshka_device_id = account['smotreshka_device_id']
        smotreshka_account_id = account['smotreshka_account_id']

        profile = TSmotreshkaProfile(SmotreshkaAccountId=smotreshka_account_id, QuasarDeviceId=quasar_device_id,
                                     SmotreshkaDeviceId=smotreshka_device_id)
        raw_profile = profile.SerializeToString().decode()
        serialized = json.dumps(raw_profile)
        result[get_cache_key(quasar_device_id, environment)] = serialized
        result[get_cache_key(smotreshka_device_id, environment)] = serialized
    return result


@click.command()
@click.option('--environment', required=True, type=click.Choice(['production', 'prestable', 'testing']), multiple=False, help='Redis environment')
@click.option('--redis-auth-key', required=True, help='Redis authentication key')
@click.option('--input-table', required=True, help='Table with quasar device ids')
@click.option('--yt-proxy', required=True, envvar='YT_PROXY', help='YT cluster proxy, envvar: YT_PROXY')
@click.option('--yt-token', required=True, envvar='YT_TOKEN', help='Yt token, envvar: YT_TOKEN')
@click.option('--mr-account', required=True, show_default=True, help='Yt quota account')
@click.option('--robot-name', required=True, help='Name of robot, who processes tables')
def main(environment, redis_auth_key, input_table, yt_proxy, yt_token, mr_account, robot_name):
    fill_yt_wrapper_info(yt_proxy, yt_token, mr_account, robot_name)
    registered_accounts_iterator = yt.wrapper.read_table(input_table, format='json')
    redis_data = get_smotreshka_registered_devices(registered_accounts_iterator, environment)
    sentinel = get_sentinel(environment, redis_auth_key)
    master = sentinel.master_for(redis_cluster_name[environment], check_connection=True)
    set_many(master, redis_data)


if __name__ == '__main__':
    main()
