import json
import time
from itertools import chain

from errorboosterclient.sentry import ErrorBoosterTransport
from logbroker.unified_agent.client.python import Client


class SmarttvEventConverter(object):
    """Преобразует данные события из формата Sentry в формат ErrorBooster."""

    TAGS_MAP = {
        # Формат source_tag: destination_tag
        'dc': 'dc',
        # Датацентр [vla, man, ...].
        'request_id': 'reqid',
        # ИД запроса requestId.
        'platform': 'platform',
        # desktop|app|tv|tvapp|station|unsupported;
        #   псевдонимы: touch - touch|touch-phone|phone; pad - pad|touch-pad
        'experiments': 'experiments',
        # Эксперименты, при которых произошло событие.
        # Формат: "<exp_description1>;<exp_description2>;..." (разбивается по ";")
        # Например: aaa=1;bb=7;ccc=yes;ddd
        'user_agent': 'useragent',
        'yandexuid': 'yandexuid',
        'path': 'page',
    }

    ADDITIONAL_KEYS = (
        # Данные события сентри, попадающие в блок additional
        'breadcrumbs',
        'contexts',
        'extra',
        'modules',
    )

    REQUEST_KEYS = (
        'data',
        'headers',
    )

    @classmethod
    def enrich(cls, source, destination, supported_keys):
        # type: (dict, dict, Iterable[str]) -> None

        if not source:
            return

        for key in supported_keys:
            value = source.get(key)
            if value:
                destination[key] = value

    @classmethod
    def convert(cls, event):
        # type: (dict) -> dict

        enrich = cls.enrich

        additional = {
            'eventid': event['event_id'],
            # уникальный идентификатор события для его адресации
        }
        enrich(source=event, destination=additional, supported_keys=cls.ADDITIONAL_KEYS)

        request = event.get('request')
        if request:
            enrich(source=request, destination=additional, supported_keys=cls.REQUEST_KEYS)

        result = {
            'timestamp': event['timestamp_'],
            # штамп возникновения события

            'level': event['level'],
            # уровень ошибки trace|debug|info|error;
            # алиасы warning - [warn|warning]; critical - [critical|fatal]

            'env': event.get('environment', ''),
            # окружение development|testing|prestable|production|pre-production

            'version': event.get('release', ''),
            # версия приложения

            'host': event['server_name'],
            # хост источника ошибки

            'language': 'python',
            # nodejs|go|python

            'source': event.get('logger', 'uncaught_exception'),
            # источник лога/ошибки

            'additional': additional,
            # произвольный набор полей вида key:string = value:string|object
            # в базе будет хранится как string:string(stringify).
            # Null-значения сюда передавать нельзя!
        }

        if request:
            result['url'] = request.get('url')

        user = event.get('user', {})
        if user:
            ip = user.get('ip_address', '')
            if ip:
                result['ip'] = ip  # адрес клиента

        tags = event.get('tags', {})
        for source_key, value in tags.items():
            if value is None:
                continue
            dest_key = cls.TAGS_MAP.get(source_key)
            if dest_key:
                result[dest_key] = value
            else:
                result['additional'][source_key] = value

        # заполнено, если ивент сгенерирован из capture_message
        message = event.get('message')

        # если ивент сгенерирован из строчки лога
        logentry = event.get('logentry')
        if logentry:
            message = logentry['message']  # строчка лога без параметров
            result['additional']['message'] = logentry['message'] % tuple(logentry['params'])

        exceptions = event.get('exception', {})

        if exceptions:
            stack = []
            # Разобранную клиентом сентри трассировку придётся вновь собрать в строку,
            # чтобы её мог переварить бустер.

            for exception in exceptions['values']:
                trace = exception['stacktrace']
                module = exception['module'] or ''

                if module:
                    module = '{module}.'.format(module=module)

                exc_message = "{module}{exception[type]}: {exception[value]}".format(
                    module=module,
                    exception=exception,
                )

                for frame in trace['frames'] or []:
                    path = frame['abs_path']
                    func = frame['function']
                    stack.append(
                        '  File "{path}", line {lineno}, in {func}\n    {line}'.format(
                            path=path,
                            lineno=frame['lineno'],
                            func=func,
                            line=str(frame['context_line'] or '').strip(),
                        )
                    )
                    result.update({'file': path, 'method': func})

            if stack:
                result['stack'] = '\n'.join(
                    chain(['Traceback (most recent call last):'], stack, [exc_message])  # noqa
                )
                if not message:
                    message = exc_message

        # текст ошибки, все сэмплы ошибок будут сгруппированы по этому названию
        result['message'] = message

        return result


def get_ua_transport(project_name, unified_agent_uri):
    client = Client(unified_agent_uri)
    session = client.create_session()

    def sender(message):
        data = json.dumps(message)
        return session.send(data, time.time())

    transport = ErrorBoosterTransport(project=project_name, sender=sender)
    transport.converter_cls = SmarttvEventConverter
    return transport
