import os

from django.conf import settings

from smarttv.utils import add_log_context, headers


class FixEmptyHostMiddleware:
    """
    Used to fix bug with response code 400 when binding to [::] and empty Host header.
    Django tries to use SERVER_NAME as host in django.http.request:HttpRequest.get_host()
    but "::" does not fit django.http.request:host_validation_re regular expression.

    It causes HTTP 400 error:
    "Invalid HTTP_HOST header: '::'. The domain name provided is not valid according to RFC 1034/1035."
    """

    def __init__(self, get_response):
        self.get_response = get_response

    def __call__(self, request):
        if 'HTTP_HOST' not in request.META and request.META.get('SERVER_NAME', '').startswith(':'):
            request.META['SERVER_NAME'] = 'localhost'

        response = self.get_response(request)
        return response


class LoggingContextMiddleware:
    DEPLOY_DC = os.getenv('DEPLOY_NODE_DC')

    DEFAULT_LOGGING_HEADERS = {
        'request_id': headers.REQUEST_ID_HEADER,
        'user_ip': headers.USER_IP_HEADER,
        'user_agent': headers.USER_AGENT_HEADER,
    }

    def __init__(self, get_response):
        self.get_response = get_response

    def __call__(self, request):
        logging_kwargs = {
            'path': request.path,
            'host': request.get_host()
        }
        if self.DEPLOY_DC:
            logging_kwargs['dc'] = self.DEPLOY_DC

        logging_headers = self.DEFAULT_LOGGING_HEADERS.copy()
        additional_logging_headers = getattr(settings, 'ADDITIONAL_LOGGING_HEADERS', None)
        if additional_logging_headers:
            logging_headers.update(additional_logging_headers)
        for parameter, header in logging_headers.items():
            header_value = request.headers.get(header)
            if header_value:
                logging_kwargs[parameter] = header_value

        add_log_context(**logging_kwargs)
        return self.get_response(request)
