import bisect
import ctypes
import functools
import multiprocessing
import time

# Histogram bins
TIMING_HGRAM_BINS = (
    0.0, 0.05, 0.1, 0.2, 0.3, 0.4, 0.5, 0.75, 1.0, 1.25, 1.5, 1.75, 2.0, 2.5, 3.0, 5.0, 7.5, 10.0
)


def path_to_signal(path):
    return path.strip('/').replace('/', '_')


class UnistatValue(object):
    # Read about sigopt postfixes
    # https://wiki.yandex-team.ru/golovan/userdocs/aggregation-types
    sigopt_postfix = ''

    def __init__(self, name):
        self.name = name

    def signal_name(self):
        return '{name}_{postfix}'.format(name=self.name, postfix=self.sigopt_postfix)

    def value(self):
        raise NotImplementedError()

    def get_metric(self):
        return self.signal_name(), self.value()


class GlobalCounter(UnistatValue):
    sigopt_postfix = 'summ'

    def __init__(self, name):
        super(GlobalCounter, self).__init__(name)
        self._counter = multiprocessing.Value(ctypes.c_longlong, 0)

    def set(self, value):
        with self._counter.get_lock():
            self._counter.value = value

    def value(self):
        return self._counter.value

    def increment(self, value=1):
        with self._counter.get_lock():
            self._counter.value += value

        return self._counter.value

    def decrement(self, value=1):
        with self._counter.get_lock():
            self._counter.value -= value

        return self._counter.value

    def __iadd__(self, other):
        return self.increment(other)

    def __isub__(self, other):
        return self.decrement(other)


class GlobalHistogram(UnistatValue):
    sigopt_postfix = 'hgram'

    def __init__(self, name, bins):
        super(GlobalHistogram, self).__init__(name)
        self._bins = bins
        self._size = len(self._bins)
        self._min = bins[0]
        self._counters = multiprocessing.Array(ctypes.c_longlong, self._size, lock=True)

    def store(self, value):
        if value > self._min:
            index = bisect.bisect_right(self._bins, value) - 1
            with self._counters.get_lock():
                self._counters[index] += 1

    def value(self):
        with self._counters.get_lock():
            data = list(self._counters)
        return list(zip(self._bins, data))


class UnistatTimer(object):
    def __init__(self, counter):
        self.counter = counter
        self._start_time = None

    def __enter__(self):
        self._start_time = time.time()
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        if not exc_val:
            total_time = time.time() - self._start_time
            self.counter.store(total_time)

    def __call__(self, func):
        @functools.wraps(func)
        def wrapped(*args, **kwargs):
            with self:
                return func(*args, **kwargs)

        return wrapped


def is_sorted(array):
    return all(array[i] <= array[i + 1] for i in range(len(array) - 1))


class UnistatManager(object):
    def __init__(self, summ_counters, hgram_counters):
        self._counters = {}

        for name in summ_counters:
            self._counters[name] = GlobalCounter(name)

        for name, bins in hgram_counters:
            assert len(bins) > 1
            assert is_sorted(bins)
            self._counters[name] = GlobalHistogram(name, bins)

    def get_counter(self, name):
        return self._counters.get(name)

    def get_timer(self, name):
        counter = self.get_counter(name)
        if counter and isinstance(counter, GlobalHistogram):
            return UnistatTimer(counter)
        else:
            return None

    def get_counter_metrics(self):
        return [counter.get_metric() for counter in self._counters.values()]
