#include "config.h"
#include "storage_http_api.h"

#include <solomon/agent/misc/labels.h>
#include <solomon/agent/misc/logger.h>
#include <solomon/agent/misc/timer_dispatcher.h>

#include <solomon/agent/lib/config/config_loader.h>
#include <solomon/agent/lib/context/context.h>
#include <solomon/agent/lib/http/server.h>
#include <solomon/agent/lib/puller/data_puller.h>
#include <solomon/agent/lib/pusher/pusher.h>
#include <solomon/agent/lib/python2/initializer.h>
#include <solomon/agent/lib/registration/registration.h>
#include <solomon/agent/lib/storage/sharded_storage.h>
#include <solomon/agent/lib/thread/pool_provider.h>

#include <solomon/agent/lib/selfmon/status/puller_status.h>
#include <solomon/agent/lib/selfmon/status/status.h>

#include <solomon/agent/lib/selfmon/service/config_page.h>
#include <solomon/agent/lib/selfmon/service/modules_page.h>
#include <solomon/agent/lib/selfmon/service/service.h>

#include <solomon/agent/modules/agent/push/push.h>
#include <solomon/agent/modules/agent/graphite/graphite.h>

#include <solomon/agent/modules/pull/http/http.h>
#include <solomon/agent/modules/pull/porto/porto.h>
#include <solomon/agent/modules/pull/python2/python2.h>
#include <solomon/agent/modules/pull/system/system.h>
#include <solomon/agent/modules/pull/unistat/unistat.h>
#include <solomon/agent/modules/pull/systemd/systemd.h>
#include <solomon/agent/modules/pull/nvidia_gpu/nvidia_gpu.h>

#include <solomon/libs/cpp/config_includes/config_includes.h>
#include <solomon/libs/cpp/signals/signals.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/svnversion/svnversion.h>

#include <util/stream/output.h>
#include <util/string/builder.h>
#include <util/string/strip.h>
#include <util/system/env.h>


using namespace NSolomon;
using namespace NAgent;

namespace {

template <class TConfig>
TLabels GetLabels(const TConfig& config) {
    TLabels labels;
    for (const TString& labelStr: config.GetLabels()) {
        labels.Add(TLabel::FromString(labelStr));
    }
    return labels;
}

TDuration GetServicePullInterval(const TServiceConfig& config) {
    TString servicePullIntervalStr = Strip(config.GetPullInterval());
    Y_ENSURE(!servicePullIntervalStr.empty(), "Missing a PullInterval field inside a service config");
    return TDuration::Parse(servicePullIntervalStr);
}

bool IsServiceConfigChanged(const TServiceConfig& lhs, const TServiceConfig& rhs) {
    if (GetServicePullInterval(lhs) != GetServicePullInterval(rhs)) {
        return true;
    }

    auto lhsLabels = GetLabels(lhs);
    auto rhsLabels = GetLabels(rhs);

    return lhsLabels != rhsLabels;
}

template <typename TPtr>
struct TPointeeEquals {
    bool operator()(const TPtr& lhs, const TPtr& rhs) const {
        return *lhs == *rhs;
    }
};

using TPullModules = THashSet<IPullModulePtr, THash<IPullModulePtr>, TPointeeEquals<IPullModulePtr>>;

class TPullModulesWithConfs {
public:
    TPullModulesWithConfs() = default;

    bool Add(IPullModulePtr module, TPullModuleConfig conf) noexcept {
        bool isNew{false};
        std::tie(std::ignore, isNew) = Modules_.emplace(module);
        if (!isNew) {
            return false;
        }

        Confs_[module] = std::move(conf);
        return true;
    }

    const TPullModuleConfig& Config(const IPullModulePtr& module) const {
        if (auto* conf = Confs_.FindPtr(module)) {
            return *conf;
        }

        ythrow yexception() << "Cannot get conf for " << module->Name() << " at " << (void*)module.Get();
    }

    void ForEach(std::function<void(IPullModulePtr, const TPullModuleConfig&)> f) const {
        for (auto&& module: Modules_) {
            auto&& conf = Confs_.at(module);
            f(module, conf);
        }
    }

    TPullModules Modules() const {
        return Modules_;
    }

private:
    TPullModules Modules_;
    THashMap<IPullModulePtr, TPullModuleConfig, THash<IPullModulePtr>, TPointeeEquals<IPullModulePtr>> Confs_;
};

} // namespace

///////////////////////////////////////////////////////////////////////////////
// TUpdateablePuller
///////////////////////////////////////////////////////////////////////////////
class TUpdateablePuller: public IServiceConfigWatcher {
public:
    explicit TUpdateablePuller(
            IShardConsumerProvider* storage,
            const TGlobalPython2Config& pyConfig,
            NMonitoring::TMetricRegistry& registry,
            IThreadPoolProvider& threadPoolProvider,
            TDuration gcDelay = TDuration::Zero())
        : PyConfig_(pyConfig)
        , Status_(registry)
        , Storage_{storage}
    {
        // TODO: pass a poolName as a parameter?
        TSimpleSharedPtr<IThreadPool> pool = threadPoolProvider.GetDefaultPool();

        DataPuller_ = CreateDataPuller(pool.Get(), &Status_, gcDelay);
        DataPuller_->Start();
    }

    ~TUpdateablePuller() {
        try {
            DataPuller_->Stop();
        } catch (...) {
        }
    }

private:
    void OnAdded(const TServiceConfig& config) override {
        auto key = ServiceConfigKey(config);
        auto modulesWithConfs = CreateModules(config);

        with_lock (ModulesLock_) {
            Modules_[key] = modulesWithConfs.Modules();
        }

        auto servicePullInterval = GetServicePullInterval(config);

        modulesWithConfs.ForEach([&] (auto&& module, auto&& moduleConfig) {
            AddModule(module, moduleConfig, config, servicePullInterval);
        });
    }

    void AddModule(
        IPullModulePtr module,
        const TPullModuleConfig& moduleConfig,
        const TServiceConfig& config,
        TDuration defaultInterval)
    {
        TDuration pullInterval;
        if (moduleConfig.GetPullInterval().empty()) {
            pullInterval = defaultInterval;
        } else {
            // pull interval is overridden for this module
            pullInterval = TDuration::Parse(moduleConfig.GetPullInterval());
        }

        IStorageConsumerProviderPtr storageConsumerProvider =
            CreateStorageConsumerProviderForShard(Storage_, config.GetProject(), config.GetService());

        const TTransformationsConfig* transformations = moduleConfig.HasTransformations()
            ? &moduleConfig.GetTransformations()
            : nullptr;

        DataPuller_->Schedule(module, pullInterval, std::move(storageConsumerProvider), transformations);
    }

    void OnRemoved(const TServiceConfig& config) override {
        auto key = ServiceConfigKey(config);

        TPullModules modules;
        with_lock (ModulesLock_) {
            auto it = Modules_.find(key);
            if (it != Modules_.end()) {
                modules = std::move(it->second);
            }
            Modules_.erase(it);
        }

        for (auto& module: modules) {
            DataPuller_->Cancel(*module);
        }
    }

    void OnChanged(
            const TServiceConfig& oldConfig,
            const TServiceConfig& newConfig) override
    {
        // if some global parameter of the service config has been changed just reload all modules
        if (IsServiceConfigChanged(oldConfig, newConfig)) {
            OnRemoved(oldConfig);
            OnAdded(newConfig);
            return;
        }

        auto key = ServiceConfigKey(oldConfig);

        auto defaultInterval = GetServicePullInterval(newConfig);
        TPullModules oldModules;
        TPullModules* serviceModules{};
        with_lock (ModulesLock_) {
            auto it = Modules_.find(key);
            if (it != Modules_.end()) {
                oldModules = it->second;
                serviceModules = &it->second;
            } else {
                OnAdded(newConfig);
                return;
            }


            auto modulesWithConfs = CreateModules(newConfig);
            modulesWithConfs.ForEach([&] (auto&& module, auto&& moduleConfig) {
                if (auto it = oldModules.find(module); it != oldModules.end()) {
                    oldModules.erase(it);
                    return;
                }

                // these are the ones that were added
                AddModule(module, moduleConfig, newConfig, defaultInterval);
                serviceModules->emplace(module);
            });

            // remove modules that are not present in the new config
            for (auto& module: oldModules) {
                DataPuller_->Cancel(*module);
                serviceModules->erase(module);
            }
        }
    }

    // TODO: replace with configurable factory
    TPullModulesWithConfs CreateModules(const TServiceConfig& config) {
        TLabels labels = GetLabels(config);

        TPullModulesWithConfs result;

        for (auto i = 0u; i < config.ModulesSize(); ++i) {
            auto&& moduleConfig = config.GetModules(i);
            IPullModulePtr module;
            switch (moduleConfig.GetTypeCase()) {
            case TPullModuleConfig::kSystem:
                module = CreateSystemModule(labels, moduleConfig.GetSystem());
                break;

            case TPullModuleConfig::kPython2:
                module = CreatePython2Module(labels, moduleConfig.GetPython2(), PyConfig_);
                break;

            case TPullModuleConfig::kSolomon:
                SA_LOG(WARN) << "Solomon pull module is deprecated. Please consider using HttpPull module instead";
                module = CreateHttpPullModule(labels, moduleConfig.GetSolomon());
                break;

            case TPullModuleConfig::kHttpPull:
                module = CreateHttpPullModule(labels, moduleConfig.GetHttpPull());
                break;

            case TPullModuleConfig::kUnistat:
                module = CreateUnistatPullModule(labels, moduleConfig.GetUnistat());
                break;

            case TPullModuleConfig::kPorto:
                module = CreatePortoPullModule(labels, moduleConfig.GetPorto());
                break;

            case TPullModuleConfig::kSystemd:
                module = CreateSystemdPullModule(labels, moduleConfig.GetSystemd());
                break;

            case TPullModuleConfig::kNvidiaGpu:
                module = CreateNvidiaGpuPullModule(labels, moduleConfig.GetNvidiaGpu());
                break;

            case TPullModuleConfig::TYPE_NOT_SET:
                ythrow yexception() << "unknown pull module type in service config "
                                    << "project: " << config.GetProject()
                                    << ", service: " << config.GetService();
            }

            bool ok = result.Add(std::move(module), moduleConfig);
            if (!ok) {
                SA_LOG(WARN) << "Module " << module->Name() << " is already present in "
                    << config.GetProject() << '/' << config.GetService() << ", skipping";
                continue;
            }
        }

        return result;
    }

public:
    const TPullerStatus& Status() const {
        return Status_;
    }

private:
    const TGlobalPython2Config& PyConfig_;
    TPullerStatus Status_;
    IShardConsumerProvider* Storage_;
    IDataPullerPtr DataPuller_;
    TAdaptiveLock ModulesLock_;
    THashMap<TString, TPullModules> Modules_;
};

class TPushModulesHolder {
public:
    explicit TPushModulesHolder(IShardConsumerProvider* writer, NMonitoring::TMetricRegistry& registry)
        : Writer_{writer}
        , Registry_{registry}
    {
    }

    void CreatePushModules(const TAgentConfig& agentConfig, IThreadPoolProvider& threadPoolProvider) {
        for (const auto& moduleConfig: agentConfig.GetModules()) {
            if (moduleConfig.HasHttpPush()) {
                const auto& pushConfig = moduleConfig.GetHttpPush();
                auto listener = CreateServerStatusListener(Registry_, {{"serverName", pushConfig.GetName()}});
                auto module = CreateHttpPushModule(pushConfig, threadPoolProvider, Writer_, listener.Get());
                PushModules_.push_back({std::move(module), std::move(listener), nullptr});
            } else if (moduleConfig.HasGraphitePush()) {
                const auto& pushConfig = moduleConfig.GetGraphitePush();
                auto listener = CreateTcpServerStatusListener(Registry_, {{"serverName", pushConfig.GetName()}});
                auto module = CreateGraphitePushModule(pushConfig, Writer_, listener.Get());
                PushModules_.push_back({std::move(module), nullptr, std::move(listener)});
            } else {
                continue;
            }

            PushModules_.back().Module->Start();
        }
    }

private:
    struct TModuleData {
        TAgentPushModulePtr Module;
        IServerStatusListenerPtr HttpStatus;
        ITcpServerStatusListenerPtr TcpStatus;
    };

private:
    IShardConsumerProvider* Writer_;
    TVector<TModuleData> PushModules_;
    NMonitoring::TMetricRegistry& Registry_;
};

///////////////////////////////////////////////////////////////////////////////
// main
///////////////////////////////////////////////////////////////////////////////
int Main(const TAgentConfig& config) {
    auto *metricRegistry = NMonitoring::TMetricRegistry::Instance();

    InitLogger(config.GetLogger(), metricRegistry);
    InitSignals();

    if (auto revision = GetArcadiaLastChange(); revision[0] != '\0') {
        metricRegistry->IntGauge({ {"sensor", "version"}, {"revision", revision} })->Set(1);
    }

    IThreadPoolProviderPtr threadPoolProvider;
    {
        TThreadPoolStatusListenerFactory listenerFactory =
            [&metricRegistry](TString poolName, NMonitoring::TLabels additionalLabels) -> IThreadPoolStatusListenerPtr {
                return CreateThreadPoolStatusListener(std::move(poolName), *metricRegistry, std::move(additionalLabels));
            };

        if (config.HasThreadPoolProvider()) {
            threadPoolProvider = CreateLazyThreadPoolProvider(config.GetThreadPoolProvider(), std::move(listenerFactory));
        } else {
            threadPoolProvider = CreateLazyThreadPoolProvider(std::move(listenerFactory));
        }
    }

    TTimerThread timerThread;

    ConstructAgentCtx(config, *metricRegistry, timerThread, threadPoolProvider);

    TRegistrationTaskPtr registrationTask;

    if (config.HasRegistration()) {
        registrationTask = MakeIntrusive<TRegistrationTask>(
                config,
                *metricRegistry,
                timerThread,
                threadPoolProvider->GetDefaultPool());

        registrationTask->Start();
    }

    NPython2::TInitializer::Instance();

    auto processStatusHolder = CreateProcessMonitoringThread(*metricRegistry);
    auto serverStatusHolder = CreateServerStatusListener(*metricRegistry, {{"serverName", "storage"}});

    TManagementServerConfig managementConf;
    TDuration pullerGcDelay;
    if (config.HasManagementServer()) {
        managementConf = config.GetManagementServer();

        const TString& gcDelayStr = managementConf.GetGcDelay();
        pullerGcDelay = gcDelayStr ? TDuration::Parse(gcDelayStr) : TDuration::Zero();
    } else {
        pullerGcDelay = TDuration::Zero();
    }


    const TStorageConfig& storageConfig = config.GetStorage();

    TTimerDispatcherPtr timerDispatcher = nullptr;
    if (storageConfig.HasAggregationOptions()) {
        const TString& aggregationIntervalStr = storageConfig.GetAggregationOptions().GetAggregationInterval();
        TDuration aggregationInterval = TDuration::Parse(aggregationIntervalStr);
        Y_ENSURE(aggregationInterval.Seconds() > 0,
                 "The smallest unit of measurement for AggregationInterval is a second");

        // TODO: specify a thread pool?
        timerDispatcher = new TTimerDispatcher{aggregationInterval};
    }

    TShardedStorage storage{storageConfig, timerDispatcher};
    auto puller = MakeIntrusive<TUpdateablePuller>(&storage, config.GetPython2(), *metricRegistry, *threadPoolProvider, pullerGcDelay);

    TConfigLoader configLoader(config.GetConfigLoader(), config.GetPython2());
    configLoader.AddWatcher(puller);
    configLoader.Start();

    TPushModulesHolder pushModules{&storage, *metricRegistry};
    pushModules.CreatePushModules(config, *threadPoolProvider);

    TLabels commonLabels = GetLabels(config);
    if (const TString hostEnv = Strip(GetEnv("SA_HOST", ""))) {
        commonLabels.Add("host", hostEnv);
    }

    THolder<NAgent::THttpServer> httpServer;
    if (config.HasHttpServer()) {
        const auto& serverConfig = config.GetHttpServer();
        httpServer.Reset(new NAgent::THttpServer(serverConfig, *threadPoolProvider, serverStatusHolder.Get()));
        httpServer->AddHandler("/version", CreateVersionHttpHandler());
        httpServer->AddHandler("/storage/shards", CreateStorageShardsHandler(&storage));
        httpServer->AddHandler("/storage/find", CreateStorageFindHandler(&storage, commonLabels));
        httpServer->AddHandler("/storage/read", CreateStorageReadHandler(&storage, commonLabels));

        TShardsConfigs shardsConfigs;
        for (const auto& shardConfig: config.GetHttpServer().GetShards()) {
            const auto& project = shardConfig.GetProject();
            const auto& service = shardConfig.GetService();

            Y_ENSURE(!project.empty(), "empty Project value");
            Y_ENSURE(!service.empty(), "empty Service value");

            TStorageShardId shardId{project, service};
            shardsConfigs[shardId] = shardConfig;
        }

        IAuthProviderPtr debugAuthProvider;
        if (auto& authMethod = serverConfig.GetDebugAuthMethod()) {
            debugAuthProvider = GetAgentCtx()->GetAuthProvider(authMethod);
            Y_ENSURE(debugAuthProvider, "no auth provider \"" << authMethod << "\"");
        }

        httpServer->AddHandler(
            "/storage/readAll",
            CreateStorageReadAllHandler(&storage, commonLabels, std::move(shardsConfigs), std::move(debugAuthProvider))
        );
        httpServer->Start();
    }

    IDataPusherPtr pusher;
    if (config.HasPush()) {
        TPushConfig pushConfig = config.GetPush();
        TSimpleSharedPtr<IThreadPool> pool = pushConfig.GetThreadPoolName()
           ? threadPoolProvider->GetThreadPool(pushConfig.GetThreadPoolName())
           : threadPoolProvider->GetDefaultPool();
        IDataPusherStatusListenerPtr statusListener = CreateDataPusherStatusListener(*metricRegistry, {});

        pusher = CreateDataPusher(
                pool,
                pushConfig,
                commonLabels,
                &storage,
                *metricRegistry,
                timerThread,
                registrationTask,
                statusListener);
        pusher->Start();
    }

    SA_LOG(INFO) << "solomon-agent started";

    THolder<TMonService> monService;

    if (config.HasManagementServer()) {
        monService = ::MakeHolder<TMonService>(managementConf, *threadPoolProvider);
        monService->AddPage(new TModuleStatePage(puller->Status()));
        monService->AddPage(new TConfigPage(config));

        monService->Start();
    } else {
        SA_LOG(INFO) << "no config for management server specified, skipping";
    }

    sigset_t blockMask;
    SigEmptySet(&blockMask);

    while (true) {
        SigSuspend(&blockMask);

        if (NeedTerminate) {
            SA_LOG(INFO) << "solomon-agent terminating";
            break;
        } else if (NeedReopenLog) {
            NeedReopenLog = 0;
            GetLogger().ReopenLog();
        }
    }

    return 0;
}

void FatalLog(const TString& message) {
    if (!GetLogger().IsNullLog()) {
        SA_LOG(FATAL) << message;
    } else {
        Cerr << message << Endl;
    }
}

int main(int argc, char* argv[]) {
    NLastGetopt::TOpts opts;
    opts.AddLongOption("config", "path to agent configuration file")
            .Optional()
            .RequiredArgument("PATH");
    opts.AddLongOption("test", "test that config is valid and exit")
            .NoArgument();
    opts.AddLongOption("example", "write config example")
            .NoArgument();
    opts.AddVersionOption();
    opts.AddHelpOption();
    opts.SetFreeArgsNum(0);

    try {
        NLastGetopt::TOptsParseResult r(&opts, argc, argv);

        if (r.Has("example")) {
            WriteConfigExample(&Cout);
            return 0;
        }

        TAgentConfig config;
        TString configPath = r.Get("config");
        if (configPath.EndsWith(".json")) {
            ParseJsonConfig(configPath, &config);
        } else {
            ParseConfig(configPath, &config);
        }
        TestConfig(config);


        MergeIncludes(config);

        if (r.Has("test")) {
            return 0;
        }

        return Main(config);
    } catch (const NLastGetopt::TUsageException& e) {
        opts.PrintUsage("solomon-agent");
    } catch (const TConfigParseException& e) {
        FatalLog(TStringBuilder() << "Cannot parse config: " << e.AsStrBuf());
    } catch (const TConfigTestException& e) {
        FatalLog(TStringBuilder() << "Config is not valid: " << e.AsStrBuf());
    } catch (...) {
        FatalLog(TStringBuilder()
                 << "Unhandled exception: "
                 << CurrentExceptionMessage()
                 << ". Process will terminate");
    }

    return 1;
}
