#include "storage_http_api.h"

#include <solomon/agent/lib/consumers/grouping_encoder.h>
#include <solomon/agent/lib/context/context.h>
#include <solomon/agent/lib/http/handler.h>
#include <solomon/agent/lib/http/headers.h>
#include <solomon/agent/lib/storage/sharded_storage.h>
#include <solomon/agent/lib/storage/storage.h>

#include <solomon/libs/cpp/multi_shard/multi_shard.h>
#include <solomon/libs/cpp/multi_shard/proto/multi_shard.pb.h>

#include <library/cpp/json/writer/json.h>
#include <library/cpp/monlib/consumers/collecting_consumer.h>
#include <library/cpp/monlib/encode/format.h>
#include <library/cpp/monlib/encode/json/json.h>
#include <library/cpp/monlib/encode/spack/spack_v1.h>
#include <library/cpp/monlib/metrics/metric_type.h>

#include <util/generic/guid.h>
#include <util/string/split.h>

#include <google/protobuf/text_format.h>

#include <array>

using NMonitoring::CompressionFromAcceptEncodingHeader;
using NMonitoring::ECompression;
using NMonitoring::EFormat;
using NMonitoring::EMetricType;
using NMonitoring::EMetricValueType;
using NMonitoring::ETimePrecision;
using NMonitoring::FormatFromAcceptHeader;

namespace NSolomon::NAgent {
namespace {

EFormat ParseFormat(const TServerRequestData& reqData) {
    const TString& formatStr = reqData.CgiParam.Get(TStringBuf("format"));
    if (!formatStr.empty()) {
        if (formatStr == TStringBuf("JSON")) {
            return EFormat::JSON;
        } else if (formatStr == TStringBuf("SPACK")) {
            return EFormat::SPACK;
        } else if (formatStr == TStringBuf("TEXT")) {
            return EFormat::TEXT;
        } else {
            ythrow THttpError(HTTP_BAD_REQUEST) << "unknown format: " << formatStr;
        }
    }
    if (TStringBuf accept = reqData.HeaderInOrEmpty("Accept")) {
        EFormat f = FormatFromAcceptHeader(accept);
        if (f != EFormat::UNKNOWN) {
            return f;
        }
    }
    return EFormat::JSON;
}

ECompression ParseCompression(const TServerRequestData& reqData) {
    if (TStringBuf acceptEncoding = reqData.HeaderInOrEmpty("Accept-Encoding")) {
        ECompression c = CompressionFromAcceptEncodingHeader(acceptEncoding);
        if (c != ECompression::UNKNOWN) {
            return c;
        }
    }
    return ECompression::IDENTITY;
}

TMaybe<TString> ParseFetcherId(const TServerRequestData& reqData) {
    if (TStringBuf fetcherId = reqData.HeaderInOrEmpty(FETCHER_ID_HEADER)) {
        return TString{fetcherId};
    }

    return Nothing();
}

TMaybe<TString> ParseClusterId(const TServerRequestData& reqData) {
    if (TStringBuf clusterId = reqData.HeaderInOrEmpty(CLUSTER_ID_HEADER)) {
        return TString{clusterId};
    }

    return Nothing();
}

TMaybe<TSeqNo> ParseStartSeqNo(const TServerRequestData& reqData) {
    if (TStringBuf startSeqNo = reqData.HeaderInOrEmpty(SEQ_NO_HEADER)) {
        ui64 result {0};
        const bool ok = TryFromString(startSeqNo, result);
        Y_ENSURE_EX(ok, THttpError(HTTP_BAD_REQUEST)
            << "x-solomon-sequencenumber header value is invalid");

        return TSeqNo{result};
    }

    return Nothing();
}

NWithMemoryInfo::TLabels ParseLabels(const TCgiParameters& params) {
    NWithMemoryInfo::TLabels labels;
    const TString& labelsStr = params.Get(TStringBuf("labels"));
    if (!labelsStr.empty()) {
        for (const auto& it: StringSplitter(labelsStr).Split(';')) {
            labels.Add(NWithMemoryInfo::TLabel::FromString(it.Token()));
        }
    }
    return labels;
}

EMatchType ParseMatchType(const TCgiParameters& params) {
    const TString& matchTypeStr = params.Get(TStringBuf("matchType"));
    if (matchTypeStr.empty()) {
        return EMatchType::EXACT;
    }
    return MatchTypeFromStr(matchTypeStr);
}

// TODO: not used after SOLOMON-5680
//ui32 ParseLimit(const TServerRequestData& reqData) {
//    const TString& limitStr = reqData.CgiParam.Get(TStringBuf("limit"));
//    ui32 limit = Max<ui32>();
//
//    if (!limitStr.empty()) {
//        TryFromString(limitStr, limit);
//    } else if (TStringBuf limitInHeader = reqData.HeaderInOrEmpty("X-Solomon-Shard-Limit")) {
//        TryFromString(limitInHeader, limit);
//    }
//
//    return limit;
//}

bool ParsePretty(const TCgiParameters& params) {
    const TString& prettyStr = params.Get(TStringBuf("pretty"));
    return prettyStr == TStringBuf("1") || prettyStr == TStringBuf("true");
}

#define Y_ENSURE_CORRECT_REQUEST(CONDITION, MESSAGE) Y_ENSURE_EX(CONDITION, THttpError(HTTP_BAD_REQUEST) << MESSAGE)

//TString ParseContinuationToken(const TBlob& requestBody, TStringBuf contentType) {
//    TMultiShardRequest fetcherReq;
//
//    bool isParsedCorrectly = false;
//    if (contentType.empty() || contentType == "application/octet-stream") {
//        isParsedCorrectly = fetcherReq.ParseFromArray(requestBody.Data(), requestBody.Size());
//    } else if (contentType == "text/plain") {
//        google::protobuf::string msg(static_cast<const char *>(requestBody.Data()), requestBody.Size());
//        isParsedCorrectly = google::protobuf::TextFormat::ParseFromString(msg, &fetcherReq);
//    } else {
//        ythrow THttpError(HTTP_BAD_REQUEST) << "unsupported Content-Type: " << contentType;
//    }
//
//    Y_ENSURE_CORRECT_REQUEST(isParsedCorrectly, "failed to parse a request");
//
//    return fetcherReq.GetContinuationToken();
//}

///////////////////////////////////////////////////////////////////////////////
// TEncoder
///////////////////////////////////////////////////////////////////////////////
class TEncoder {
public:
    explicit TEncoder(const TServerRequestData& reqData) {
        EFormat format = ParseFormat(reqData);
        switch (format) {
        case EFormat::JSON: {
            bool pretty = ParsePretty(reqData.CgiParam);
            Impl_ = NMonitoring::BufferedEncoderJson(&Stream_, pretty ? 2 : 0);
            ContentType_ = NMonitoring::NFormatContenType::JSON;
            break;
        }
        case EFormat::SPACK: {
            ECompression compression = ParseCompression(reqData);
            NMonitoring::EMetricsMergingMode mergeMetrics = NMonitoring::EMetricsMergingMode::MERGE_METRICS;

            Impl_ = EncoderSpackV1(&Stream_, ETimePrecision::SECONDS, compression, mergeMetrics);
            ContentType_ = NMonitoring::NFormatContenType::SPACK;
            break;
        }
        default:
            ythrow yexception() << "unsupported format: " << format;
        }
    }

    NMonitoring::IMetricConsumer* AsConsumer() const {
        return Impl_.Get();
    }

    TStringBuf ContentType() const {
        return ContentType_;
    }

    const TString& Content() const {
        Impl_->Close();
        return Stream_.Str();
    }

private:
    TStringStream Stream_;
    NMonitoring::IMetricEncoderPtr Impl_;
    TStringBuf ContentType_;
};

///////////////////////////////////////////////////////////////////////////////
// TStorageShardsHandler
///////////////////////////////////////////////////////////////////////////////
class TStorageShardsHandler: public IHttpHandler {
public:
    explicit TStorageShardsHandler(TShardedStorage* storage)
        : Storage_(storage)
    {
    }

private:
    void OnGet(const THttpRequest& req, THttpResponse* resp) const override {
        NJsonWriter::TBuf jsonBuf;
        jsonBuf.BeginList();

        bool pretty = ParsePretty(req.RD.CgiParam);
        if (pretty) {
            jsonBuf.SetIndentSpaces(2);
        }

        Storage_->ForEachShard(
                [&jsonBuf](const TString& project, const TString& service) {
                    jsonBuf.BeginObject();
                    jsonBuf.WriteKey(TStringBuf("project"));
                    jsonBuf.WriteString(project);
                    jsonBuf.WriteKey(TStringBuf("service"));
                    jsonBuf.WriteString(service);
                    jsonBuf.EndObject();
                });
        jsonBuf.EndList();
        resp->SetHttpCode(HttpCodes::HTTP_OK);
        resp->SetContentType(TStringBuf("application/json"));
        resp->SetContent(jsonBuf.Str());
    }

private:
    TShardedStorage* Storage_;
};

void AddCommonLabels(NMonitoring::IMetricConsumer* consumer, const TLabels& CommonLabels) {
    consumer->OnLabelsBegin();
    for (auto&& l: CommonLabels) {
        consumer->OnLabel(l.Name(), l.Value());
    }
    consumer->OnLabelsEnd();
}

///////////////////////////////////////////////////////////////////////////////
// TStorageFindHandler
///////////////////////////////////////////////////////////////////////////////
class TStorageFindHandler: public IHttpHandler {
public:
    TStorageFindHandler(TShardedStorage* storage, const TLabels& commonLabels)
        : Storage_(storage)
        , CommonLabels_(commonLabels)
    {
    }

private:
    void OnGet(const THttpRequest& req, THttpResponse* resp) const override {
        const TString& project = req.RD.CgiParam.Get(TStringBuf("project"));
        const TString& service = req.RD.CgiParam.Get(TStringBuf("service"));

        if (project.empty() || service.empty()) {
            throw THttpError(HTTP_BAD_REQUEST)
                    << "specify 'project' and 'service' parameters";
        }

        const auto fetcherId = ParseFetcherId(req.RD);

        TQuery query(ParseLabels(req.RD.CgiParam));
        query.MatchType(ParseMatchType(req.RD.CgiParam))
                // TODO: not used after SOLOMON-5680
//             .Limit(ParseLimit(req.RD))
             .ConsumerId(fetcherId);

        const auto maybeOffset = ParseStartSeqNo(req.RD);
        if (maybeOffset) {
            query.Offset(*maybeOffset);
        }

        TEncoder encoder(req.RD);
        auto result = Storage_->Find(project, service, query, encoder.AsConsumer());

        if (CommonLabels_.Size()) {
            AddCommonLabels(encoder.AsConsumer(), CommonLabels_);
        }

        resp->SetHttpCode(HttpCodes::HTTP_OK);
        resp->SetContentType(encoder.ContentType());
        resp->SetContent(encoder.Content());

        resp->AddHeader(HAS_MORE_HEADER, result.HasMore);
        resp->AddHeader(NEXT_SEQ_NO_HEADER, ToString(static_cast<ui64>(result.SeqNo)));
    }

private:
    TShardedStorage* Storage_;
    const TLabels& CommonLabels_;
};

///////////////////////////////////////////////////////////////////////////////
// TStorageReadHandler
///////////////////////////////////////////////////////////////////////////////
class TStorageReadHandler: public IHttpHandler {
public:
    TStorageReadHandler(TShardedStorage* storage, const TLabels& commonLabels)
        : Storage_(storage)
        , CommonLabels_(commonLabels)
    {
    }

private:
    void OnGet(const THttpRequest& req, THttpResponse* resp) const override {
        const TString& project = req.RD.CgiParam.Get(TStringBuf("project"));
        const TString& service = req.RD.CgiParam.Get(TStringBuf("service"));

        if (project.empty() || service.empty()) {
            throw THttpError(HTTP_BAD_REQUEST)
                    << "specify 'project' and 'service' parameters";
        }

        const auto fetcherId = ParseFetcherId(req.RD);

        TQuery query(ParseLabels(req.RD.CgiParam));
        query.MatchType(ParseMatchType(req.RD.CgiParam))
                // TODO: not used after SOLOMON-5680
//             .Limit(ParseLimit(req.RD))
             .ConsumerId(fetcherId);

        const auto maybeOffset = ParseStartSeqNo(req.RD);

        if (maybeOffset) {
            query.Offset(*maybeOffset);

            if (query.ConsumerId()) {
                Storage_->Commit(project, service, *query.ConsumerId(), *maybeOffset);
            }
        }

        TEncoder encoder(req.RD);
        auto result = Storage_->Read(project, service, query, encoder.AsConsumer());

        if (CommonLabels_.Size()) {
            AddCommonLabels(encoder.AsConsumer(), CommonLabels_);
        }

        resp->SetHttpCode(HttpCodes::HTTP_OK);
        resp->SetContentType(encoder.ContentType());
        resp->SetContent(encoder.Content());

        resp->AddHeader(HAS_MORE_HEADER, result.HasMore);
        resp->AddHeader(NEXT_SEQ_NO_HEADER, ToString(static_cast<ui64>(result.SeqNo)));
    }

private:
    TShardedStorage* Storage_;
    const TLabels& CommonLabels_;
};

using TShardKeyToMetrics = THashMap<TShardKey, TVector<NMonitoring::TMetricData>, TShardKeyHasher>;

class TEncoderCollectingByShardKey : public NMonitoring::IMetricEncoder {
public:
    TEncoderCollectingByShardKey(
            const TShardKey& shardKey,
            TShardKeyToMetrics& metricsCollection,
            TMaybe<TLabels> commonLabels = Nothing())
        : ShardKey_{shardKey}
        , MetricsCollection_{metricsCollection}
    {
        if (commonLabels) {
            CollectingConsumer_.CommonLabels = commonLabels.GetRef();
        }
    }

    void Close() override {
        auto it = MetricsCollection_.find(ShardKey_);
        if (it == MetricsCollection_.end()) {
            MetricsCollection_[ShardKey_] = std::move(CollectingConsumer_.Metrics);
        } else {
            auto& metrics = CollectingConsumer_.Metrics;
            it->second.insert(
                it->second.end(),
                std::make_move_iterator(metrics.begin()),
                std::make_move_iterator(metrics.end())
            );
        }
    }

    void OnStreamBegin() override {
        CollectingConsumer_.OnStreamBegin();
    }

    void OnStreamEnd() override {
        CollectingConsumer_.OnStreamEnd();
    }

    void OnCommonTime(TInstant time) override {
        CollectingConsumer_.OnCommonTime(time);
    }

    void OnMetricBegin(NMonitoring::EMetricType type) override {
        CollectingConsumer_.OnMetricBegin(type);
    }

    void OnMetricEnd() override {
        CollectingConsumer_.OnMetricEnd();
    }

    void OnLabelsBegin() override {
        CollectingConsumer_.OnLabelsBegin();
    }

    void OnLabelsEnd() override {
        CollectingConsumer_.OnLabelsEnd();
    }

    void OnLabel(TStringBuf name, TStringBuf value) override {
        CollectingConsumer_.OnLabel(name, value);
    }

    template <typename TValue>
    void OnPoint(TInstant time, TValue value) {
        auto& metric = CollectingConsumer_.Metrics.back();
        auto ts = ComputeTime(metric.Kind, time, CollectingConsumer_.CommonTime);

        if constexpr (std::is_same_v<TValue, double>) {
            CollectingConsumer_.OnDouble(ts, value);
        } else if constexpr  (std::is_same_v<TValue, i64>) {
            CollectingConsumer_.OnInt64(ts, value);
        } else if constexpr  (std::is_same_v<TValue, ui64>) {
            CollectingConsumer_.OnUint64(ts, value);
        } else if constexpr  (std::is_same_v<TValue, NMonitoring::IHistogramSnapshotPtr>) {
            CollectingConsumer_.OnHistogram(ts, std::move(value));
        } else if constexpr (std::is_same_v<TValue, NMonitoring::ISummaryDoubleSnapshotPtr>) {
            CollectingConsumer_.OnSummaryDouble(ts, std::move(value));
        } else if constexpr (std::is_same_v<TValue, NMonitoring::TLogHistogramSnapshotPtr>) {
            CollectingConsumer_.OnLogHistogram(ts, std::move(value));
        } else {
            static_assert(TDependentFalse<TValue>, "unknown type");
        }
    }

    void OnDouble(TInstant time, double value) override {
        OnPoint(time, value);
    }

    void OnInt64(TInstant time, i64 value) override {
        OnPoint(time, value);
    }

    void OnUint64(TInstant time, ui64 value) override {
        OnPoint(time, value);
    }

    void OnHistogram(TInstant time, NMonitoring::IHistogramSnapshotPtr snapshot) override {
        OnPoint(time, std::move(snapshot));
    }

    void OnSummaryDouble(TInstant time, NMonitoring::ISummaryDoubleSnapshotPtr snapshot) override {
        OnPoint(time, std::move(snapshot));
    }

    void OnLogHistogram(TInstant time, NMonitoring::TLogHistogramSnapshotPtr snapshot) override {
        OnPoint(time, std::move(snapshot));
    }

private:
    NMonitoring::TCollectingConsumerImpl<TAgentLabels> CollectingConsumer_{true};
    TLabels CommonLabels_;
    TShardKey ShardKey_;
    TShardKeyToMetrics& MetricsCollection_;
};

constexpr std::array<std::pair<TStringBuf, EClusterType>, 6> ConsumerIdPrefixToClusterType = {
        std::make_pair(TStringBuf("internal/pre"), EClusterType::PRESTABLE),
        std::make_pair(TStringBuf("internal/prod"), EClusterType::PRODUCTION),

        std::make_pair(TStringBuf("cloud/pre"), EClusterType::CLOUD_PREPROD),
        std::make_pair(TStringBuf("cloud/prod"), EClusterType::CLOUD_PROD),

        std::make_pair(TStringBuf("cloud/gpn"), EClusterType::CLOUD_GPN),
        std::make_pair(TStringBuf("cloud/israel"), EClusterType::CLOUD_ISRAEL),
};

EClusterType ConsumerIdToClusterType(TString consumerId) {
    EClusterType ret = EClusterType::UNKNOWN_CLUSTER;

    for (auto& [prefix, clusterType]: ConsumerIdPrefixToClusterType) {
        if (consumerId.StartsWith(prefix)) {
            ret = clusterType;
            break;
        }
    }

    return ret;
}

///////////////////////////////////////////////////////////////////////////////
// TStorageReadAllHandler
///////////////////////////////////////////////////////////////////////////////
class TStorageReadAllHandler: public IHttpHandler {
private:
    struct TShardTransformations {
        TShardKeySubstitutions Substitutions;
        bool ToPreserveOriginal;
    };

public:
    TStorageReadAllHandler(
            TShardedStorage* storage,
            const TLabels& commonLabels,
            TShardsConfigs&& shardsConfigs,
            IAuthProviderPtr debugAuthProvider)
        : Storage_(storage)
        , CommonLabels_(commonLabels)
        , DebugAuthProvider_(std::move(debugAuthProvider))
        , AgentCtx_(GetAgentCtx())
    {
        auto g = Guard(ShardsTransformationsLock_);

        for (const auto& [shardId, shardConfig]: shardsConfigs) {
            const auto& override = shardConfig.GetShardKeyOverride();

            TShardKeySubstitutions substitutions{
                override.GetProject(), override.GetCluster(), override.GetService(),
                override.GetDoNotAppendHostLabel(),
            };

            ShardsTransformations_[shardId] = MakeHolder<TShardTransformations>(TShardTransformations{
                std::move(substitutions),
                shardConfig.GetPreserveOriginal()
            });
        }
    }

private:
    using TOffsets = THashMap<TStorageShardId, TSeqNo>;
    using TOffsetsPtr = TAtomicSharedPtr<TOffsets>;

    TQuery ConstructTemplateQuery(const THttpRequest& req) const {
        TQuery query;

        auto consumerId = ParseClusterId(req.RD);
        if (!consumerId) {
            consumerId = ParseFetcherId(req.RD);
        }

        query.ConsumerId(consumerId);

        return query;
    }

    void OnPost(const THttpRequest& req, THttpResponse* resp) override {
        using namespace NMultiShard;

        // Make it const to prevent its usage inside a shard's request (where a separate query should be constructed)
        const TQuery query = ConstructTemplateQuery(req);

        // TODO: not used after SOLOMON-5680
//        i64 limit = ParseLimit(req.RD);
        auto format = ParseFormat(req.RD);
        TStringBuf contentType;

        switch (format) {
            case EFormat::JSON:
                contentType = JSON_CONTENT_TYPE;
                break;
            case EFormat::SPACK:
                contentType = SPACK_CONTENT_TYPE;
                break;
            case EFormat::TEXT:
                contentType = TEXT_CONTENT_TYPE;
                break;
            default:
                ythrow THttpError(HTTP_BAD_REQUEST) << "unknown data format was requested: " << format;
        }

        TShardKeyToMetrics shardKeyToMetrics;
        TString multiShardData;
        TStringOutput outputStream(multiShardData);
        auto multiShardEncoder = CreateMultiShardEncoder(format, outputStream);

        TVector<TStorageShardId> shardIds;
        Storage_->ForEachShard([&shardIds](const TString& project, const TString& service) {
            shardIds.emplace_back(TStorageShardId{project, service});
        });

        bool hasMore = false;

        multiShardEncoder->SetHeader({"", EFormatVersion::V1});

        for (const TStorageShardId& shardId: shardIds) {
            // TODO: not used after SOLOMON-5680
//            if (limit <= 0) {
//                break;
//            }

            auto shardQuery{query};
            // TODO: not used after SOLOMON-5680
//            shardQuery.Limit(limit);

            if (query.ConsumerId() && !query.ConsumerId().Empty()) {
                // will shift an offset value to dirty
                auto signalForResetting = TSeqNo{0};
                Storage_->Commit(shardId.Project, shardId.Service, *shardQuery.ConsumerId(), signalForResetting);
            }

            TReadResult res;
            auto g = Guard(ShardsTransformationsLock_);

            if (auto it = ShardsTransformations_.find(shardId); it != ShardsTransformations_.end()) {
                TGroupingEncoder encoderGeneratingNewShards{
                    it->second->Substitutions,
                    [&shardKeyToMetrics](const TShardKey& shardKey, auto&&) {
                        return MakeHolder<TEncoderCollectingByShardKey>(shardKey, shardKeyToMetrics);
                    },
                    CommonLabels_
                };

                if (it->second->ToPreserveOriginal) {
                    TShardKey shardKey{shardId.Project, GetAgentCtx()->GetCluster(), shardId.Service};
                    TEncoderCollectingByShardKey e{shardKey, shardKeyToMetrics, CommonLabels_};

                    res = Storage_->Read(shardId.Project, shardId.Service, shardQuery, &e);
                    e.Close();

                    hasMore |= res.HasMore;
//                    limit -= res.NumOfMetrics;
                }

                res = Storage_->Read(shardId.Project, shardId.Service, shardQuery, &encoderGeneratingNewShards);
                static_cast<NMonitoring::IMetricEncoder*>(&encoderGeneratingNewShards)->Close();
            } else {
                TShardKey shardKey{shardId.Project, GetAgentCtx()->GetCluster(), shardId.Service};
                TEncoderCollectingByShardKey e{shardKey, shardKeyToMetrics, CommonLabels_};

                res = Storage_->Read(shardId.Project, shardId.Service, shardQuery, &e);
                e.Close();
            }

            hasMore |= res.HasMore;
//            limit -= res.NumOfMetrics;
        }

        for (auto& [shardKey, metrics]: shardKeyToMetrics) {
            multiShardEncoder->OnShardBegin(shardKey.Project, shardKey.Service, shardKey.Cluster);

            multiShardEncoder->OnStreamBegin();
            for (auto& metric: metrics) {
                multiShardEncoder->OnMetricBegin(metric.Kind);

                multiShardEncoder->OnLabelsBegin();
                for (auto& l: metric.Labels) {
                    multiShardEncoder->OnLabel(l.Name(), l.Value());
                }
                multiShardEncoder->OnLabelsEnd();

                for (size_t i = 0; i != metric.Values->Size(); ++i) {
                    auto ts = (*metric.Values)[i].GetTime();
                    auto value = (*metric.Values)[i].GetValue();

                    switch (metric.Kind) {
                        case EMetricType::RATE:
                        case EMetricType::COUNTER:
                            multiShardEncoder->OnUint64(ts, value.AsUint64());
                            break;
                        case EMetricType::GAUGE:
                            multiShardEncoder->OnDouble(ts, value.AsDouble());
                            break;
                        case EMetricType::IGAUGE:
                            multiShardEncoder->OnInt64(ts, value.AsInt64());
                            break;
                        case EMetricType::HIST:
                        case EMetricType::HIST_RATE:
                            multiShardEncoder->OnHistogram(ts, value.AsHistogram());
                            break;
                        case EMetricType::DSUMMARY:
                            multiShardEncoder->OnSummaryDouble(ts, value.AsSummaryDouble());
                            break;
                        case EMetricType::LOGHIST:
                            multiShardEncoder->OnLogHistogram(ts, value.AsLogHistogram());
                            break;
                        case EMetricType::UNKNOWN:
                            ythrow yexception() << "unknown metric type for metric " << metric.Labels;
                    }
                }

                multiShardEncoder->OnMetricEnd();
            }

            multiShardEncoder->OnStreamEnd();
            multiShardEncoder->OnShardEnd();
        }

        multiShardEncoder->Close();

        resp->SetHttpCode(HttpCodes::HTTP_OK);
        resp->SetContentType(contentType);
        resp->SetContent(multiShardData);

        resp->AddHeader(HAS_MORE_HEADER, hasMore);

        IAuthProviderPtr authProvider;
        if (DebugAuthProvider_) {
            authProvider = DebugAuthProvider_;
        } else {
            if (query.ConsumerId() && !query.ConsumerId().Empty()) {
                auto clusterType = ConsumerIdToClusterType(*query.ConsumerId());
                if (clusterType != EClusterType::UNKNOWN_CLUSTER) {
                    authProvider = GetAgentCtx()->GetAuthProvider(clusterType);
                }
            }
        };

        if (authProvider) {
            THashMap<TString, TString> headers;
            authProvider->AddCredentials(headers);

            for (auto& [key, value]: headers) {
                resp->AddHeader(key, value);
            }
        }
    }

private:
    TShardedStorage* Storage_;
    const TLabels& CommonLabels_;

    TAdaptiveLock ShardsTransformationsLock_;
    THashMap<TStorageShardId, TAtomicSharedPtr<TShardTransformations>> ShardsTransformations_;

    IAuthProviderPtr DebugAuthProvider_;
    const IAgentContext* AgentCtx_;
};

} // namespace

IHttpHandlerPtr CreateStorageShardsHandler(TShardedStorage* storage) {
    return MakeHolder<TStorageShardsHandler>(storage);
}

IHttpHandlerPtr CreateStorageFindHandler(TShardedStorage* storage, const TLabels& commonLabels) {
    return MakeHolder<TStorageFindHandler>(storage, commonLabels);
}

IHttpHandlerPtr CreateStorageReadHandler(TShardedStorage* storage, const TLabels& commonLabels) {
    return MakeHolder<TStorageReadHandler>(storage, commonLabels);
}

IHttpHandlerPtr CreateStorageReadAllHandler(
        TShardedStorage* storage,
        const TLabels& commonLabels,
        TShardsConfigs&& shardsConfigs,
        IAuthProviderPtr debugAuthProvider)
{
    return MakeHolder<TStorageReadAllHandler>(
            storage,
            commonLabels,
            std::move(shardsConfigs),
            std::move(debugAuthProvider));
}

} // namespace NSolomon::NAgent

