import time
import random


"""
In order for Solomon Agent to collect data from Python module, a Python
class or several with a pull() method have to be defined within a file.
Methods declarations for such a class are listed below.

The Python Solomon library is used. For API details see:
https://wiki.yandex-team.ru/solomon/libs/monlib_python/
"""


class MyAwesomePullModule:
    """
    A class with a pull() method. Created only once during module's lifetime
    """

    def __init__(self, logger, registry):
        """
        In this constructor, logger and registry objects are passed down
        from Solomon Agent's Python interpreter. One can store them for later
        use or interact with them as they wish.

        :param logger: An object for writing info into Solomon Agent's logs
        :param registry: An object through which metric objects are created
        """

        # save it once for later use
        self._logger = logger

        # construct metric objects which will store metric values
        # with different types. metrics constructed through the registry object
        # will be automatically consumed. see pull() below for details
        self._reqs = registry.counter({'metric': 'requests_count'})
        self._users = registry.gauge({'metric': 'users_count'})
        self._rx = registry.rate({'metric': 'bytes_rx'})

        # there are two types of histograms: rate and counters; they can be
        # created using histogram_rate or histogram_counter methods respectively.
        #
        # both methods have following signature: histogram_*(labels: dict, type: str, **kwargs)
        #
        # labels is a dict used to identify the metric object
        #
        # type is a string determining the way histogram buckets are defined. Valid values are:
        # 'explicit', 'linear' and 'exponential'. kwargs will differ for every type. See below


        # 'explicit': buckets -- required; list of bucket bounds
        # (-INF, 0], (0, 1], (1, 2], (2, 5], (5, 10], (10, 20], (20, +INF)
        self._execution_time_hist_rate = registry.histogram_rate(
            {'metric': 'execTimeRate'},
            'explicit',
            buckets=[0, 1, 2, 5, 10, 20])

        # 'linear': bucket_count -- required; defines the number of buckets
        # start_value -- required; upper bound of the first bucket
        # bucket_width -- required; the difference between the upper and lower bounds for each bucket
        # (-INF, 5], (5, 20], (20, 35], (35, 50], (50, 65], (65, +INF)
        self._execution_time_hist_counter = registry.histogram_counter(
            {'metric': 'execTimeCounter'},
            'linear',
            bucket_count=6, start_value=5, bucket_width=15)

        # 'exponential': bucket_count -- required; defines the number of buckets
        # base -- required; the exponential growth factor for the buckets width
        # scale -- optional (default: 1.); the linear scale for the buckets
        # (-INF, 3], (3, 6], (6, 12], (12, 24], (24, 48], (48, +INF)
        self._execution_time_hist_counter2 = registry.histogram_rate(
            {'metric': 'execTimeCounter2'},
            'exponential',
            base=2, bucket_count=6, scale=3.)

    def pull(self, ts, consumer):
        """
        A method for data collection. Will be triggered every PullInterval seconds.
        ts and consumer are passed down from Solomon Agent

        :param ts: Timestamp with a current time value
        :type ts: int

        :param consumer: An object through which metrics are sent to Agent
        :type consumer: object

        :returns: Result of execution: 0 or None -- everything is fine.
        If non-zero, a module will halt without being executed ever again.
        (Until the list of modules returned by ConfigLoader is changed)
        :rtype: Optional[int]
        """

        # Available methods are: https://nda.ya.ru/t/BOi9UZIV3W6JWP
        self._logger.debug("pull() is triggered")

        # (1) metrics from registry will be automatically consumed
        self._reqs.inc()
        self._users.set(10)
        self._rx.add(20)

        self._execution_time_hist_rate.collect(int(random.random() * 100))
        self._execution_time_hist_counter.collect(int(random.random() * 100))

        # second argument to `collect` is the number of points with the specified value
        # we want to add to the histogram
        self._execution_time_hist_counter2.collect(int(random.random() * 100), 42)

        # (2) dynamic metrics with a single value. constructed for every pull request
        consumer.counter({'my': 'counter1'}, ts, 1)
        consumer.gauge({'my': 'gauge1'}, ts, 2.3)
        # RATE metrics should not have ts, so pass 0 as a ts value
        consumer.rate({'my': 'rate1'}, 0, 4)

        # (3) dynamic metrics with multiple values
        consumer.counter({'my': 'counter2'}, [
            time.time() - 1, 1,
            time.time()    , 2,
            time.time() + 1, 3,
        ])
        consumer.gauge({'my': 'gauge2'}, [
            time.time() - 1, 2.3,
            time.time()    , 4.5,
            time.time() + 1, 6.7,
        ])

        # histograms can be written using a list of bounds and a list of corresponding values
        # NB: bucket bounds must be sorted
        hist_buckets = [0, 1000, 1000000]
        # size of the value list may be equal to size of bound list or greater than it by one
        # in the latter case last value in the value list will indicate the number of values
        # falling between the upper bound and +inf
        hist_values = [1, 2, 3, 5]

        # e.g. (-inf; 0]: 1; (0; 1000]: 2; (1000; 1000000]: 3; (1000000; +inf): 5
        consumer.histogram_rate({'my': 'hist_rate'}, ts, hist_buckets, hist_values)

        # if list sizes are equal, than we'll put 0 into the last bucket
        # e.g. (-inf; 0]: 1; (0; 10]: 3; (10; +inf): 0;
        consumer.histogram_counter({'my': 'hist_couner'}, ts, [0, 10], [1, 3])

##
## one file can contain several modules
##
class MyPoorPullModule:
    def __init__(self, logger, registry):
        self._counter = 0
        self._logger = logger

    def pull(self, ts, consumer):
        consumer.counter({'self': 'counter'}, ts, self._counter)
        self._counter += 1
        self._logger.info("<< poor module [%d] >>" % self._counter)
