#pragma once

#include <solomon/agent/misc/timer_thread.h>
#include <solomon/agent/protos/auth.pb.h>

#include <solomon/libs/cpp/cloud/iam/client.h>
#include <solomon/libs/cpp/error_or/error_or.h>

#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/monlib/push_client/auth.h>

#include <util/thread/pool.h>


namespace NSolomon::NAgent {

/**
 * Creates an auth provider which auto-updates TVM tokens
 *
 * @param tvmConfig
 * @return an auth provider inflating http headers with a TVM token
 */
NSolomon::IAuthProviderPtr CreateTvmAuthProvider(const TvmConfig& tvmConfig);

/**
 * Creates an auth provider with a fixed OAuth value
 *
 * @param conf
 * @return an auth provider inflating http headers with an OAuth token
 */
NSolomon::IAuthProviderPtr CreateOAuthProvider(const OAuthConfig& conf);

/**
 * Creates an auth provider which auto-updates IAM tokens going to IAM endpoints
 *
 * @param iamConf
 * @param registry
 * @param timer
 * @param threadPool
 * @return an auth provider inflating http headers with an IAM token
 */
NSolomon::IAuthProviderPtr CreateIamAuthProvider(
        const IamConfig& iamConf,
        NMonitoring::TMetricRegistry& registry,
        TTimerThread& timer,
        TSimpleSharedPtr<IThreadPool> threadPool);

/**
 * Creates an auth provider which auto-updates IAM tokens going to vm's metadata service
 *
 * @param metadataServiceConf
 * @param registry
 * @param timer
 * @param threadPool
 * @return an auth provider inflating http headers with an IAM token
 */
NSolomon::IAuthProviderPtr CreateMetadataServiceAuthProvider(
        const MetadataServiceConfig& metadataServiceConf,
        NMonitoring::TMetricRegistry& registry,
        TTimerThread& timer,
        TSimpleSharedPtr<IThreadPool> threadPool);

/**
 * Creates various auth providers from a proto auth message depending on a message type
 */
template <class TProtoMessageWithAuth>
NSolomon::IAuthProviderPtr CreateAuthProvider(
        const TProtoMessageWithAuth& msgWithAuth,
        NMonitoring::TMetricRegistry& registry,
        TTimerThread& timer,
        TSimpleSharedPtr<IThreadPool> threadPool)
{
    if (msgWithAuth.HasOAuthConfig()) {
        return NSolomon::NAgent::CreateOAuthProvider(msgWithAuth.GetOAuthConfig());
    } else if (msgWithAuth.HasTvmConfig()) {
        return CreateTvmAuthProvider(msgWithAuth.GetTvmConfig());
    } else if (msgWithAuth.HasIamConfig()) {
        return CreateIamAuthProvider(msgWithAuth.GetIamConfig(), registry, timer, std::move(threadPool));
    } else if (msgWithAuth.HasMetadataServiceConfig()) {
        return CreateMetadataServiceAuthProvider(
                msgWithAuth.GetMetadataServiceConfig(),
                registry,
                timer,
                std::move(threadPool));
    } else {
        return nullptr;
    }
}

} // namespace NSolomon::NAgent
