#include "auth.h"

#include <solomon/agent/misc/logger.h>

#include <solomon/libs/cpp/cloud/iam/client.h>
#include <solomon/libs/cpp/cloud/iam/counters.h>

#include <library/cpp/resource/resource.h>

#include <util/generic/ptr.h>
#include <util/string/builder.h>
#include <util/string/strip.h>


namespace NSolomon::NAgent {
namespace {

constexpr TStringBuf GRPC_IAM_CLOUD_PROD = "ts.private-api.cloud.yandex.net:4282";
constexpr TStringBuf GRPC_IAM_CLOUD_PREPROD = "ts.private-api.cloud-preprod.yandex.net:4282";
constexpr TStringBuf GRPC_IAM_CLOUD_GPN = "ts.private-api.ycp.gpn.yandexcloud.net:443";
constexpr TStringBuf GRPC_IAM_CLOUD_ISRAEL = "ts.private-api.yandexcloud.co.il:14282";

constexpr TStringBuf INSTANCE_OVERLAY_METADATA_ADDR = "http://169.254.169.254:80/computeMetadata/v1/instance/service-accounts/default/token";
constexpr TStringBuf INSTANCE_UNDERLAY_METADATA_ADDR = "http://localhost:6770/computeMetadata/v1/instance/service-accounts/default/token";

class TIamExecutor: public NSolomon::NCloud::IExecutor {
public:
    TIamExecutor(TTimerThread& timerThread, TSimpleSharedPtr<IThreadPool> pool)
        : TimerThread_{timerThread}
        , ThreadPool_{pool}
    {}

    void Schedule(std::function<void(void)> fn, TDuration delay, TDuration interval) override {
        auto task = MakeFuncTimerTask(ThreadPool_.Get(), std::move(fn));
        TimerThread_.Schedule(task, delay, interval);

        if (interval != TDuration::Zero()) {
            PeriodicTasks_.emplace_back(task);
        }
    }

    void Stop() override {
        for (auto&& task: PeriodicTasks_) {
            task->Cancel();
        }
    }

private:
    // TODO: is it safe to use a ref instead of a smart ptr?
    TTimerThread& TimerThread_;
    TSimpleSharedPtr<IThreadPool> ThreadPool_;

    TVector<ITimerTaskPtr> PeriodicTasks_;
};

class TCloudLogger: public NSolomon::NCloud::ILogger {
public:
    TCloudLogger(TString logPrefix)
        : LogPrefix_{std::move(logPrefix)}
    {
    }

private:
    void Write(ELogPriority severity, TString msg) override {
        switch (severity) {
            case TLOG_EMERG:
            case TLOG_ALERT:
            case TLOG_CRIT:
                SA_LOG(FATAL) << LogPrefix_ << msg;
                break;
            case TLOG_ERR:
                SA_LOG(ERROR) << LogPrefix_ << msg;
                break;
            case TLOG_WARNING:
            case TLOG_NOTICE:
                SA_LOG(WARN) << LogPrefix_ << msg;
                break;
            case TLOG_INFO:
                SA_LOG(INFO) << LogPrefix_ << msg;
                break;
            case TLOG_DEBUG:
            case TLOG_RESOURCES:
                SA_LOG(DEBUG) << LogPrefix_ << msg;
                break;
            default:
                SA_LOG(DEBUG) << LogPrefix_ << "unknown severity level: " << severity << "; msg: " << msg;
                break;
        }
    }

private:
    const TString LogPrefix_;
};

class TIamProvider: public NSolomon::IAuthProvider {
public:
    ~TIamProvider() {
        if (Executor_) {
            Executor_->Stop();
        }
    }

    TIamProvider(
            const IamConfig& iamConfig,
            const TString& grpcAddress,
            NMonitoring::TMetricRegistry& registry,
            TTimerThread& timer,
            TSimpleSharedPtr<IThreadPool> threadPool)
    {
        if (!timer.IsStarted()) {
            timer.Start();
        }

        TString serviceAccountId = Strip(iamConfig.GetServiceAccountId());
        TString keyId = Strip(iamConfig.GetKeyId());
        TString publicKeyFile = Strip(iamConfig.GetPublicKeyFile());
        TString privateKeyFile = Strip(iamConfig.GetPrivateKeyFile());
        TString updatePeriodStr = Strip(iamConfig.GetUpdatePeriod());

        Y_ENSURE(!serviceAccountId.empty(), "ServiceAccountId is empty");
        Y_ENSURE(!keyId.empty(), "KeyId is empty");
        Y_ENSURE(!publicKeyFile.empty(), "PublicKeyFile is empty");
        Y_ENSURE(!privateKeyFile.empty(), "PrivateKeyFile is empty");

        NSolomon::NCloud::TUpdatingProviderConf conf;

        try {
            conf.UpdatePeriod = updatePeriodStr.empty() ? TDuration::Hours(1) : TDuration::Parse(updatePeriodStr);
        } catch (...) {
            ythrow yexception() << "failed to parse UpdatePeriod in IamConfig: " << CurrentExceptionMessage();
        }

        conf.ServiceAccountId = serviceAccountId;
        conf.KeyId = keyId;
        conf.PublicKeyFile = publicKeyFile;
        conf.PrivateKeyFile = privateKeyFile;

        Executor_ = MakeHolder<TIamExecutor>(timer, std::move(threadPool));

        yandex::solomon::config::rpc::TGrpcClientConfig grpcConf;

        grpcConf.AddAddresses(grpcAddress);
        grpcConf.SetWorkerThreads(1);

        if (grpcAddress == GRPC_IAM_CLOUD_GPN) {
            if (!grpcConf.HasSslOptions()) {
                auto* sslOpts = grpcConf.MutableSslOptions();
                sslOpts->SetPemRootCerts(NResource::Find("GPNInternalRootCA.crt"));
            }
        }

        IamClient_ = NSolomon::NCloud::CreateGrpcIamClient(grpcConf, registry);

        auto counters = MakeHolder<NSolomon::NCloud::TDefaultCounters>(registry);
        auto logger = MakeIntrusive<TCloudLogger>("IAM: ");

        TokenProvider_ = NSolomon::NCloud::CreateUpdatingProvider(
                conf,
                *Executor_,
                IamClient_,
                std::move(counters),
                std::move(logger));
        TokenProvider_->Start(NCloud::EStartPolicy::Async);
    }

private:
    void AddCredentials(THashMap<TString, TString>& headers) const override {
        if (auto token = TokenProvider_->Token()) {
            headers["Authorization"] = TStringBuilder() << "Bearer " << token->Token();
        } else  {
            ythrow yexception() << "Cannot obtain IAM token";
        }
    }

private:
    THolder<NSolomon::NCloud::IExecutor> Executor_;
    NSolomon::NCloud::IIamClientPtr IamClient_;
    NSolomon::NCloud::ITokenProviderPtr TokenProvider_;
};

class TMetadataServiceAuthProvider: public NSolomon::IAuthProvider {
public:
    explicit TMetadataServiceAuthProvider(NSolomon::NCloud::ITokenProviderPtr tokenProviderPtr)
        : TokenProvider_{std::move(tokenProviderPtr)}
    {
    }

private:
    void AddCredentials(THashMap<TString, TString>& headers) const override {
        if (auto token = TokenProvider_->Token()) {
            headers["Authorization"] = TStringBuilder() << "Bearer " << token->Token();
        } else  {
            ythrow yexception() << "Cannot obtain IAM token from metadata service";
        }
    }

private:
    NSolomon::NCloud::ITokenProviderPtr TokenProvider_;
};

TString RetrieveGrpcAddressForIam(const IamConfig& iamConf) {
    TString grpcAddress;

    switch (iamConf.GetCluster()) {
        case EClusterType::CLOUD_PROD:
            grpcAddress = GRPC_IAM_CLOUD_PROD;
            break;
        case EClusterType::CLOUD_PREPROD:
            grpcAddress = GRPC_IAM_CLOUD_PREPROD;
            break;
        case EClusterType::CLOUD_GPN:
            grpcAddress = GRPC_IAM_CLOUD_GPN;
            break;
        case EClusterType::CLOUD_ISRAEL:
            grpcAddress = GRPC_IAM_CLOUD_ISRAEL;
            break;
        default:
            // Neither Internal nor Cloud cluster -> Custom url. Then get an address from the config
            grpcAddress = Strip(iamConf.GetCustomGrpcAddress());
            Y_ENSURE(
                    !grpcAddress.empty(),
                    "IamConfig::CustomGrpcAddress is not set or IamConfig::Cluster is set to an incorrect value: "
                    << EClusterType_Name(iamConf.GetCluster()));
            break;
    }

    return grpcAddress;
}

TString RetrieveMetadataServiceAddress(const MetadataServiceConfig& conf) {
    TString metadataServiceAddress;

    switch (conf.GetInstanceType()) {
        case EMetadataInstanceType::OVERLAY:
            metadataServiceAddress = INSTANCE_OVERLAY_METADATA_ADDR;
            break;
        case EMetadataInstanceType::UNDERLAY:
            metadataServiceAddress = INSTANCE_UNDERLAY_METADATA_ADDR;
            break;
        default:
            metadataServiceAddress = Strip(conf.GetUrl());
            Y_ENSURE(
                    !metadataServiceAddress.empty(),
                    "MetadataServiceConfig::Url is not set or MetadataServiceConfig::InstanceType is set to an incorrect value: "
                    << EMetadataInstanceType_Name(conf.GetInstanceType()));
            break;
    }

    return metadataServiceAddress;
}

} // namespace

NSolomon::IAuthProviderPtr CreateIamAuthProvider(
        const IamConfig& iamConf,
        NMonitoring::TMetricRegistry& registry,
        TTimerThread& timer,
        TSimpleSharedPtr<IThreadPool> threadPool)
{
    auto grpcAddress = RetrieveGrpcAddressForIam(iamConf);

    return new TIamProvider{iamConf, std::move(grpcAddress), registry, timer, std::move(threadPool)};
}

NSolomon::IAuthProviderPtr CreateMetadataServiceAuthProvider(
        const MetadataServiceConfig& metadataServiceConf,
        NMonitoring::TMetricRegistry& registry,
        TTimerThread& timer,
        TSimpleSharedPtr<IThreadPool> threadPool)
{
    NSolomon::NCloud::TMetadataServiceProviderConf conf;
    const TString& updatePeriodStr = metadataServiceConf.GetUpdatePeriod();

    try {
        conf.UpdatePeriod = updatePeriodStr.empty() ? TDuration::Hours(1) : TDuration::Parse(updatePeriodStr);
    } catch (...) {
        ythrow yexception() << "failed to parse UpdatePeriod in MetadataServiceConfig: " << CurrentExceptionMessage();
    }

    conf.MetadataServiceAddress = RetrieveMetadataServiceAddress(metadataServiceConf);

    if (!timer.IsStarted()) {
        timer.Start();
    }
    auto executor = MakeHolder<TIamExecutor>(timer, std::move(threadPool));
    NSolomon::NCloud::ILoggerPtr logger = MakeIntrusive<TCloudLogger>("MetadataServiceTokenProvider: ");

    NSolomon::NCloud::ITokenProviderPtr tokenProviderPtr = CreateMetadataServiceTokenProvider(
            std::move(conf),
            std::move(executor),
            registry,
            std::move(logger));
    tokenProviderPtr->Start(NCloud::EStartPolicy::Async);

    return new TMetadataServiceAuthProvider{std::move(tokenProviderPtr)};
}

} // namespace NSolomon::NAgent
