#pragma once

#include <solomon/libs/cpp/http/client/http.h>

#include <util/generic/hash.h>
#include <util/generic/yexception.h>
#include <util/string/strip.h>
#include <util/system/env.h>

namespace NSolomon::NAgent {

class TAuthTokenProcessor: private TNonCopyable {
public:
    static TAuthTokenProcessor* Instance() {
        return Singleton<TAuthTokenProcessor>();
    }

    void InflateHttpHeaders(IHeaders& headers) {
        if (!HasToken_) {
            return;
        }

        const THeader& httpHeader = TokenTypeToHttpHeader_.at(AuthTokenType_);
        headers.Add(httpHeader.Name, TString{httpHeader.Value} + AuthToken_);
    }

    bool HasToken() const {
        return HasToken_;
    }

private:
    Y_DECLARE_SINGLETON_FRIEND();
    TAuthTokenProcessor() {
        Init();
    }

    void Init() {
        Y_ENSURE(!HasToken_, "Init() cannot be called when a token is already gathered");

        TString authTokenType = Strip(GetEnv("SA_AUTH_TYPE", ""));
        authTokenType.to_lower();

        if (authTokenType.empty()) {
            return;
        }

        auto it = TokenTypeStrToEnum_.find(authTokenType);
        Y_ENSURE(it != TokenTypeStrToEnum_.end(),
                 "Unknown authorization token type: SA_AUTH_TYPE=\"" << authTokenType << "\"");
        AuthTokenType_ = it->second;

        TString authorizationToken = Strip(GetEnv("SA_AUTH_TOKEN", ""));

        Y_ENSURE(!authorizationToken.empty(),
                 "SA_AUTH_TYPE is set to " << authTokenType << ", but no authorization token is specified inside an"
                                                               " SA_AUTH_TOKEN variable");

        HasToken_ = true;
        AuthToken_ = authorizationToken;
    }

private:
    bool HasToken_ = false;

    enum class EAuthTokenType {
        None,
        OAuth,
        IAM,
        TVM_Service,
    };
    EAuthTokenType AuthTokenType_;
    TString AuthToken_;

    const THashMap<TStringBuf, EAuthTokenType> TokenTypeStrToEnum_ = {
        {"oauth", EAuthTokenType::OAuth},
        {"iam", EAuthTokenType::IAM},
        {"tvm_service", EAuthTokenType::TVM_Service},
    };

    struct THeader {
        const TStringBuf Name;
        const TStringBuf Value;
    };

    const THashMap<EAuthTokenType, THeader> TokenTypeToHttpHeader_ = {
        {EAuthTokenType::OAuth, {"Authorization", "OAuth "}},
        {EAuthTokenType::IAM, {"Authorization", "Bearer "}},
        {EAuthTokenType::TVM_Service, {"X-Ya-Service-Ticket", ""}},
    };
};

} // namespace NSolomon::NAgent
