#include "auth.h"

#include <solomon/agent/misc/logger.h>

#include <library/cpp/tvmauth/client/facade.h>

#include <util/stream/file.h>

namespace NSolomon::NAgent {
namespace {

constexpr ui32 PROD_TVM_ID{2010242};
constexpr ui32 PRE_TVM_ID{2010240};
constexpr ui32 TEST_TVM_ID{2010238};
constexpr ui32 CLOUD_PROD_TVM_ID{2015581};
constexpr ui32 CLOUD_PREPROD_TVM_ID{2015583};

class TTicketParserLogger: public NTvmAuth::ILogger {
public:
    void Log(int lvl, const TString& msg) override {
        switch (lvl) {
            case 7:
                SA_LOG(DEBUG) << LogPrefix_ << msg;
                break;
            case 6:
            case 5:
                SA_LOG(INFO) << LogPrefix_ << msg;
                break;
            case 4:
                SA_LOG(WARN) << LogPrefix_ << msg;
                break;
            case 3:
                SA_LOG(ERROR) << LogPrefix_ << msg;
                break;
            default: {
                if (lvl > 7) {
                    SA_LOG(TRACE) << LogPrefix_ << msg;
                } else if (lvl < 3) {
                    SA_LOG(FATAL) << LogPrefix_ << msg;
                }
            }
        }
    }

private:
    static constexpr TStringBuf LogPrefix_{TStringBuf("TVM: ")};
};

class TTvmProvider: public NSolomon::IAuthProvider {
public:
    TTvmProvider(NTvmAuth::NTvmApi::TClientSettings&& setts, ui32 dstTvmId) {
        TvmClient_ = MakeHolder<NTvmAuth::TTvmClient>(setts, MakeIntrusive<TTicketParserLogger>());
        TokenProvider_ = CreateTvmProvider(*TvmClient_, dstTvmId);
    }

private:
    void AddCredentials(THashMap<TString, TString> &headers) const override {
        TokenProvider_->AddCredentials(headers);
    }

private:
    THolder<NTvmAuth::TTvmClient> TvmClient_;
    NSolomon::IAuthProviderPtr TokenProvider_;
};

ui32 RetrieveTvmId(const TvmConfig& tvmConfig) {
    ui32 dstTvmId;

    switch (tvmConfig.GetCluster()) {
        case EClusterType::PRODUCTION:
            dstTvmId = PROD_TVM_ID;
            break;
        case EClusterType::PRESTABLE:
            dstTvmId = PRE_TVM_ID;
            break;
        case EClusterType::TESTING:
            dstTvmId = TEST_TVM_ID;
            break;
        case EClusterType::CLOUD_PROD:
            dstTvmId = CLOUD_PROD_TVM_ID;
            break;
        case EClusterType::CLOUD_PREPROD:
            dstTvmId = CLOUD_PREPROD_TVM_ID;
            break;
        default:
            // neither internal nor Cloud -> Custom url. Then get an id from the config
            dstTvmId = tvmConfig.GetCustomDstId();
            Y_ENSURE(dstTvmId != 0, "CustomDstId is not set");
            break;
    }

    return dstTvmId;
}

} // namespace

NSolomon::IAuthProviderPtr CreateTvmAuthProvider(const TvmConfig& tvmConfig) {
    const auto& secretFilePath = tvmConfig.GetSecretFile();
    Y_ENSURE(!secretFilePath.empty(), "file with a TVM client secret is not specified");

    TFileInput in{secretFilePath};
    auto clientSecret = Strip(in.ReadAll());
    Y_ENSURE(!clientSecret.empty(), "TVM client secret is empty");

    NTvmAuth::NTvmApi::TClientSettings setts;
    setts.SetSelfTvmId(tvmConfig.GetClientId());
    // TODO: add a disk cache
    // setts.SetDiskCache();

    ui32 dstTvmId = RetrieveTvmId(tvmConfig);
    setts.EnableServiceTicketsFetchOptions(clientSecret, NTvmAuth::NTvmApi::TClientSettings::TDstVector{dstTvmId});

    return new TTvmProvider{std::move(setts), dstTvmId};
}

} // namespace NSolomon::NAgent
