#include <solomon/agent/lib/auth/auth.h>
#include <solomon/agent/misc/timer_thread.h>
#include <solomon/agent/protos/endpoint_with_auth.pb.h>

#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/resource/resource.h>
#include <library/cpp/testing/gtest/gtest.h>

#include <util/system/tempfile.h>
#include <util/thread/pool.h>

using namespace NSolomon;
using namespace NAgent;

THolder<TTempFileHandle> CreateTmpFile(const TString& content) {
    auto tmpFilePtr = MakeHolder<TTempFileHandle>();
    tmpFilePtr->Write(content.data(), content.size());
    tmpFilePtr->FlushData();

    return tmpFilePtr;
}

THolder<TTempFileHandle> SetOauth(TEndpointWithAuth& endpoint) {
    auto tmpFilePtr = CreateTmpFile(NResource::Find("oauth.secret"));

    auto* oauth = endpoint.MutableOAuthConfig();
    oauth->SetSecretFile(tmpFilePtr->Name());

    return tmpFilePtr;
}

TEST(TAuthProviderTest, HappyPath) {
    NMonitoring::TMetricRegistry registry;
    TTimerThread timer;
    auto threadPoolPtr = MakeSimpleShared<TThreadPool>();

    TEndpointWithAuth endpoint;
    endpoint.SetType(EClusterType::PRODUCTION);

    auto tmpFilePtr = SetOauth(endpoint);

    NSolomon::IAuthProviderPtr authProviderPtr;
    ASSERT_NO_THROW(authProviderPtr = CreateAuthProvider(endpoint, registry, timer, threadPoolPtr));

    THashMap<TString, TString> headers;
    authProviderPtr->AddCredentials(headers);

    ASSERT_EQ(headers.size(), 1u);

    auto [hName, hValue]  = (*headers.begin());

    ASSERT_EQ(hName, "Authorization");
    ASSERT_EQ(hValue, "OAuth oauth_token_value");
}

TEST(TAuthProviderTest, NoAuthSpecified) {
    NMonitoring::TMetricRegistry registry;
    TTimerThread timer;
    auto threadPoolPtr = MakeSimpleShared<TThreadPool>();

    TEndpointWithAuth endpoint;

    auto providerPtr = CreateAuthProvider(endpoint, registry, timer, threadPoolPtr);
    ASSERT_FALSE(providerPtr);
}

TEST(TAuthProviderTest, EmptyTvmFields) {
    NMonitoring::TMetricRegistry registry;
    TTimerThread timer;
    auto threadPoolPtr = MakeSimpleShared<TThreadPool>();

    {
        TEndpointWithAuth endpoint;
        endpoint.SetType(EClusterType::PRODUCTION);
        endpoint.MutableTvmConfig();

        ASSERT_THROW_MESSAGE_HAS_SUBSTR(
                CreateAuthProvider(endpoint, registry, timer, threadPoolPtr),
                yexception,
                "file with a TVM client secret is not specified");
    }

    {
        TEndpointWithAuth endpoint;
        endpoint.SetType(EClusterType::PRODUCTION);

        auto tmpFilePtr = CreateTmpFile(NResource::Find("empty.secret"));
        endpoint.MutableTvmConfig()->SetSecretFile(tmpFilePtr->Name());

        ASSERT_THROW_MESSAGE_HAS_SUBSTR(
                CreateAuthProvider(endpoint, registry, timer, threadPoolPtr),
                yexception,
                "TVM client secret is empty");
    }
}

TEST(TAuthProviderTest, EmptyIamFields) {
    NMonitoring::TMetricRegistry registry;
    TTimerThread timer;
    auto threadPoolPtr = MakeSimpleShared<TThreadPool>();

    {
        TEndpointWithAuth endpoint;
        endpoint.MutableIamConfig();

        ASSERT_THROW_MESSAGE_HAS_SUBSTR(
                CreateAuthProvider(endpoint, registry, timer, threadPoolPtr),
                yexception,
                "CustomGrpcAddress is not set");
    }

    {
        TEndpointWithAuth endpoint;
        auto* c = endpoint.MutableIamConfig();
        c->SetCluster(EClusterType::CLOUD_PROD);

        ASSERT_THROW_MESSAGE_HAS_SUBSTR(
                CreateAuthProvider(endpoint, registry, timer, threadPoolPtr),
                yexception,
                "ServiceAccountId is empty");
    }

    {
        TEndpointWithAuth endpoint;
        auto* c = endpoint.MutableIamConfig();
        c->SetCluster(EClusterType::CLOUD_PROD);
        c->SetServiceAccountId("123");

        ASSERT_THROW_MESSAGE_HAS_SUBSTR(
                CreateAuthProvider(endpoint, registry, timer, threadPoolPtr),
                yexception,
                "KeyId is empty");
    }

    {
        TEndpointWithAuth endpoint;
        auto* iam = endpoint.MutableIamConfig();
        iam->SetCluster(EClusterType::CLOUD_PROD);
        iam->SetServiceAccountId("123");
        iam->SetKeyId("123");

        ASSERT_THROW_MESSAGE_HAS_SUBSTR(
                CreateAuthProvider(endpoint, registry, timer, threadPoolPtr),
                yexception,
                "PublicKeyFile is empty");
    }

    {
        TEndpointWithAuth endpoint;
        auto* iam = endpoint.MutableIamConfig();
        iam->SetCluster(EClusterType::CLOUD_PROD);
        iam->SetServiceAccountId("123");
        iam->SetKeyId("123");

        auto tmpFilePtr = CreateTmpFile(NResource::Find("empty.secret"));
        iam->SetPublicKeyFile(tmpFilePtr->Name());

        ASSERT_THROW_MESSAGE_HAS_SUBSTR(
                CreateAuthProvider(endpoint, registry, timer, threadPoolPtr),
                yexception,
                "PrivateKeyFile is empty");
    }
}

TEST(TAuthProviderTest, EmptyOAuthFields) {
    NMonitoring::TMetricRegistry registry;
    TTimerThread timer;
    auto threadPoolPtr = MakeSimpleShared<TThreadPool>();

    {
        TEndpointWithAuth endpoint;
        endpoint.SetType(EClusterType::PRODUCTION);
        endpoint.MutableOAuthConfig();

        ASSERT_THROW_MESSAGE_HAS_SUBSTR(
                CreateAuthProvider(endpoint, registry, timer, threadPoolPtr),
                yexception,
                "file with an OAuth token is not specified");
    }

    {
        TEndpointWithAuth endpoint;
        endpoint.SetType(EClusterType::PRODUCTION);

        auto tmpFilePtr = CreateTmpFile(NResource::Find("empty.secret"));
        endpoint.MutableOAuthConfig()->SetSecretFile(tmpFilePtr->Name());

        ASSERT_THROW_MESSAGE_HAS_SUBSTR(
                CreateAuthProvider(endpoint, registry, timer, threadPoolPtr),
                yexception,
                "OAuth token is empty");
    }
}
