#include <solomon/agent/lib/auth/token_processor.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;
using namespace NAgent;

TEST(TAuthTokenProcessorTest, NoToken) {
    TAuthTokenProcessor::Instance();
    // No exception after the second call
    TAuthTokenProcessor *tokenProcessor = TAuthTokenProcessor::Instance();

    ui64 count{0};
    auto countHeaders = [&] (auto&&, auto&&) { ++count; };
    auto headers = Headers();
    headers->ForEach(countHeaders);
    ASSERT_EQ(count, 0u);

    tokenProcessor->InflateHttpHeaders(*headers);
    headers->ForEach(countHeaders);
    ASSERT_EQ(count, 0u);
}

TEST(TAuthTokenProcessorTest, TypeIsSpecifiedButTokenIsNot) {
    {
        SetEnv("SA_AUTH_TYPE", "oauth");
        ASSERT_THROW(TAuthTokenProcessor::Instance(), yexception);
    }

    {
        SetEnv("SA_AUTH_TYPE", "iam");
        ASSERT_THROW(TAuthTokenProcessor::Instance(), yexception);
    }

    {
        SetEnv("SA_AUTH_TYPE", "tvm_service");
        ASSERT_THROW(TAuthTokenProcessor::Instance(), yexception);
    }
}

void AssertSingleHeader(IHeaders& headers, TStringBuf name, TStringBuf expectedValue) {
    ui64 count{0};
    headers.ForEach([&] (auto&&, auto&&) { ++count; });

    ASSERT_EQ(count, 1u);
    auto h = headers.Find(name);
    ASSERT_TRUE(h.Defined());
    ASSERT_EQ(*h, expectedValue);
}

TEST(TAuthTokenProcessorTest, HappyPathOAuth) {
    SetEnv("SA_AUTH_TYPE", "OAuth");
    SetEnv("SA_AUTH_TOKEN", "oauth-token");

    TAuthTokenProcessor *tokenProcessor = TAuthTokenProcessor::Instance();

    auto headers = Headers();
    tokenProcessor->InflateHttpHeaders(*headers);
    AssertSingleHeader(*headers, "Authorization", "OAuth oauth-token");
}

TEST(TAuthTokenProcessorTest, HappyPathIAM) {
    SetEnv("SA_AUTH_TYPE", "IAM");
    SetEnv("SA_AUTH_TOKEN", "iam-token");

    TAuthTokenProcessor *tokenProcessor = TAuthTokenProcessor::Instance();

    auto headers = Headers();
    tokenProcessor->InflateHttpHeaders(*headers);
    AssertSingleHeader(*headers, "Authorization", "Bearer iam-token");
}

TEST(TAuthTokenProcessorTest, HappyPathTVMService) {
    SetEnv("SA_AUTH_TYPE", "TVM_SERVICE");
    SetEnv("SA_AUTH_TOKEN", "tvm-service-token");

    TAuthTokenProcessor *tokenProcessor = TAuthTokenProcessor::Instance();

    auto headers = Headers();
    tokenProcessor->InflateHttpHeaders(*headers);
    AssertSingleHeader(*headers, "X-Ya-Service-Ticket", "tvm-service-token");
}

TEST(TAuthTokenProcessorTest, UnknownTokenType) {
    SetEnv("SA_AUTH_TYPE", "some-token-type");
    ASSERT_THROW(TAuthTokenProcessor::Instance(), yexception);
}

TEST(TAuthTokenProcessorTest, IgnoringTokenTypeCase) {
    SetEnv("SA_AUTH_TYPE", "oauth");
    SetEnv("SA_AUTH_TOKEN", "oauth-token");

    TAuthTokenProcessor *tokenProcessor = TAuthTokenProcessor::Instance();

    auto headers = Headers();
    tokenProcessor->InflateHttpHeaders(*headers);
    AssertSingleHeader(*headers, "Authorization", "OAuth oauth-token");
}
