#include "config_loader.h"

#include <solomon/agent/misc/logger.h>
#include <solomon/agent/protos/loader_config.pb.h>

#include <util/system/thread.h>

#include <google/protobuf/util/message_differencer.h>


namespace NSolomon {
namespace NAgent {

using google::protobuf::util::MessageDifferencer;

TString ServiceConfigKey(const TServiceConfig& config) {
    TString key;
    key.reserve(config.GetProject().size() + config.GetService().size() + 1);
    key.append(config.GetProject());
    key.append(';');
    key.append(config.GetService());
    return key;
}

TConfigLoader::TConfigLoader(
        const TConfigLoaderConfig& config,
        const TGlobalPython2Config& pyConfig)
{
    if (config.HasStatic()) {
        auto loader = CreateStaticLoader(config.GetStatic());
        Loaders_.push_back(std::move(loader));
    }

    if (config.HasFileLoader()) {
        auto loader = CreateFileLoader(config.GetFileLoader());
        Loaders_.push_back(std::move(loader));
    }

    if (config.HasHttpLoader()) {
        auto loader = CreateHttpLoader(config.GetHttpLoader());
        Loaders_.push_back(std::move(loader));
    }

    if (config.HasPython2Loader()) {
        auto loader = CreatePython2Loader(config.GetPython2Loader(), pyConfig);
        Loaders_.push_back(std::move(loader));
    }
}

TConfigLoader::~TConfigLoader() {
    Stop();
}

namespace {

TDuration TimeBeforeNextInterval(TDuration execTimeDuration, TDuration intervalDuration) {
    ui64 execTime = execTimeDuration.MicroSeconds();
    ui64 interval = intervalDuration.MicroSeconds();

    return TDuration::MicroSeconds(interval - (execTime % interval));
}

} // namespace

void TConfigLoader::Start() {
    TDuration maxUpdateInterval = TDuration::Zero();

    for (const IServiceConfigLoaderPtr& loader: Loaders_) {
        const TDuration updateInterval = loader->UpdateInterval();

        if (updateInterval >= TDuration::Max()) {
            LoadConfigs(loader.Get());
        } else {
            maxUpdateInterval = Max(maxUpdateInterval, updateInterval);
            RunInBackground(loader->Name(), [this, loader, updateInterval]() {
                TInstant start = TInstant::Now();

                try {
                    LoadConfigs(loader.Get());

                    return TimeBeforeNextInterval(TInstant::Now() - start, updateInterval);
                } catch (...) {
                    SA_LOG(ERROR) << loader->Name() << ": cannot load configs: "
                                  << CurrentExceptionMessage();

                    TDuration timeBeforeNextInterval = TimeBeforeNextInterval(TInstant::Now() - start,
                                                                              updateInterval);
                    return timeBeforeNextInterval + updateInterval;
                }
            });
        }
    }

    if (maxUpdateInterval > TDuration::Zero()) {
        RunInBackground("ConfigsGc", [this, gcInterval = maxUpdateInterval * 2] {
            TInstant start = TInstant::Now();

            GcConfigs();

            return TimeBeforeNextInterval(TInstant::Now() - start, gcInterval);
        });
    }
}

void TConfigLoader::Stop() {
    TBackgroundThreads::Stop();
}

void TConfigLoader::AddLoader(IServiceConfigLoaderPtr loader) {
    Loaders_.push_back(std::move(loader));
}

void TConfigLoader::AddWatcher(IServiceConfigWatcherPtr watcher) {
    Watchers_.push_back(std::move(watcher));
}

void TConfigLoader::LoadConfigs(IServiceConfigLoader* loader) {
    TVector<TServiceConfig> addedConfigs;
    TVector<std::pair<TServiceConfig, TServiceConfig>> changedConfigs;

    auto loadedConfigs = loader->Load();
    for (const TServiceConfig& newConfig: loadedConfigs) {
        TString configKey = ServiceConfigKey(newConfig);

        with_lock (ServiceConfigsMutex_) {
            TServiceConfigTtl* oldConfig = ServiceConfigs_.FindPtr(configKey);
            if (oldConfig == nullptr) {
                addedConfigs.push_back(newConfig);
                auto entry = ServiceConfigs_.emplace(
                        configKey,
                        TServiceConfigTtl{ newConfig, {}, {} });
                oldConfig = &entry.first->second;
            } else if (!MessageDifferencer::Equals(oldConfig->Config, newConfig)) {
                changedConfigs.emplace_back(oldConfig->Config, newConfig);
                oldConfig->Config = newConfig;
            }

            auto ttl = loader->UpdateInterval() + loader->UpdateInterval(); // XXX: TDuration addition is saturating, but multiplication is not
            oldConfig->LastUpdate = TInstant::Now();
            oldConfig->ExpirationTime = oldConfig->LastUpdate + ttl;
        }
    }

    for (const TServiceConfig& config: addedConfigs) {
        for (const IServiceConfigWatcherPtr& w: Watchers_) {
            w->OnAdded(config);
        }
    }

    for (const auto& configs: changedConfigs) {
        for (const IServiceConfigWatcherPtr& w: Watchers_) {
            w->OnChanged(configs.first, configs.second);
        }
    }
}

void TConfigLoader::GcConfigs() {
    TVector<TServiceConfig> removedConfigs;
    TInstant now = TInstant::Now();

    with_lock (ServiceConfigsMutex_) {
        for (auto it = ServiceConfigs_.begin(),
                 end = ServiceConfigs_.end(); it != end; )
        {
            const TServiceConfigTtl& config = it->second;
            if (now > config.ExpirationTime) {
                removedConfigs.push_back(config.Config);
                ServiceConfigs_.erase(it++);
            } else {
                ++it;
            }
        }
    }

    for (const TServiceConfig& config: removedConfigs) {
        for (const IServiceConfigWatcherPtr& w: Watchers_) {
            w->OnRemoved(config);
        }
    }
}

} // namespace NAgent
} // namespace NSolomon
