#pragma once

#include <solomon/agent/misc/background_threads.h>
#include <solomon/agent/protos/service_config.pb.h>

#include <util/system/spinlock.h>
#include <util/generic/hash.h>


namespace NSolomon {
namespace NAgent {

class TConfigLoaderConfig;
class TFileLoaderConfig;
class THttpLoaderConfig;
class TPython2LoaderConfig;
class TGlobalPython2Config;
class TStaticConfig;


///////////////////////////////////////////////////////////////////////////////
// IServiceConfigLoader
///////////////////////////////////////////////////////////////////////////////
struct IServiceConfigLoader: public TAtomicRefCount<IServiceConfigLoader> {
    virtual ~IServiceConfigLoader() = default;

    virtual TStringBuf Name() const = 0;
    virtual TDuration UpdateInterval() const = 0;

    virtual TVector<TServiceConfig> Load() = 0;
};

using IServiceConfigLoaderPtr = TIntrusivePtr<IServiceConfigLoader>;


///////////////////////////////////////////////////////////////////////////////
// IServiceConfigWatcher
///////////////////////////////////////////////////////////////////////////////
struct IServiceConfigWatcher: public TAtomicRefCount<IServiceConfigWatcher> {
    virtual ~IServiceConfigWatcher() = default;

    virtual void OnAdded(const TServiceConfig& config) = 0;
    virtual void OnRemoved(const TServiceConfig& config) = 0;
    virtual void OnChanged(
            const TServiceConfig& oldConfig,
            const TServiceConfig& newConfig) = 0;
};

using IServiceConfigWatcherPtr = TIntrusivePtr<IServiceConfigWatcher>;


///////////////////////////////////////////////////////////////////////////////
// TConfigLoader
///////////////////////////////////////////////////////////////////////////////
class TConfigLoader: private TBackgroundThreads {
    struct TServiceConfigTtl {
        TServiceConfig Config;
        TInstant LastUpdate;
        TInstant ExpirationTime;
    };

public:
    TConfigLoader(
            const TConfigLoaderConfig& config,
            const TGlobalPython2Config& pyConfig);

    ~TConfigLoader();

    void Start();
    void Stop();

    // XXX: not threadsafe, must be called before calling Start()
    void AddLoader(IServiceConfigLoaderPtr loader);

    // XXX: not threadsafe, must be called before calling Start()
    void AddWatcher(IServiceConfigWatcherPtr watcher);

private:
    void LoadConfigs(IServiceConfigLoader* loader);
    void GcConfigs();

private:
    TVector<IServiceConfigWatcherPtr> Watchers_;
    TVector<IServiceConfigLoaderPtr> Loaders_;

    TAdaptiveLock ServiceConfigsMutex_;
    THashMap<TString, TServiceConfigTtl> ServiceConfigs_;
};


///////////////////////////////////////////////////////////////////////////////
// free functions
///////////////////////////////////////////////////////////////////////////////

TString ServiceConfigKey(const TServiceConfig& config);

IServiceConfigLoaderPtr CreateStaticLoader(const TStaticConfig& config);
IServiceConfigLoaderPtr CreateFileLoader(const TFileLoaderConfig& config);
IServiceConfigLoaderPtr CreateHttpLoader(const THttpLoaderConfig& config);
IServiceConfigLoaderPtr CreatePython2Loader(
        const TPython2LoaderConfig& config,
        const TGlobalPython2Config& pyConfig);

} // namespace NAgent
} // namespace NSolomon
