#include "config_loader.h"

#include <solomon/agent/misc/logger.h>
#include <solomon/agent/protos/loader_config.pb.h>

#include <library/cpp/json/json_value.h>
#include <library/cpp/protobuf/json/json2proto.h>
#include <library/cpp/protobuf/util/pb_io.h>

#include <util/folder/path.h>
#include <util/string/join.h>
#include <util/stream/file.h>


namespace NSolomon {
namespace NAgent {
namespace {

class TFileConfigLoader final: public IServiceConfigLoader {
public:
    TFileConfigLoader(const TFileLoaderConfig& config)
        : UpdateInterval_(TDuration::Parse(config.GetUpdateInterval()))
        , Config_(config)
        , WorkDir_{config.GetConfigsDir() ? config.GetConfigsDir() : "./"}
        , LoadFromDir_{config.GetLoadFromDir()}
    {
        SA_LOG(DEBUG) << Name() << ": start { "
                      << "configsDir: " << config.GetConfigsDir() << ", "
                      << "configFiles: [ "
                      << JoinSeq(", ", config.GetConfigsFiles())
                      << " ] }"
                      << "loadAll: [ " << LoadFromDir_ << " ]";
    }

    ~TFileConfigLoader() {
        SA_LOG(DEBUG) << Name() << ": stop";
    }

private:
    TStringBuf Name() const override {
        return TStringBuf("FileConfigLoader");
    }

    TDuration UpdateInterval() const override {
        return UpdateInterval_;
    }

    TVector<TServiceConfig> Load() override {
        TVector<TServiceConfig> result;

        auto parseAndAppend = [&result] (const TFsPath& path) {
            try {
                TServiceConfig serviceConfig;
                TFileInput in{path};

                if (path.GetExtension() == "json") {
                    NJson::TJsonReaderConfig readerConfig;
                    readerConfig.DontValidateUtf8 = true;

                    NJson::TJsonValue jsonValue;
                    NJson::ReadJsonTree(&in, &readerConfig, &jsonValue, true);

                    NProtobufJson::Json2Proto(jsonValue, serviceConfig,
                                              NProtobufJson::TJson2ProtoConfig().SetCastFromString(true));
                } else {
                    ParseFromTextFormat(in, serviceConfig);
                }

                result.emplace_back();
                result.back().Swap(&serviceConfig);
            } catch (const TIoException&) {
                SA_LOG(WARN) << CurrentExceptionMessage();
            } catch (...) {
                SA_LOG(WARN) << "cannot parse TServiceConfig from: " << path;
            }
        };

        for (const TString& fileName: Config_.GetConfigsFiles()) {
            // TODO: support different file types
            // TODO: remember file mtimex

            // if the file path is already absolute disregard working dir
            TFsPath absPath{fileName};
            if (!absPath.IsAbsolute()) {
                absPath = WorkDir_ / absPath;
            }

            parseAndAppend(absPath);
        }

        if (LoadFromDir_.Exists()) {
            TVector<TFsPath> children;
            LoadFromDir_.List(children);

            for (auto&& child: children) {
                if (child.IsFile() || child.IsSymlink()) {
                    parseAndAppend(child);
                }
            }
        }

        SA_LOG(DEBUG) << Name() << ": load";
        return result;
    }

private:
    TDuration UpdateInterval_;
    TFileLoaderConfig Config_;
    const TFsPath WorkDir_;
    const TFsPath LoadFromDir_;
};

} // namespace

IServiceConfigLoaderPtr CreateFileLoader(const TFileLoaderConfig& config) {
    return IServiceConfigLoaderPtr(new TFileConfigLoader(config));
}

} // namespace NAgent
} // namespace NSolomon
