#include "config_loader.h"

#include <solomon/agent/misc/logger.h>
#include <solomon/agent/protos/loader_config.pb.h>

#include <library/cpp/http/simple/http_client.h>
#include <library/cpp/http/misc/httpdate.h>
#include <library/cpp/protobuf/json/json2proto.h>
#include <library/cpp/svnversion/svnversion.h>

#include <util/stream/str.h>

#if defined(_win_) || defined(_cygwin_)
#   define PLATFORM_ID "Windows"
#elif defined(_linux_)
#   define PLATFORM_ID "Linux"
#elif defined(_darwin_)
#   define PLATFORM_ID "Darwin"
#elif defined(_freebsd_)
#   define PLATFORM_ID "FreeBSD"
#else
#   define PLATFORM_ID "Unknown"
#endif


namespace NSolomon {
namespace NAgent {
namespace {

///////////////////////////////////////////////////////////////////////////////
// TCacheAwareHttpClient
///////////////////////////////////////////////////////////////////////////////
class TCacheAwareHttpClient: public TSimpleHttpClient {
public:
    explicit TCacheAwareHttpClient(const TOptions& options)
        : TSimpleHttpClient(options)
    {
    }

    void ProcessResponse(
            TStringBuf relativeUrl,
            THttpInput& input,
            IOutputStream* output,
            const unsigned statusCode) const override
    {
        if (statusCode >= 200 && statusCode < 300) {
            Y_VERIFY_DEBUG(output);
            TransferData(&input, output);
        } else if (statusCode == 304) {
            // requested content is not modified
        } else {
            TString rest = input.ReadAll();
            ythrow THttpRequestException(statusCode)
                    << "Got " << statusCode << " from " << Host << relativeUrl
                    << "\nFull http response:\n" << rest;
        }
    }
};

TCacheAwareHttpClient::TOptions HttpClientOpts(const THttpLoaderConfig& config) {
    TCacheAwareHttpClient::TOptions options(config.GetUrl());
    if (const auto& socketTimeout = config.GetSocketTimeout()) {
        options.SocketTimeout(TDuration::Parse(socketTimeout));
    }
    if (const auto& connectTimeout = config.GetConnectTimeout()) {
        options.ConnectTimeout(TDuration::Parse(connectTimeout));
    }
    return options;
}

TString HttpUserAgent() {
    return TString("SolomonAgent/") + GetArcadiaLastChange() + " " PLATFORM_ID;
}

///////////////////////////////////////////////////////////////////////////////
// THttpConfigLoader
///////////////////////////////////////////////////////////////////////////////
class THttpConfigLoader final: public IServiceConfigLoader {
public:
    THttpConfigLoader(const THttpLoaderConfig& config)
        : UpdateInterval_(TDuration::Parse(config.GetUpdateInterval()))
        , UrlPath_(GetPathAndQuery(config.GetUrl(), true))
        , Format_(config.GetFormat())
        , UserAgent_(HttpUserAgent())
        , HttpClient_(HttpClientOpts(config))
    {
        SA_LOG(DEBUG) << Name() << ": start { " << config.ShortDebugString() << '}';
    }

    ~THttpConfigLoader() {
        SA_LOG(DEBUG) << Name() << ": stop";
    }

private:
    TStringBuf Name() const override {
        return TStringBuf("HttpConfigLoader");
    }

    TDuration UpdateInterval() const override {
        return UpdateInterval_;
    }

    TVector<TServiceConfig> Load() override {
        SA_LOG(DEBUG) << Name() << ": load";

        TCacheAwareHttpClient::THeaders headers;
        headers.emplace(TStringBuf("User-Agent"), UserAgent_);
        if (LastLoad_ != TInstant::Zero()) {
            headers.emplace(
                    TStringBuf("If-Modified-Since"),
                    FormatHttpDate(LastLoad_.TimeT()));
        }
        LastLoad_ = TInstant::Now();

        TTempBufOutput tempBuf(256);
        HttpClient_.DoGet(UrlPath_, &tempBuf, headers);

        if (tempBuf.Filled() != 0) {
            if (Format_ == EConfigFormat::JSON) {
                return ParseJson(tempBuf);
            }

            // TODO: add proto bin and text parsing
        }

        return {};
    }

    TVector<TServiceConfig> ParseJson(const TTempBuf& tempBuf) {
        NJson::TJsonValue jsonValue;

        {
            TMemoryInput in(tempBuf.Data(), tempBuf.Filled());
            NJson::TJsonReaderConfig readerConfig;
            readerConfig.DontValidateUtf8 = true;
            NJson::ReadJsonTree(&in, &readerConfig, &jsonValue, true);
        }

        TVector<TServiceConfig> configs;
        if (jsonValue.IsArray()) {
            for (const NJson::TJsonValue& jsonItem: jsonValue.GetArray()) {
                configs.emplace_back();
                NProtobufJson::Json2Proto(jsonItem, configs.back());
            }
        } else if (jsonValue.IsMap()) {
            configs.emplace_back();
            NProtobufJson::Json2Proto(jsonValue, configs.back());
        } else {
            static const int MaxSize = 100;
            TStringBuf result(tempBuf.Data(), tempBuf.Filled());
            ythrow yexception() << "expected to get JSON map or list, but got: "
                                << result.Head(MaxSize)
                                << (result.size() >= MaxSize ? " ..." : "");
        }

        return configs;
    }

private:
    const TDuration UpdateInterval_;
    const TString UrlPath_;
    const EConfigFormat Format_;
    const TString UserAgent_;
    TCacheAwareHttpClient HttpClient_;
    TInstant LastLoad_ = TInstant::Zero();
};

} // namespace

IServiceConfigLoaderPtr CreateHttpLoader(const THttpLoaderConfig& config) {
    return IServiceConfigLoaderPtr(new THttpConfigLoader(config));
}

} // namespace NAgent
} // namespace NSolomon
