#include "config_loader.h"

#include <solomon/agent/lib/python2/code_module.h>
#include <solomon/agent/lib/python2/error.h>
#include <solomon/agent/lib/python2/py_string.h>
#include <solomon/agent/lib/python2/gil.h>
#include <solomon/agent/lib/python2/logger.h>
#include <solomon/agent/lib/python2/cast_proto.h>
#include <solomon/agent/lib/python2/types.h>
#include <solomon/agent/misc/logger.h>
#include <solomon/agent/protos/loader_config.pb.h>
#include <solomon/agent/protos/python2_config.pb.h>


namespace NSolomon {
namespace NAgent {
namespace {

TVector<TServiceConfig> CastServiceConfigs(const PyObject* list) {
    TVector<TServiceConfig> result;
    Py_ssize_t size = PyList_GET_SIZE(list);
    for (Py_ssize_t i = 0; i < size; ++i) {
        const PyObject* item = PyList_GET_ITEM(list, i);
        result.emplace_back();
        NPython2::TProtoConverter(&result.back())
                .Convert(item);
    }
    return result;
}

class TPython2ConfigLoader final: public IServiceConfigLoader {
public:
    TPython2ConfigLoader(
            const TPython2LoaderConfig& config,
            const TGlobalPython2Config& pyConfig)
        : ClassName_(config.GetClassName())
        , UpdateInterval_(TDuration::Parse(config.GetUpdateInterval()))
    {
        NPython2::TGilGuard gil;

        auto moduleCode = NPython2::LoadPyModule(
                    config.GetModuleName(),
                    config.GetFilePath(),
                    pyConfig.GetIgnorePycFiles());

        // (1) check class
        NPython2::TObjectPtr moduleClass = PyObject_GetAttrString(
                moduleCode.Get(), ClassName_.c_str());
        PY_ENSURE(moduleClass,
                "cannot find class '" << ClassName_
                << "' in " << config.GetFilePath());
        PY_ENSURE_TYPE(NPython2::IsPythonClass, moduleClass.Get(),
                "expected to get a class by name '" << ClassName_
                << "' in " << config.GetFilePath());

        // (2) create an instance
        NPython2::TObjectPtr initArgs = PyTuple_New(1);
        PyTuple_SET_ITEM(initArgs.Get(), 0, NPython2::GetLogger().Release());

        NPython2::TObjectPtr initKwArgs;
        if (config.ParamsSize() > 0) {
            initKwArgs.ResetSteal(PyDict_New());
            for (const auto& it: config.GetParams()) {
                NPython2::TObjectPtr key = NPython2::ToPyString(it.first);
                NPython2::TObjectPtr value = NPython2::ToPyString(it.second);
                int rc = PyDict_SetItem(initKwArgs.Get(), key.Get(), value.Get());
                PY_ENSURE(rc == 0, "cannot insert item into init kwargs for " << ClassName_);
            }
        }

        NPython2::TObjectPtr pyModule = PyObject_Call(moduleClass.Get(), initArgs.Get(), initKwArgs.Get());
        PY_ENSURE(pyModule,
                "cannot create instance of " << ClassName_ << " class"
                 << ", python error: " << NPython2::LastErrorAsString());

        // (3) check load() method
        NPython2::TObjectPtr pyLoadMethod = PyObject_GetAttrString(pyModule.Get(), "load");

        PY_ENSURE(pyLoadMethod,
                "cannot find method load() in class " << ClassName_);
        PY_ENSURE(PyMethod_Check(pyLoadMethod.Get()),
                "expected to get a method load() but class " << ClassName_
                << " contains " << NPython2::TypeName(pyLoadMethod.Get())
                << " with this name");

        pyModule.Swap(PyModule_);
        pyLoadMethod.Swap(PyLoadMethod_);

        SA_LOG(DEBUG) << "start " << Name() << " config loader";
    }

    ~TPython2ConfigLoader() {
        // predictable destroy python objects under GIL
        NPython2::TGilGuard gil;
        PyLoadMethod_.Reset();
        PyModule_.Reset();

        SA_LOG(DEBUG) << "stop " << Name() << " config loader";
    }

private:
    TStringBuf Name() const override {
        return ClassName_;
    }

    TDuration UpdateInterval() const override {
        return UpdateInterval_;
    }

    TVector<TServiceConfig> Load() override {
        SA_LOG(DEBUG) << "load configs from " << Name();

        NPython2::TGilGuard gil;

        NPython2::TObjectPtr args = PyTuple_New(0);
        NPython2::TObjectPtr res =
                PyObject_Call(PyLoadMethod_.Get(), args.Get(), nullptr);

        PY_ENSURE(res,
            "cannot load configs from " << Name() << ": "
            << NPython2::LastErrorAsString());

        PY_ENSURE_TYPE(PyList_Check, res.Get(),
            "load() method of " << Name() << " must return List");

        return CastServiceConfigs(res.Get());
    }

private:
    const TString ClassName_;
    const TDuration UpdateInterval_;
    NPython2::TObjectPtr PyModule_;
    NPython2::TObjectPtr PyLoadMethod_;
};

} // namespace

IServiceConfigLoaderPtr CreatePython2Loader(
        const TPython2LoaderConfig& config,
        const TGlobalPython2Config& pyConfig)
{
    return IServiceConfigLoaderPtr(new TPython2ConfigLoader(config, pyConfig));
}

} // namespace NAgent
} // namespace NSolomon
