#include <solomon/agent/lib/config/config_loader.h>

#include <solomon/agent/protos/loader_config.pb.h>
#include <solomon/agent/protos/python2_config.pb.h>

#include <library/cpp/resource/resource.h>
#include <library/cpp/testing/gtest/gtest.h>

#include <util/string/cast.h>
#include <util/system/event.h>

#include <google/protobuf/text_format.h>
#include <google/protobuf/util/message_differencer.h>

using namespace NSolomon;
using namespace NAgent;

using google::protobuf::TextFormat;
using google::protobuf::util::MessageDifferencer;

TServiceConfig CreateTestServiceConfig(int number, const TString& labels) {
    auto numberStr = ToString<int>(number);

    TServiceConfig config;
    config.SetProject("project" + numberStr);
    config.SetService("service" + numberStr);
    config.AddLabels(labels);
    config.SetPullInterval("2s");
    config.AddModules();

    return config;
}

TServiceConfig CreateBigServiceConfig() {
    TServiceConfig config;

    // SOLOMON-4682
    google::protobuf::string s(NResource::Find("/big_service_config.conf"));

    TextFormat::ParseFromString(s, &config);
    return config;
}

///////////////////////////////////////////////////////////////////////////////
// TTestLoader
///////////////////////////////////////////////////////////////////////////////
class TTestLoader: public IServiceConfigLoader {
public:
    void AddConfig(const TServiceConfig& config) {
        with_lock (Lock_) {
            Configs_.push_back(config);
        }
    }

    void RemoveFirstConfig() {
        with_lock (Lock_) {
            if (!Configs_.empty()) {
                Configs_.erase(Configs_.begin());
            }
        }
    }

private:
    TStringBuf Name() const override {
        return "TTestLoader";
    }

    TDuration UpdateInterval() const override {
        return TDuration::MilliSeconds(10);
    }

    TVector<TServiceConfig> Load() override {
        with_lock (Lock_) {
            return Configs_;
        }
    }

private:
    TAdaptiveLock Lock_;
    TVector<TServiceConfig> Configs_;
};

///////////////////////////////////////////////////////////////////////////////
// TTestWatcher
///////////////////////////////////////////////////////////////////////////////
class TTestWatcher: public IServiceConfigWatcher {
public:
    const TServiceConfig& WaitAdded() {
        AddEvent_.WaitT(TDuration::Seconds(5));
        return LastAdded_;
    }

    const TServiceConfig& WaitRemoved() {
        RemoveEvent_.WaitT(TDuration::Seconds(5));
        return LastRemoved_;
    }

    const std::pair<TServiceConfig, TServiceConfig>& WaitChanged() {
        ChangeEvent_.WaitT(TDuration::Seconds(5));
        return LastChanged_;
    }

private:
    void OnAdded(const TServiceConfig& config) override {
        Cerr << "Added config: " << config.ShortDebugString() << Endl;
        LastAdded_ = config;
        AddEvent_.Signal();
    }

    void OnRemoved(const TServiceConfig& config) override {
        Cerr << "Removed config: " << config.ShortDebugString() << Endl;
        LastRemoved_ = config;
        RemoveEvent_.Signal();
    }

    void OnChanged(
            const TServiceConfig& oldConfig,
            const TServiceConfig& newConfig) override
    {
        Cerr << "Changed config: "
             << oldConfig.ShortDebugString() << " => "
             << newConfig.ShortDebugString() << Endl;
        LastChanged_ = std::make_pair(oldConfig, newConfig);
        ChangeEvent_.Signal();
    }

private:
    TAutoEvent AddEvent_;
    TAutoEvent RemoveEvent_;
    TAutoEvent ChangeEvent_;

    TServiceConfig LastAdded_;
    TServiceConfig LastRemoved_;
    std::pair<TServiceConfig, TServiceConfig> LastChanged_;
};

bool IsConfigInitialized(const TServiceConfig& config) {
    auto fd = config.GetDescriptor()->FindFieldByName("Project");
    return config.GetMetadata().reflection->HasField(config, fd);
}

///////////////////////////////////////////////////////////////////////////////
// TConfigLoaderTest
///////////////////////////////////////////////////////////////////////////////
class TConfigLoaderTest: public ::testing::Test {
public:
    TGlobalPython2Config PyConfig;
    TConfigLoaderConfig EmptyConfig;
};

TEST_F(TConfigLoaderTest, ServiceConfigKey) {
    TServiceConfig serviceConfig;
    serviceConfig.SetProject("project");
    serviceConfig.SetService("service");
    TString key = NAgent::ServiceConfigKey(serviceConfig);
    ASSERT_EQ(key, "project;service");
}

TEST_F(TConfigLoaderTest, WatchingAdded) {
    auto loader = MakeIntrusive<TTestLoader>();
    auto watcher = MakeIntrusive<TTestWatcher>();

    TConfigLoader configLoader(EmptyConfig, PyConfig);
    configLoader.AddLoader(loader);
    configLoader.AddWatcher(watcher);
    configLoader.Start();

    auto config = CreateTestServiceConfig(1, "label=value");
    loader->AddConfig(config);

    const auto& addedConfig = watcher->WaitAdded();
    ASSERT_TRUE(MessageDifferencer::Equals(config, addedConfig));

    configLoader.Stop();
}

TEST_F(TConfigLoaderTest, WatchingRemoved) {
    auto loader = MakeIntrusive<TTestLoader>();
    auto watcher = MakeIntrusive<TTestWatcher>();

    TConfigLoader configLoader(EmptyConfig, PyConfig);
    configLoader.AddLoader(loader);
    configLoader.AddWatcher(watcher);
    configLoader.Start();

    auto firstConfig = CreateTestServiceConfig(1, "label=value");
    loader->AddConfig(firstConfig);

    const auto& addedConfig1 = watcher->WaitAdded();
    ASSERT_TRUE(MessageDifferencer::Equals(firstConfig, addedConfig1));

    auto secondConfig = CreateTestServiceConfig(2, "label=value");
    loader->AddConfig(secondConfig);

    const auto& addedConfig2 = watcher->WaitAdded();
    ASSERT_TRUE(MessageDifferencer::Equals(secondConfig, addedConfig2));

    {
        loader->RemoveFirstConfig();
        const auto& removedConfig = watcher->WaitRemoved();
        ASSERT_TRUE(MessageDifferencer::Equals(firstConfig, removedConfig));
    }
    {
        loader->RemoveFirstConfig();
        const auto& removedConfig = watcher->WaitRemoved();
        ASSERT_TRUE(MessageDifferencer::Equals(secondConfig, removedConfig));
    }

    configLoader.Stop();
}

TEST_F(TConfigLoaderTest, WatchingChanged) {
    auto loader = MakeIntrusive<TTestLoader>();
    auto watcher = MakeIntrusive<TTestWatcher>();

    TConfigLoader configLoader(EmptyConfig, PyConfig);
    configLoader.AddLoader(loader);
    configLoader.AddWatcher(watcher);
    configLoader.Start();

    auto config = CreateTestServiceConfig(1, "label=value");
    loader->AddConfig(config);

    const auto& addedConfig = watcher->WaitAdded();
    ASSERT_TRUE(MessageDifferencer::Equals(config, addedConfig));

    auto newConfig = CreateTestServiceConfig(1, "newLabel=newValue");
    loader->AddConfig(newConfig);

    const auto& changedConfig = watcher->WaitChanged();
    ASSERT_TRUE(MessageDifferencer::Equals(config, changedConfig.first));
    ASSERT_TRUE(MessageDifferencer::Equals(newConfig, changedConfig.second));

    configLoader.Stop();
}

TEST_F(TConfigLoaderTest, WatchingNoChanges) {
    auto loader = MakeIntrusive<TTestLoader>();
    auto watcher = MakeIntrusive<TTestWatcher>();

    TConfigLoader configLoader(EmptyConfig, PyConfig);
    configLoader.AddLoader(loader);
    configLoader.AddWatcher(watcher);
    configLoader.Start();

    auto config = CreateBigServiceConfig();
    loader->AddConfig(config);

    const auto& addedConfig = watcher->WaitAdded();
    ASSERT_TRUE(MessageDifferencer::Equals(config, addedConfig));

    auto newConfig = CreateBigServiceConfig();
    loader->AddConfig(newConfig);

    const auto& changedConfig = watcher->WaitChanged();
    // No changes
    ASSERT_FALSE(IsConfigInitialized(changedConfig.first));
    ASSERT_FALSE(IsConfigInitialized(changedConfig.second));

    configLoader.Stop();
}
