#include <solomon/agent/lib/config/config_loader.h>
#include <solomon/agent/protos/loader_config.pb.h>

#include <library/cpp/http/server/http_ex.h>
#include <library/cpp/testing/common/network.h>
#include <library/cpp/testing/gtest/gtest.h>

#include <google/protobuf/util/message_differencer.h>

using namespace NSolomon;
using namespace NAgent;
using google::protobuf::util::MessageDifferencer;

///////////////////////////////////////////////////////////////////////////////
// TTestHttpServer
///////////////////////////////////////////////////////////////////////////////
class TTestHttpServer: public THttpServer, public THttpServer::ICallBack {
public:
    TTestHttpServer(ui16 port, const TServiceConfig& response)
        : THttpServer(this, TOptions(port).AddBindAddress("localhost"))
        , Response_(response)
    {
    }

private:
    TClientRequest* CreateClient() override {
        struct TRequest: public THttpClientRequestEx {
            TTestHttpServer* Parent_ = nullptr;

            TRequest(TTestHttpServer* parent): Parent_(parent) {}

            bool Reply(void* /*tsr*/) override {
                Y_ENSURE(ProcessHeaders(), "cannot parse headers");
                Y_ENSURE(RD.HeaderIn("User-Agent") != nullptr, "no User-Agent header");

                if (const TString* lms = RD.HeaderIn("If-Modified-Since")) {
                    TInstant::ParseHttpDeprecated(*lms); // check date is parseable
                    // assume that configs are not modified
                    Output() << "HTTP/1.1 304 Not Modified\r\n\r\n";
                    return true;
                }

                Output() << "HTTP/1.1 200 Ok\r\n\r\n";
                if (RD.ScriptName() == TStringBuf("/configs.json")) {
                    Output() << '[' << Parent_->Response_.AsJSON() << ']';
                    return true;
                }

                ythrow yexception() << "unknown path: " << RD.ScriptName();
            }
        };

        return new TRequest(this);
    }

private:
    const TServiceConfig& Response_;
};

///////////////////////////////////////////////////////////////////////////////
// THttpConfigLoaderTest
///////////////////////////////////////////////////////////////////////////////
class TTestServiceConfig: public TServiceConfig {
public:
    TTestServiceConfig() {
        SetProject("solomon");
        SetService("stockpile");
        SetPullInterval("10s");
        AddLabels("test=label");

        {
            TPullModuleConfig* module = AddModules();
            TPullPython2Config* python = module->MutablePython2();
            python->SetFilePath("/some/path/to/file.py");
            python->SetModuleName("test");
            python->SetClassName("TestPullModule");
        }
    }
};

TEST(THttpConfigLoaderTest, LoadJson) {
    TTestServiceConfig testConfig;

    auto port = NTesting::GetFreePort();
    TTestHttpServer httpServer(port, testConfig);
    httpServer.Start();

    IServiceConfigLoaderPtr loader;
    {
        THttpLoaderConfig loaderConfig;
        loaderConfig.SetUpdateInterval("10s");
        loaderConfig.SetUrl("http://localhost:" + ToString<ui16>(port) + "/configs.json");
        loaderConfig.SetFormat(EConfigFormat::JSON);
        loaderConfig.SetConnectTimeout("10s");
        loaderConfig.SetSocketTimeout("30s");

        loader = CreateHttpLoader(loaderConfig);
    }

    // first get without If-Modified-Since header
    {
        TVector<TServiceConfig> configs = loader->Load();
        ASSERT_EQ(configs.size(), 1u);

        const TServiceConfig& config = configs.front();
        ASSERT_TRUE(MessageDifferencer::Equals(testConfig, config));
    }

    // second get with If-Modified-Since header
    {
        TVector<TServiceConfig> configs = loader->Load();
        ASSERT_EQ(configs.size(), 0u);
    }
}
