#include <solomon/agent/lib/config/config_loader.h>
#include <solomon/agent/lib/python2/code_module.h>
#include <solomon/agent/lib/python2/initializer.h>
#include <solomon/agent/protos/loader_config.pb.h>
#include <solomon/agent/protos/python2_config.pb.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/system/tempfile.h>

using namespace NSolomon;
using namespace NAgent;

NPython2::TInitializer* Initializer = NPython2::TInitializer::Instance();

TStringBuf TestPy = R"(
class MyLoader:
    def __init__(self, logger, p1='1', p2='2', **params):
        assert p1 == '111'
        assert p2 == '2'
        assert 'p3' in params
        assert params['p3'] == '333'

    def load(self):
        return [{
            'Project': 'solomon',
            'Service': 'stockpile',
            'Labels': [
                'my=label1',
                'my=label2',
            ],
            'PullInterval': '15s',
            'Modules': [
                {'System': {
                }},
                {'Python2': {
                    'FilePath': '/some/path/to/file.py',
                    'ModuleName': 'test',
                    'ClassName': 'MyPullModule',
                    'Params': {
                        'p1': 'one',
                        'p2': 'two',
                        'p3': 'three',
                    },
                }},
                {
                    'System': {
                    },
                    'PullInterval': '5s',
                },
            ],
        }]
)";

TEST(TPy2ConfigLoader, Load) {
    IServiceConfigLoaderPtr loader;
    {
        TTempFileHandle tempFile;
        tempFile.Write(TestPy.data(), TestPy.size());
        tempFile.FlushData();

        TPython2LoaderConfig config;
        config.SetUpdateInterval("10s");
        config.SetFilePath(tempFile.Name());
        config.SetModuleName("test");
        config.SetClassName("MyLoader");
        auto* params = config.MutableParams();
        params->insert({ "p1", "111"});
        params->insert({ "p3", "333"});

        TGlobalPython2Config pyConfig;
        loader = CreatePython2Loader(config, pyConfig);
    }

    TVector<TServiceConfig> configs = loader->Load();
    ASSERT_EQ(configs.size(), 1u);

    // (1) project & service
    const TServiceConfig& config = configs.front();
    ASSERT_EQ(config.GetProject(), "solomon");
    ASSERT_EQ(config.GetService(), "stockpile");

    // (2) labels
    ASSERT_EQ(config.LabelsSize(), 2u);
    ASSERT_EQ(config.GetLabels(0), "my=label1");
    ASSERT_EQ(config.GetLabels(1), "my=label2");

    // (3) other options
    ASSERT_EQ(config.GetPullInterval(), "15s");

    // (4) modules
    ASSERT_EQ(config.ModulesSize(), 3u);

    // (4.a) system module
    {
        const TPullModuleConfig& module = config.GetModules(0);
        ASSERT_TRUE(module.HasSystem());
    }

    // (4.b) python2 module
    {
        const TPullModuleConfig& module = config.GetModules(1);
        ASSERT_TRUE(module.HasPython2());

        const TPullPython2Config& python2 = module.GetPython2();
        ASSERT_EQ(python2.GetFilePath(), "/some/path/to/file.py");
        ASSERT_EQ(python2.GetModuleName(), "test");
        ASSERT_EQ(python2.GetClassName(), "MyPullModule");

        const auto& params = python2.GetParams();
        ASSERT_EQ(params.size(), 3u);
        ASSERT_EQ(params.at("p1"), "one");
        ASSERT_EQ(params.at("p2"), "two");
        ASSERT_EQ(params.at("p3"), "three");
    }

    // (4.c) system module with defined PullInterval
    {
        const TPullModuleConfig& module = config.GetModules(2);
        ASSERT_TRUE(module.HasSystem());
        ASSERT_EQ(module.GetPullInterval(), "5s");
    }
}
