#pragma once

#include "grouping_encoder_states.h"

#include <solomon/agent/misc/labels.h>
#include <solomon/agent/misc/logger.h>
#include <solomon/libs/cpp/error_or/error_or.h>

#include <library/cpp/monlib/encode/encoder.h>
#include <library/cpp/monlib/encode/encoder_state.h>
#include <library/cpp/monlib/encode/json/json.h>
#include <library/cpp/monlib/encode/spack/spack_v1.h>
#include <library/cpp/monlib/metrics/labels.h>

#include <util/datetime/base.h>
#include <util/digest/multi.h>
#include <util/generic/string.h>
#include <util/generic/vector.h>


namespace NSolomon::NAgent {

class TShardKey {
public:
    const TString Project;
    const TString Cluster;
    const TString Service;

    TShardKey() = delete;
    TShardKey(const TString& project, const TString& cluster, const TString& service)
        : Project{project}
        , Cluster{cluster}
        , Service{service}
    {
    }

    bool operator==(const TShardKey& other) const {
        return std::tie(Project, Cluster, Service) == std::tie(other.Project, other.Cluster, other.Service);
    }
};

struct TShardKeyHasher {
    ui64 operator()(const TShardKey& shardKey) const {
        return MultiHash(shardKey.Project, shardKey.Cluster, shardKey.Service);
    }
};

using TShardKeyOrError = TErrorOr<TShardKey, TString>;

class TRemovableLabels: public TAgentLabels {
public:
    using TBase = TAgentLabels;
    using TBase::TBase;

    void RemoveAt(size_t idx) {
        auto& v = AsVector();
        std::swap(v[idx], v.back());
        v.pop_back();
    }
};

class TShardKeySubstitutions {
public:
    TShardKeySubstitutions()
        : IsProjectValueATemplate_{false}
        , IsClusterValueATemplate_{false}
        , IsServiceValueATemplate_{false}
        , DoNotAppendHostLabel_{false}
    {}

    TShardKeySubstitutions(const TShardKeySubstitutions& other) {
        CopyFrom(other);
    }

    TShardKeySubstitutions& operator=(const TShardKeySubstitutions& other) {
        CopyFrom(other);
        return *this;
    }

    TShardKeySubstitutions(
            const TString& project,
            const TString& cluster,
            const TString& service,
            bool doNotAppendHostLabel = false)
    {
        Y_ENSURE(!project.empty(), "Project parameter cannot be empty in a TShardKeySubstitutions");
        Y_ENSURE(!cluster.empty(), "Cluster parameter cannot be empty in a TShardKeySubstitutions");
        Y_ENSURE(!service.empty(), "Service parameter cannot be empty in a TShardKeySubstitutions");

        std::tie(Project_, IsProjectValueATemplate_) =  ProcessTemplateValue(project);
        std::tie(Cluster_, IsClusterValueATemplate_) =  ProcessTemplateValue(cluster);
        std::tie(Service_, IsServiceValueATemplate_) =  ProcessTemplateValue(service);

        DoNotAppendHostLabel_ = doNotAppendHostLabel;
    }

    TShardKeyOrError Substitute(TRemovableLabels& labels) const {
        TString project = IsProjectValueATemplate_ ? TString() : Project_;
        TString cluster = IsClusterValueATemplate_ ? TString() : Cluster_;
        TString service = IsServiceValueATemplate_ ? TString() : Service_;
        bool isHostPresent = false;

        for (size_t i = 0; i < labels.size();) {
            if (!project.empty() && !cluster.empty() && !service.empty() && isHostPresent) {
                break;
            }

            TAgentLabel& label = labels[i];
            TStringBuf labelName = label.Name();

            if (IsProjectValueATemplate_ && labelName == Project_) {
                project = label.Value();
            } else if (IsClusterValueATemplate_ && labelName == Cluster_) {
                cluster = label.Value();
            } else if (IsServiceValueATemplate_ && labelName == Service_) {
                service = label.Value();
            } else {
                if (labelName == TStringBuf("host")) {
                    isHostPresent = true;
                }

                ++i;
                continue;
            }

            labels.RemoveAt(i);
        }

        TString errMsg;

        for (auto& tuple: {
            std::forward_as_tuple(project, Project_),
            std::forward_as_tuple(cluster, Cluster_),
            std::forward_as_tuple(service, Service_)
        }) {
            const auto& part = std::get<0>(tuple);
            const auto& label = std::get<1>(tuple);
            if (part.empty()) {
                errMsg += errMsg.empty() ? "" : ", ";
                errMsg += "could not find label \"";
                errMsg += label;
                errMsg += "\"";
            }
        }

        if (!errMsg.empty()) {
            errMsg = TStringBuilder() << "could not construct a new shard key value: " << errMsg;
            return errMsg;
        }

        // SOLOMON-5277
        if (!isHostPresent && DoNotAppendHostLabel_) {
            labels.Add(TStringBuf("host"), "");
        }

        return TShardKey{project, cluster, service};
    }

    explicit operator bool() const noexcept {
        return !Project_.empty() && !Cluster_.empty() && !Service_.empty();
    }

private:
    void CopyFrom(const TShardKeySubstitutions& other) {
        Project_ = other.Project_;
        IsProjectValueATemplate_ = other.IsProjectValueATemplate_;
        Cluster_ = other.Cluster_;
        IsClusterValueATemplate_ = other.IsClusterValueATemplate_;
        Service_ = other.Service_;
        IsServiceValueATemplate_ = other.IsServiceValueATemplate_;

        DoNotAppendHostLabel_ = other.DoNotAppendHostLabel_;
    }

    std::pair<TString, bool> ProcessTemplateValue(const TString& labelValue) {
        TString result;
        bool isValueATemplate;

        TStringBuf templateValue = labelValue;
        if (templateValue.SkipPrefix("{{") && templateValue.ChopSuffix("}}")) {
            isValueATemplate = true;
            result = templateValue;
        } else {
            isValueATemplate = false;
            result = labelValue;
        }

        return {result, isValueATemplate};
    }

private:
    TString Project_;
    bool IsProjectValueATemplate_;

    TString Cluster_;
    bool IsClusterValueATemplate_;

    TString Service_;
    bool IsServiceValueATemplate_;

    bool DoNotAppendHostLabel_{false};
};

using TShardsData = TVector<std::pair<const TShardKey, TString>>;
/**
 * A proxy that groups all metrics by a resulting shard key to corresponding encoders.
 * In the end, N binary strings should be constructed for each shard data
 */
class TGroupingEncoder final: public NMonitoring::IMetricEncoder {
public:
    using TEncoderFactory = std::function<NMonitoring::IMetricEncoderPtr(const TShardKey&, IOutputStream*)>;
    using TEncoderState = NMonitoring::TEncoderStateImpl<EGroupingEncoderState>;

    explicit TGroupingEncoder(
            const TShardKeySubstitutions& shardKeySubstitutions,
            TEncoderFactory&& factory,
            TMaybe<TAgentLabels> commonLabels = Nothing())
        : ShardKeySubstitutions_{shardKeySubstitutions}
        , Factory_{std::move(factory)}
    {
        if (commonLabels) {
            CommonLabels_ = commonLabels.GetRef();
        }
    }

private:
    void OnStreamBegin() override {
        State_.Expect(TEncoderState::EState::ROOT);
        // Nothing to do here, since a corresponding consumer is not selected yet
    }

    void OnMetricBegin(NMonitoring::EMetricType type) override {
        State_.Switch(TEncoderState::EState::ROOT, TEncoderState::EState::METRIC);
        Type_ = type;
        // Nothing to do here, since a corresponding consumer is not selected yet
    }

    void OnLabelsBegin() override {
        if (State_ == TEncoderState::EState::METRIC) {
            State_ = TEncoderState::EState::METRIC_LABELS;
        } else if (State_ == TEncoderState::EState::ROOT) {
            State_ = TEncoderState::EState::COMMON_LABELS;
        } else {
            State_.ThrowInvalid("expected METRIC or ROOT");
        }
    }

    void OnLabel(TStringBuf name, TStringBuf value) override {
        TAgentLabels* labels;

        if (State_ == TEncoderState::EState::METRIC_LABELS) {
            labels = &Labels_;
        } else if (State_ == TEncoderState::EState::COMMON_LABELS) {
            labels = &CommonLabels_;
        } else {
            State_.ThrowInvalid("expected METRIC_LABELS or COMMON_LABELS");
        }

        labels->Add(name, value);
    }

    void ResetMetricState() {
        Type_ = NMonitoring::EMetricType::UNKNOWN;
        Labels_.clear();
    }

    void OnLabelsEnd() override {
        if (State_ == TEncoderState::EState::COMMON_LABELS) {
            State_ = TEncoderState::EState::ROOT;
            return;
        } else if (State_ != TEncoderState::EState::METRIC_LABELS) {
            State_.ThrowInvalid("expected METRIC_LABELS or COMMON_LABELS");
        }

        State_.Switch(TEncoderState::EState::METRIC_LABELS, TEncoderState::EState::METRIC);

        for (auto& cl: CommonLabels_) {
            Labels_.Add(cl);
        }

        TShardKeyOrError shardKeyOrError = ShardKeySubstitutions_.Substitute(Labels_);

        if (shardKeyOrError.Fail()) {
            TString labelsRepr = "{";
            for (size_t i = 0; i != Labels_.Size(); ++i) {
                auto& label = Labels_[i];

                labelsRepr += label.Name();
                labelsRepr += "=";
                labelsRepr += label.Value();
                if (i < Labels_.Size() - 1) {
                    labelsRepr += ", ";
                }
            }
            labelsRepr += "}";

            SA_LOG(WARN) << "skipping a metric " << labelsRepr << ": " << shardKeyOrError.Error();

            State_.Switch(TEncoderState::EState::METRIC, TEncoderState::EState::SKIP_METRIC);
            ResetMetricState();

            return;
        } else {
            CurrentEncoder_ = GetOrConstructAnEncoder(shardKeyOrError.Value());
        }

        CurrentEncoder_->OnMetricBegin(Type_);
        CurrentEncoder_->OnLabelsBegin();

        for (auto&& label: Labels_) {
            CurrentEncoder_->OnLabel(label.Name(), label.Value());
        }

        CurrentEncoder_->OnLabelsEnd();
    }

    void OnDouble(TInstant time, double value) override {
        if (State_ == TEncoderState::EState::SKIP_METRIC) {
            return;
        }

        State_.Expect(TEncoderState::EState::METRIC);
        Y_ENSURE(CurrentEncoder_, "No encoder was specified before processing the data");

        CurrentEncoder_->OnDouble(time, value);
    }

    void OnInt64(TInstant time, i64 value) override {
        if (State_ == TEncoderState::EState::SKIP_METRIC) {
            return;
        }

        State_.Expect(TEncoderState::EState::METRIC);
        Y_ENSURE(CurrentEncoder_, "No encoder was specified before processing the data");

        CurrentEncoder_->OnInt64(time, value);
    }

    void OnUint64(TInstant time, ui64 value) override {
        if (State_ == TEncoderState::EState::SKIP_METRIC) {
            return;
        }

        State_.Expect(TEncoderState::EState::METRIC);
        Y_ENSURE(CurrentEncoder_, "No encoder was specified before processing the data");

        CurrentEncoder_->OnUint64(time, value);
    }

    void OnHistogram(TInstant time, NMonitoring::IHistogramSnapshotPtr snapshot) override {
        if (State_ == TEncoderState::EState::SKIP_METRIC) {
            return;
        }

        State_.Expect(TEncoderState::EState::METRIC);
        Y_ENSURE(CurrentEncoder_, "No encoder was specified before processing the data");

        CurrentEncoder_->OnHistogram(time, std::move(snapshot));
    }

    void OnSummaryDouble(TInstant time, NMonitoring::ISummaryDoubleSnapshotPtr snapshot) override {
        if (State_ == TEncoderState::EState::SKIP_METRIC) {
            return;
        }

        State_.Expect(TEncoderState::EState::METRIC);
        Y_ENSURE(CurrentEncoder_, "No encoder was specified before processing the data");

        CurrentEncoder_->OnSummaryDouble(time, std::move(snapshot));
    }

    void OnLogHistogram(TInstant time, NMonitoring::TLogHistogramSnapshotPtr snapshot) override {
        if (State_ == TEncoderState::EState::SKIP_METRIC) {
            return;
        }
        State_.Expect(TEncoderState::EState::METRIC);
        Y_ENSURE(CurrentEncoder_, "No encoder was specified before processing the data");

        CurrentEncoder_->OnLogHistogram(time, std::move(snapshot));
    }

    void OnMetricEnd() override {
        if (State_ == TEncoderState::EState::SKIP_METRIC) {
            State_ = TEncoderState::EState::ROOT;
            return;
        }

        State_.Switch(TEncoderState::EState::METRIC, TEncoderState::EState::ROOT);
        CurrentEncoder_->OnMetricEnd();

        // Clear all metric related variables
        CurrentEncoder_ = nullptr;
        ResetMetricState();
    }

    void OnStreamEnd() override {
        State_.Expect(TEncoderState::EState::ROOT);

        for (auto& it: EncoderDataByShardKey_) {
            it.second.Encoder->OnStreamEnd();
        }
    }

    void Close() override {
        for (auto& it: EncoderDataByShardKey_) {
            it.second.Encoder->Close();
        }
    }

    void OnCommonTime(TInstant commonTime) override {
        CommonTime_ = commonTime;
    }

    NMonitoring::IMetricEncoder* GetOrConstructAnEncoder(const TShardKey& shardKey) {
        auto it = EncoderDataByShardKey_.find(shardKey);

        if (it == EncoderDataByShardKey_.end()) {
            std::tie(it, std::ignore) = EncoderDataByShardKey_.emplace(
                    std::piecewise_construct,
                    std::forward_as_tuple(shardKey),
                    std::forward_as_tuple(shardKey, Factory_)
            );
            auto* encoder = it->second.Encoder.Get();

            encoder->OnStreamBegin();
            if (CommonTime_) {
                encoder->OnCommonTime(CommonTime_);
            }
        }

        return it->second.Encoder.Get();
    }

public:
    TShardsData GetShardsData() {
        TShardsData data(::Reserve(EncoderDataByShardKey_.size()));

        for (auto& it: EncoderDataByShardKey_) {
            data.emplace_back(it.first, it.second.DataStream.Str());
        }

        return data;
    }

private:
    const TShardKeySubstitutions& ShardKeySubstitutions_;

    TEncoderState State_;
    TInstant CommonTime_ = TInstant::Zero();

    // Metric related
    NMonitoring::EMetricType Type_ = NMonitoring::EMetricType::UNKNOWN;
    TRemovableLabels Labels_;
    TAgentLabels CommonLabels_;
    NMonitoring::IMetricEncoder* CurrentEncoder_ = nullptr;

    struct TEncoderData {
        TEncoderData(const TShardKey& shardKey, TEncoderFactory factory) {
            Encoder = factory(shardKey, &DataStream);
        }

        TStringStream DataStream;
        NMonitoring::IMetricEncoderPtr Encoder;
    };

    THashMap<TShardKey, TEncoderData, TShardKeyHasher> EncoderDataByShardKey_;
    TEncoderFactory Factory_;
};

inline NMonitoring::IMetricEncoderPtr SpackEncoderFactory(const TShardKey&, IOutputStream* stream) {
    return EncoderSpackV1(
        stream,
        NMonitoring::ETimePrecision::SECONDS,
        NMonitoring::ECompression::ZSTD,
        NMonitoring::EMetricsMergingMode::MERGE_METRICS);
}

inline NMonitoring::IMetricEncoderPtr JsonEncoderFactory(const TShardKey&, IOutputStream* stream) {
    return NMonitoring::BufferedEncoderJson(stream, 0);
}

} // namespace NSolomon::NAgent
